
type vrm_data_loc = string 
type vrm_code_loc = string
type vrm_constant = int
type vrm_comment = string (* for instructional purposes! *)

datatype vrm_operation = 
       (* data operations *) 
         VRM_Nop of vrm_comment
       | VRM_Set of vrm_data_loc * vrm_constant * vrm_comment 
       | VRM_Mov of vrm_data_loc * vrm_data_loc * vrm_comment
       | VRM_Add of vrm_data_loc * vrm_data_loc * vrm_data_loc * vrm_comment
       | VRM_Sub of vrm_data_loc * vrm_data_loc * vrm_data_loc * vrm_comment
       | VRM_Mul of vrm_data_loc * vrm_data_loc * vrm_data_loc * vrm_comment
       (* control flow operations *) 
       | VRM_Hlt of vrm_comment 
       | VRM_Jmp of vrm_code_loc * vrm_comment 
       | VRM_Ifz of vrm_data_loc * vrm_code_loc * vrm_comment 
       | VRM_Ifp of vrm_data_loc * vrm_code_loc * vrm_comment 
       | VRM_Ifn of vrm_data_loc * vrm_code_loc * vrm_comment 
       (* input/output *) 
       | VRM_Pri of vrm_data_loc * vrm_comment
       | VRM_Prb of vrm_data_loc * vrm_comment

datatype vrm_code = 
         VRM_Code of vrm_operation 
       | VRM_Labelled of vrm_code_loc * vrm_operation 

type vrm_assembler = vrm_code list 

fun vrm_operation_to_strings (VRM_Hlt remark)                  = ("hlt",                                 remark)  
  | vrm_operation_to_strings (VRM_Nop remark)                  = ("nop",                                 remark) 
  | vrm_operation_to_strings (VRM_Jmp (cl, remark))            = ("jmp " ^ cl,                           remark) 
  | vrm_operation_to_strings (VRM_Set (dl, c, remark))         = ("set " ^ dl  ^ " " ^ (Int.toString c), remark) 
  | vrm_operation_to_strings (VRM_Mov (dl1, dl2, remark))      = ("mov " ^ dl1 ^ " " ^ dl2,              remark)
  | vrm_operation_to_strings (VRM_Add (dl1, dl2, dl3, remark)) = ("add " ^ dl1 ^ " " ^ dl2 ^ " " ^ dl3,  remark)
  | vrm_operation_to_strings (VRM_Sub (dl1, dl2, dl3, remark)) = ("sub " ^ dl1 ^ " " ^ dl2 ^ " " ^ dl3,  remark)
  | vrm_operation_to_strings (VRM_Mul (dl1, dl2, dl3, remark)) = ("mul " ^ dl1 ^ " " ^ dl2 ^ " " ^ dl3,  remark)
  | vrm_operation_to_strings (VRM_Ifz (dl, cl, remark))        = ("ifz " ^ dl  ^ " " ^ cl,               remark)
  | vrm_operation_to_strings (VRM_Ifp (dl, cl, remark))        = ("ifp " ^ dl  ^ " " ^ cl,               remark)
  | vrm_operation_to_strings (VRM_Ifn (dl, cl, remark))        = ("ifn " ^ dl  ^ " " ^ cl,               remark)
  | vrm_operation_to_strings (VRM_Pri (dl, remark))            = ("pri " ^ dl,                           remark) 
  | vrm_operation_to_strings (VRM_Prb (dl, remark))            = ("prb " ^ dl,                           remark) 

fun vrm_code_to_strings (VRM_Code oper) = ("", vrm_operation_to_strings oper) 
  | vrm_code_to_strings (VRM_Labelled (l,oper)) = (l ^ " : ", vrm_operation_to_strings oper)

fun vrm_program_to_strings [] = [] 
  | vrm_program_to_strings (c::rest) = (vrm_code_to_strings c) :: (vrm_program_to_strings rest)

fun vrm_inspect_program_strings ll lo [] = (ll, lo)
  | vrm_inspect_program_strings ll lo ((label, (oper, _))::rest) = 
       vrm_inspect_program_strings (Int.max(ll, String.size label)) (Int.max(lo, String.size oper)) rest 

fun vrm_program_with_nice_indentation ll lo [] = ""
  | vrm_program_with_nice_indentation ll lo ((label, (oper, remark)):: rest) = 
    (label ^ (Library.n_spaces (ll - (String.size label))) ^ oper ^ (Library.n_spaces (lo - (String.size oper))) ^ " %" ^ remark ^ "\n") 
    ^ (vrm_program_with_nice_indentation ll lo rest) 

fun vrm_program_to_string prog = 
    let val strs = vrm_program_to_strings prog 
        val (longest_label, longest_operation) = vrm_inspect_program_strings 0 0 strs 
    in 
        vrm_program_with_nice_indentation longest_label longest_operation strs 
    end 
