

open AST_expr;  (* for defs of loc, oper, unary_oper, type_expr *) 

datatype normal_expr = 
         Normal_SetInteger of loc * int
       | Normal_SetBoolean of loc * bool
       | Normal_UnaryOp of unary_oper * loc * loc 
       | Normal_Op of oper * loc * loc * loc
       | Normal_Assign of loc * loc 
       | Normal_Seq of normal_expr list 
       | Normal_If of into_expr * normal_expr * normal_expr
       | Normal_While of into_expr * normal_expr
       | Normal_Print of (type_expr option) * loc 

and into_expr = Into of normal_expr * loc 

(* pretty printing using Mosml's PP module *) 
                         
fun ppne pps (Normal_SetInteger (l,n)) = 
    (PP.add_string pps l; 
     PP.add_string pps " := ";
     PP.add_string pps (Int.toString n))
  | ppne pps (Normal_SetBoolean (l,b)) = 
    (PP.add_string pps l; 
     PP.add_string pps " := ";
     PP.add_string pps (Bool.toString b))
  | ppne pps (Normal_UnaryOp (uopr, l1, l2)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps l1;
       PP.add_string pps " : = " ; 
       PP.add_string pps ((unary_to_string uopr) ^ " "); 
       PP.add_string pps ("!" ^ l2 ^ ")")
      )
  | ppne pps (Normal_Op (opr,l1,l2,l3)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps l1;
       PP.add_string pps " : = " ; 
       PP.add_string pps ("!" ^ l2); 
       PP.add_string pps (" " ^ (op_to_string opr) ^ " "); 
       PP.add_string pps ("!" ^ l3);
       PP.add_string pps ")"
      )
  | ppne pps (Normal_If (e1, e2, e3)) = 
      (PP.begin_block pps PP.CONSISTENT 0; 
      PP.add_string pps "if "; 
      ppinto pps e1; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " then "; 
      ppne pps e2; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " else "; 
      ppne pps e3;
      PP.end_block pps)
  | ppne pps (Normal_Assign (l1, l2)) =  
       (
        PP.add_string pps l1; 
        PP.add_string pps " := !"; 
        PP.add_string pps l2
       )
  | ppne pps (Normal_Seq []) =  () 
  | ppne pps (Normal_Seq [e]) =  ppne pps e 
  | ppne pps (Normal_Seq (e::el)) =  
       (PP.begin_block pps PP.CONSISTENT 0; 
        PP.add_string pps "("; 		 
        ppne pps e; 
        PP.add_string pps "; "; 
        PP.add_break pps (0, 0); 	
	ppne_list pps el;
        PP.add_string pps ")"; 		 
        PP.end_block pps)
  | ppne pps (Normal_While (e1, e2)) = 
      (PP.begin_block pps PP.CONSISTENT 0; 
       PP.add_string pps "while "; 
       ppinto pps e1; 
       PP.add_string pps " do "; 
       PP.add_break pps (0, 0); 
       PP.begin_block pps PP.CONSISTENT 3;		
       PP.add_string pps "("; 
       ppne pps e2; 
       PP.add_string pps ")"; 
       PP.end_block pps;
       PP.end_block pps) 
  | ppne pps (Normal_Print (NONE, l)) = 
      (
       PP.add_string pps "print "; 
       PP.add_string pps ("!" ^ l)
      ) 
  | ppne pps (Normal_Print (SOME t, l)) = 
      (
       PP.add_string pps "print("; 
       PP.add_string pps (type_expr_to_string t); 
       PP.add_string pps (", !" ^ l ^ ")")
      ) 
and ppne_list pps [] = () 
  | ppne_list pps [e] =  ppne pps e 
  | ppne_list pps (e::el) =  
	(ppne pps e;
         PP.add_string pps "; "; 
         PP.add_break pps (0, 0);		
         ppne_list pps el)
and ppinto pps (Into(e, l)) = (ppne pps e; PP.add_string pps (" into " ^ l))

fun pp_normal_expr expr = 
    let val ppstrm = PP.mk_ppstream {
                consumer  = fn s => TextIO.output(TextIO.stdOut, s), 
                linewidth = 80,               
                flush     = fn () => TextIO.flushOut TextIO.stdOut}
    in 
        PP.begin_block ppstrm PP.CONSISTENT 0; 
        ppne ppstrm expr ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm 
    end 
                                    
    
(* this is used for debugging and for attaching "origin comments " 
   to bits of intermediate representations (such as assembly code).
   The string contains no line breaks. 
*) 
fun normal_to_string (Normal_SetInteger(l, n))      = l ^ " := " ^ (Int.toString n)
  | normal_to_string (Normal_SetBoolean(l, b))      = l ^ " := " ^ (Bool.toString b)
  | normal_to_string (Normal_UnaryOp (uopr,l1, l2)) = "(" ^ (unary_to_string uopr) ^ " " ^  l1 ^ " " ^ l2 ^ ")"
  | normal_to_string (Normal_Op (opr, l1, l2, l3))  = "(" ^ (op_to_string opr) ^ " " ^ l1 ^ " " ^ l2 ^ " " ^ l3 ^ ")" 
  | normal_to_string (Normal_If (e1, e2, e3))       = "(if " ^ (into_to_string e1) ^ " then " ^ (normal_to_string e2) ^ " else " ^  (normal_to_string e3) ^ ")"
  | normal_to_string (Normal_Assign (l1, l2))       = "(" ^ l1 ^ " : = !" ^  l2 ^ ")"
  | normal_to_string (Normal_Seq [])                = "" 
  | normal_to_string (Normal_Seq [e])               = normal_to_string e 
  | normal_to_string (Normal_Seq (e::rest))         = "(" ^ (normal_to_string e) ^ "; " ^ (normal_to_string (Normal_Seq rest)) ^ ")"
  | normal_to_string (Normal_While (e1, e2))        = "(while " ^ (into_to_string e1) ^ " do " ^ (normal_to_string e2) ^ ")"
  | normal_to_string (Normal_Print (NONE, l))       = "print(!" ^ l ^ ") "
  | normal_to_string (Normal_Print (SOME t, l))     = "print(" ^ (type_expr_to_string t) ^ ", !" ^ l ^ ")"
and into_to_string (Into (e, l))                    = (normal_to_string e) ^ " into "  ^ l
