open AST_vrm_assembler; 
open AST_vrm_bytecode; 

fun mk_vrm_code_loc_map prog = 
    let fun aux string_to_code_loc m [] = 
               (m, string_to_code_loc)
          | aux string_to_code_loc m ((VRM_Code _)::rest) = 
               aux string_to_code_loc (m+1) rest
          | aux string_to_code_loc m ((VRM_Labelled(l, _))::rest) = 
               aux (Library.update(string_to_code_loc, l, Word8.fromInt m)) (m+1) rest
    in aux Library.empty_env 0 prog end 


val next = Library.new_counter(); 

fun new_word () = Word8.fromInt (next ()); 

val find = Library.lookup; 

fun visit_vrm_data_loc dlm dl = 
     (let val _ = find(dlm, dl) in dlm end) 
       handle (Library.Missing _) => Library.update(dlm, dl, new_word()) 

fun update_vrm_data_loc_map dlm (VRM_Set (dl, _, _))    = visit_vrm_data_loc dlm dl 
  | update_vrm_data_loc_map dlm (VRM_Mov (dl, _, _))    = visit_vrm_data_loc dlm dl 
  | update_vrm_data_loc_map dlm (VRM_Add (dl, _, _, _)) = visit_vrm_data_loc dlm dl 
  | update_vrm_data_loc_map dlm (VRM_Sub (dl, _, _, _)) = visit_vrm_data_loc dlm dl 
  | update_vrm_data_loc_map dlm (VRM_Mul (dl, _, _, _)) = visit_vrm_data_loc dlm dl 
  | update_vrm_data_loc_map dlm  _                      = dlm

fun mk_vrm_data_loc_map prog = 
    let fun aux dlm [] = dlm 
          | aux dlm ((VRM_Code c)::rest) = aux (update_vrm_data_loc_map dlm c) rest
          | aux dlm ((VRM_Labelled(_, c))::rest) = aux (update_vrm_data_loc_map dlm c) rest
    in aux Library.empty_env prog end 


fun vrm_assemble_op clm dlm (VRM_Hlt _)                  = VRM_B_Hlt 
  | vrm_assemble_op clm dlm (VRM_Nop _)                  = VRM_B_Nop
  | vrm_assemble_op clm dlm (VRM_Jmp (cl, _))            = VRM_B_Jmp(find(clm, cl))
  | vrm_assemble_op clm dlm (VRM_Set (dl, c, _))         = VRM_B_Set(find(dlm, dl), Word8.fromInt c)
  | vrm_assemble_op clm dlm (VRM_Mov (dl1, dl2, _))      = VRM_B_Mov(find(dlm, dl1), find(dlm, dl2))
  | vrm_assemble_op clm dlm (VRM_Add (dl1, dl2, dl3, _)) = VRM_B_Add(find(dlm, dl1), find(dlm, dl2), find(dlm, dl3))
  | vrm_assemble_op clm dlm (VRM_Sub (dl1, dl2, dl3, _)) = VRM_B_Sub(find(dlm, dl1), find(dlm, dl2), find(dlm, dl3))
  | vrm_assemble_op clm dlm (VRM_Mul (dl1, dl2, dl3, _)) = VRM_B_Mul(find(dlm, dl1), find(dlm, dl2), find(dlm, dl3))
  | vrm_assemble_op clm dlm (VRM_Ifz (dl, cl, _))        = VRM_B_Ifz(find(dlm, dl), find(clm, cl))
  | vrm_assemble_op clm dlm (VRM_Ifp (dl, cl, _))        = VRM_B_Ifp(find(dlm, dl), find(clm, cl))
  | vrm_assemble_op clm dlm (VRM_Ifn (dl, cl, _))        = VRM_B_Ifn(find(dlm, dl), find(clm, cl))
  | vrm_assemble_op clm dlm (VRM_Pri (dl, _))            = VRM_B_Pri(find(dlm, dl))
  | vrm_assemble_op clm dlm (VRM_Prb (dl, _))            = VRM_B_Prb(find(dlm, dl))

fun vrm_assemble_list clm dlm m [] = [] 
  | vrm_assemble_list clm dlm m ((VRM_Code opr)::rest) = (Word8.fromInt m, vrm_assemble_op clm dlm opr) :: (vrm_assemble_list clm dlm (m+1) rest)
  | vrm_assemble_list clm dlm m ((VRM_Labelled(_, opr))::rest) = (Word8.fromInt m, vrm_assemble_op clm dlm opr) :: (vrm_assemble_list clm dlm (m+1) rest)

fun vrm_assemble prog = 
    let val (inst_count, clm) = mk_vrm_code_loc_map prog 
        val dlm = mk_vrm_data_loc_map prog 
        val code = vrm_assemble_list clm dlm 0 prog 
    in (inst_count, code) end 
