
open AST_normal_expr; 
open AST_vrm_assembler; 

val next_loc = Library.new_counter(); 
fun new_location () = "_X" ^ (Int.toString (next_loc ())) 
val next_label = Library.new_counter(); 
fun new_label () = "_l" ^ (Int.toString (next_label ())) 

fun vrm_label_sequence [] = (new_label(), []) 
  | vrm_label_sequence ((VRM_Code oper)        :: rest        ) = 
    let val l = new_label() 
    in 
       (l, (VRM_Labelled(l, oper)) :: rest) 
    end 
  | vrm_label_sequence ((VRM_Labelled(l, oper)) :: rest) = 
       (l, (VRM_Labelled(l, oper)) :: rest) 

fun vrm_insert_remark_in_op new (VRM_Hlt remark)                  = VRM_Hlt (new ^ " " ^ remark)
  | vrm_insert_remark_in_op new (VRM_Nop remark)                  = VRM_Nop (new ^ " " ^ remark)                  
  | vrm_insert_remark_in_op new (VRM_Jmp (cl, remark))            = VRM_Jmp (cl, (new ^ " " ^ remark))            
  | vrm_insert_remark_in_op new (VRM_Set (dl, c, remark))         = VRM_Set (dl, c, (new ^ " " ^ remark))         
  | vrm_insert_remark_in_op new (VRM_Mov (dl1, dl2, remark))      = VRM_Mov (dl1, dl2, (new ^ " " ^ remark))      
  | vrm_insert_remark_in_op new (VRM_Add (dl1, dl2, dl3, remark)) = VRM_Add (dl1, dl2, dl3, (new ^ " " ^ remark)) 
  | vrm_insert_remark_in_op new (VRM_Sub (dl1, dl2, dl3, remark)) = VRM_Sub (dl1, dl2, dl3, (new ^ " " ^ remark)) 
  | vrm_insert_remark_in_op new (VRM_Mul (dl1, dl2, dl3, remark)) = VRM_Mul (dl1, dl2, dl3, (new ^ " " ^ remark)) 
  | vrm_insert_remark_in_op new (VRM_Ifz (dl, cl, remark))        = VRM_Ifz (dl, cl, (new ^ " " ^ remark))        
  | vrm_insert_remark_in_op new (VRM_Ifp (dl, cl, remark))        = VRM_Ifp (dl, cl, (new ^ " " ^ remark))        
  | vrm_insert_remark_in_op new (VRM_Ifn (dl, cl, remark))        = VRM_Ifn (dl, cl, (new ^ " " ^ remark))        
  | vrm_insert_remark_in_op new (VRM_Pri (dl, remark))            = VRM_Pri (dl, (new ^ " " ^ remark))            
  | vrm_insert_remark_in_op new (VRM_Prb (dl, remark))            = VRM_Prb (dl, (new ^ " " ^ remark))            

fun vrm_insert_remark remark [] = [] 
  | vrm_insert_remark remark ((VRM_Code oper) :: rest) = (VRM_Code (vrm_insert_remark_in_op remark oper)) :: rest
  | vrm_insert_remark remark ((VRM_Labelled(l, oper)) :: rest) = ((VRM_Labelled(l, vrm_insert_remark_in_op remark oper)) :: rest) 

val zero_loc    = "_Zero"
val true_loc    = "_TRUE"
val false_loc   = "_FALSE"

val constants_code = [
     VRM_Code(VRM_Set(zero_loc, 0, " zero")), 
     VRM_Code(VRM_Set(true_loc, 1, " true value")), 
     VRM_Code(VRM_Set(false_loc, 0, " false value"))
    ] 

fun normal_expr_to_vrm_code_list (Normal_SetInteger (l, n))     = 
       [VRM_Code(VRM_Set(l, n, ""))]
  | normal_expr_to_vrm_code_list (Normal_SetBoolean (l, true))  = 
       [VRM_Code(VRM_Set(l, 1, ""))]
  | normal_expr_to_vrm_code_list (Normal_SetBoolean (l, false)) = 
       [VRM_Code(VRM_Set(l, 0, ""))]
  | normal_expr_to_vrm_code_list (Normal_UnaryOp (Neg, l1, l2)) = 
       [VRM_Code(VRM_Sub(l1, zero_loc, l2, " neg"))]
  | normal_expr_to_vrm_code_list (Normal_UnaryOp (Not, l1, l2)) = 
      let val l3 = new_label() 
          and l4 = new_label() 
      in 
         [
          VRM_Code(VRM_Ifz(l2, l3, "start not ... ")), 
          VRM_Code(VRM_Mov(l1, false_loc, "get false value")), 
          VRM_Code(VRM_Jmp (l4, "")), 
          VRM_Labelled(l3, VRM_Mov(l1, true_loc, "get true value")), 
          VRM_Labelled(l4, VRM_Nop " ... end not")
          ]
      end 
  | normal_expr_to_vrm_code_list (Normal_Op (Plus, l1, l2, l3)) = [VRM_Code(VRM_Add(l1, l2, l3, ""))]
  | normal_expr_to_vrm_code_list (Normal_Op (Mult, l1, l2, l3)) = [VRM_Code(VRM_Mul(l1, l2, l3, ""))]
  | normal_expr_to_vrm_code_list (Normal_Op (Subt, l1, l2, l3)) = [VRM_Code(VRM_Sub(l1, l2, l3, ""))]
  | normal_expr_to_vrm_code_list (Normal_Op (GTEQ, l1, l2, l3)) = 
      let val l4 = new_location() 
          and l_false = new_label() 
          and l_end = new_label() 
      in 
         [VRM_Code(VRM_Sub(l4, l2, l3, "start >= ...")), 
          VRM_Code(VRM_Ifn(l4, l_false, "")), 
          VRM_Code(VRM_Mov(l1, true_loc, "get true")), 
          VRM_Code(VRM_Jmp (l_end, "")), 
          VRM_Labelled(l_false, VRM_Mov(l1, false_loc, "get false")), 
          VRM_Labelled(l_end, VRM_Nop "... end >=")
         ]
      end 
  | normal_expr_to_vrm_code_list (Normal_Assign(l1, l2))        = [VRM_Code(VRM_Mov(l1, l2, ""))] 
  | normal_expr_to_vrm_code_list (Normal_Seq el)                = List.concat (List.map normal_expr_to_vrm_code_list el)
  | normal_expr_to_vrm_code_list (Normal_If(Into(e1, t), e2, e3))         =  
    let val cl_cond = normal_expr_to_vrm_code_list e1 
        and cl_then = normal_expr_to_vrm_code_list e2 
        and cl_else = normal_expr_to_vrm_code_list e3 
    in 
        let val (l_else, cl_else_new) = vrm_label_sequence cl_else
            and l_end = new_label() 
        in 
           (vrm_insert_remark "start if (condition)  ... " cl_cond)
           @ [VRM_Code(VRM_Ifz(t, l_else, "test of if ..."))] 
           @ (vrm_insert_remark "start then ... " cl_then)
           @ [VRM_Code(VRM_Jmp (l_end, "... end then ..."))] 
           @ (vrm_insert_remark "start else ... " cl_else_new) 
           @ [VRM_Labelled(l_end, VRM_Nop "... end if")]
        end 
    end  
  | normal_expr_to_vrm_code_list (Normal_While(Into(e1, d), e2))     = 
    let val cl_cond = normal_expr_to_vrm_code_list e1
        and cl_body = normal_expr_to_vrm_code_list e2
        and l_end   = new_label()   
    in 
        let val (l_cond, cl_cond_new) = vrm_label_sequence cl_cond
        in 
           (vrm_insert_remark "start while ... " cl_cond_new) 
           @ [VRM_Code(VRM_Ifz(d, l_end, "test of while ..."))] 
           @ cl_body 
           @ [VRM_Code(VRM_Jmp (l_cond, "... go back to while condition ")), 
              VRM_Labelled(l_end, VRM_Nop "... end while")]
        end 
    end  
  | normal_expr_to_vrm_code_list (Normal_Print(SOME TEint, l))  = [VRM_Code (VRM_Pri (l, ""))]
  | normal_expr_to_vrm_code_list (Normal_Print(SOME TEbool, l)) = [VRM_Code (VRM_Prb (l, ""))]
  | normal_expr_to_vrm_code_list (Normal_Print(NONE, l))        = 
      Library.internal_error "normal_expr_to_vrm_code_list : Normal_Print has no type"
  | normal_expr_to_vrm_code_list (Normal_Print(SOME t , l))     = 
      Library.internal_error ("normal_expr_to_vrm_code_list : Normal_Print has type " ^ (type_expr_to_string t))

fun vrm_code_gen e = 
       constants_code 
       @ (normal_expr_to_vrm_code_list e) 
       @ [VRM_Code (VRM_Hlt " that's all folks!")] 


