(* 
 * The compiler itself
 *)

(* there are 4 passes
 * 1 - decend the code, building up a list of functions and arguments
 * 2 - accend back up, optomising using peepholeing
 * 3 - decend the assembly language, producing the PWC (portable word code)
 * 4 - accend back, putting values into the text
 *
 *  (Pass 3 is supplied in the form of the assembler)
 *
 *)

    
(*
 *  Pass One, Decode the Text, evaluating arguments
 *
 *)

local
    val Access = BCODES_Access;
    val {textLength,fetchWord,fetchByte,...} = Text;
    val {AddEntry,InTable,...} = MakeDict ((op <):int*int->bool);
    val dict = ref NewTable;
    fun NotifyFamLabel (n) =  if InTable(n,!dict)  (* don't add twice *)
				 then n
			     else (dict := AddEntry(n,0,!dict);n);
    fun ResetFamLabels () = (dict := NewTable);
    val flowerror = (Comp_ncg "Flow of control Error");

    val internalerror = (Comp_ncg "Internal Semantic Error");
    val imposible = (Comp_ncg "Imposible Semantic Error");
in
    fun PassOne (txt) =
	let
	    val  lentext = textLength txt;
		
	    fun ExtractB(m,n)= 
		  if n = 0 
		      then [] 
		  else fetchByte(txt,m)::ExtractB(m+1,n-1);

	    fun ExtractW(m,n)= 
		if n = 0 
		    then [] 
		else fetchWord(txt,m)::ExtractW(m+2,n-1)

	    fun signB n' = if n' > 127 then n'-256 else n';
	    fun signW n' = if n' > 32767 then n'-65536 else n';
	    fun fixitem n x = NotifyFamLabel (n+(signW x));
	    fun getlist n = map (fixitem n) (ExtractW(n+2,fetchWord(txt,n)));

	    fun findfullarg (x,n) =
		jumpcase x of 
	PRIM    => (0,NON)
      | BYTE 	=> (1,INT (fetchByte(txt,n)))
      | WORD 	=> (2,INT (fetchWord(txt,n)))
      | CONST x => (0,INT x)
      | LIST 	=> let val x = getlist(n) in (length x * 2 +2,INTLIST x) end
      | BYTE2P	=> (2,INT2 (fetchByte(txt,n),fetchByte(txt,n+1)+1))
      | WORD2 	=> (4,INT2 (fetchWord(txt,n),fetchWord(txt,n+2)))
      | WORD2P	=> (4,INT2 (fetchWord(txt,n),signW (fetchWord(txt,n+2))+ 1))
      | BY_CON x => (1,INT2 (fetchByte(txt,n),x))
      | CON_BY x => (1,INT2 (x,fetchByte(txt,n)))
      | WD_CON x => (2,INT2 (fetchWord(txt,n),x))
      | CON_WD x => (2,INT2 (x,fetchWord(txt,n)))
      | CONST2 x => (0,INT2 x)
      | JBYTE   => (1,JINT (NotifyFamLabel((signB (fetchByte(txt,n)))+n+1)))
      | JWORD   => (2,JINT (NotifyFamLabel((signW (fetchWord(txt,n)))+n+2)));
	    
				  
	    fun ExtractByte (n,sofar) =
		if n = lentext 
		    then (n,sofar,!dict,txt)
		else 
		    if n > lentext
			then raise (Comp_ncg "Byte DeCodeing")
		    else 
			let
			    val hd' = fetchByte(txt,n);
			    val _ = if !(#PrintByte Ncopt)
					then (outstring ("%"^($ hd')^":\n"))
				    else ()
			    val (ln,arg,code) = case Access hd' of
				INSERT => (0,NON,hd')
			      | BCDEF (a,i) => 
				    let 
					val (ln,arg) = findfullarg(a,n+1)
				    in 
					(ln,arg,i)
				    end
			in
			    ExtractByte(ln+1+n,(ln+1,arg,code)::sofar)
			end

	      val _ = ResetFamLabels ();
	      val _ = #ResetNCLabels NativeCoder();
	      val _ = if !(#PrintDebug Ncopt) 
			  then outstring("[ 1 - Decoding Bytes ]\n")
		      else ()

	in		  
	    ExtractByte(0,[]) (* All passes are tail recurive *)
	end
end;


(*
 * Now Pass Two, *** This is the vital pass ***
 *
 *) 


(* 
 *  Extract things from the machine 
 *
 *)

local
    val {InTable,...} = MakeDict ((op <):int*int->bool)
    val {fetchByte,...} = Text;
    val {Linker,DefLink,Byte_Insertion = bci,HeapType,...} = NativeCoder;
    val Expensive = [(DefLink,10000,[])];
    val Cheap = [(DefLink,0,[])];
    val In = 10000; (* infinity *)
    val {NulifyDy,FindDy,...} = DynamicDB;
    exception Fail;
in
    local
	nonfix HeapStatus 80 1;
	fun HeapStatus (a:int):int = HeapStatus(a);
    in
	val _ = HeapStatus(HeapType)  (* Change the status of the heap  *)
    end;
    
    fun PassTwo(totallen,t,env,txt) = 
	let
	(* argument , type, placement, function, previous link, sofar*)

	val peephole = !(#PHStack Ncopt);

	fun ZoomUp ([],iflab,sofar,lab) = 
	    let
		val togo = fst (hd sofar); (*ignore the previous argument*)
		fun  addtop (i,c,cd) =
		    let
			val (lnk,cst) = if (not peephole) 
			    orelse InTable(lab,env)
					    then  ([],0)  (* already added *)
			else Linker(DefLink,i,lab,true)
		    in
			(cst+c,lnk::cd)  (* total cost + all code *)
		    end;
		val tops = map addtop togo;
		fun (*native*) best (x,y) = if (fst(x):int) < fst(y) then x else y;
		val (k,code) = fold best (tl tops) (hd tops);

		val _ = if !(#PrintDebug Ncopt) 
			    then outstring("cost = "^($ k)^" ]\n")
			else ();
	    in
		(code,txt)
	    end

	  | ZoomUp ((len,main_arg,bytecode)::cd,iflab,sofar,lab') =
	    let
		val lab = lab'-len;
	
		fun ExtractB(m,n)= 
		    if n = 0 
			then [] 
		    else fetchByte(txt,m)::ExtractB(m+1,n-1);

		val flowerror = (Comp_ncg "Flow of Control Error");
	
		fun (*native*) findnewanswers (mapping,index) =
		    let
			fun (*native*) lookback' (n,l) = 
			    if n = 0 
				then (fst (hd l),[])
			    else 
				let
				    val (a,b) = lookback'(n-1,tl l)
				in
				    (a,(snd(hd l))::b)
				end
			val (old_code,old_args) = lookback'(index,sofar)
			val native = case mapping of 
			    FN x => (x main_arg)		(*normal     *)
			     | FNL x => (x (main_arg::old_args))(*overlapping*)
			     | BCI => (case main_arg of
				       INTLIST _ => raise flowerror
				     | JINT _    => raise flowerror
				     | _ =>(bci ((ExtractB(lab,len)))))
			     				(*insert text*)
			     | ANSWER x => x	(*short cut  *)
				   handle (Comp_ncg x) => raise (Comp_ncg x)
		| _  =>raise (Comp_ncg ("Match Failure : "^($ bytecode)));
		    in	
			(native,old_code)  (* code, back code *)
		    end;


    fun (*native*) join (a,b,c,d:int) (a',d',b') =  (* new code X old code *)
	let
	    val (lk,lc) = Linker(c,a',lab,false)   (* cant cut BB *)
	in
	    (a,d+d'+lc,b::lk::b')  (* add costs and linkage *)
	end;

    fun (*native*) findbest ((code,previous),bestsofar) = 
	let 
	    val new_natives = map (join code) previous;
	    fun (*native*) best(x,y) = 
		let
		    fun g13 (x,_,_) = x;
		    fun g23 (_,x,_) = x;
		    fun findb [] = [x]   (* New Type *)
		      | findb (l as (h::t))  = 
			       if g13 x = g13 h (* same type *)
				   then 
				       if ((g23 x):int) < g23 h  (* cheaper *)
					   then x::t
				       else raise Fail 	     (* not best *)
			       else (h)::findb t          (* try again *)

		in
		    (findb y) handle Fail => y
		end
	in 
	    fold best new_natives bestsofar
	end;
	val possibles = map (findnewanswers) (FindDy bytecode);
	val newtop = fold findbest possibles [];
	    in
		if (not peephole) orelse InTable(lab,env)
		    then 
			let
			    val _ = NulifyDy ();   (* Mark Basic Block *)
			    fun join (a,b,c) =
				let
				    val (lk,lc) = Linker(DefLink,a,lab,true) 
				in
				    (DefLink,b+lc,lk::c)
				end;
			    val newtop' = map join newtop;
			in	
			  ZoomUp(cd,true,(newtop',main_arg)::sofar,lab)
			end
		else ZoomUp(cd,false,(newtop,main_arg)::sofar,lab)
	    end

	val _ = if !(#PrintDebug Ncopt) 
		    then outstring("[ 2 - Compiling Code: ")
		else ()
    in
	ZoomUp(t,peephole,[([(DefLink,0,[[]])],NON)],totallen)
    end;
end;

(*
 *
 *  Pass Three is suplied by the Machine specific File
 *
 *)

val PassThree = #Assembler NativeCoder;

(*
 *  Pass Four: Place the code into the text, resolving labels 
 *
 *)

local
    nonfix EQPTR 127 2;     (* keep this pretty quiet *)
    fun EQPTR(x: 'a, y: 'a): bool = EQPTR(x,y);
    val {textLength,storeWord,makenativetext,getliterals,VetText,...} = Text;
in	

    fun PassFour (n,l,t,f) =  (* (word list,length,text,fn) -> (text) *)
	let 
	    val nct = makenativetext(t,l*2,getliterals(t)); (* 1 = NC *)
	    val _ = if EQPTR(nct,t) (* if there is no new text *)
		then raise (Comp_ncg "Arch Type Failure")
		    else ();
	    
	    fun (*native*) filltext (n) = if null n then nct else 
		let 
		    fun (*native*) insert(x,z) = 
			if null x 
			    then () 
			else (storeWord(hd x,nct,z);insert(tl x,z+2))
			    

		    val _ = case hd n of 
			Word(x,z)   => insert(x,z*2)
		      | Jump(x,z) =>  storeWord(f(x,z),nct,z*2)
		in 
		    filltext(tl n)
		end;
	
	    val _ = if !(#PrintDebug Ncopt) 
			then outstring("[ 4 - Creating Text]\n")
		    else ()

	    val nt = filltext (n)

	in	
	    #ByteCodeSize NCinfo := !(#ByteCodeSize NCinfo)+textLength(t);
	    #NativeCodeSize NCinfo := !(#NativeCodeSize NCinfo)+textLength(nt);
	    nt
	end;
end;

(*
 *  The Native Code Generator's Top Level Function: text -> text
 *
 *)
local
    val {VetText,nametext,...} = Text;
in
    fun NCG bct = (
		   let
		       val (w,l,f) = (PassThree o PassTwo o PassOne) bct;
		   in
		       PassFour(w,l,bct,f)
 		   end )
		   handle Comp_ncg s => 
		       if !(#PrintDebug Ncopt) 
			   then 
			       (outstring(
   "-- Native Code Generation Failure: "^s^" ("^(nametext bct)^")\n"
					  );bct)
		       else if !(#PrintMsgs Ncopt) 
	then (outstring("[Using Byte-Code for "^(nametext bct)^" ]\n");bct)
			    else bct
end;
