/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package uk.ac.cam.acr31.android.oggstreamer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import uk.ac.cam.acr31.oggdecoder.EndOfStreamException;
import uk.ac.cam.acr31.oggdecoder.ImpossibleExceptionError;
import uk.ac.cam.acr31.oggdecoder.InvalidStreamException;
import uk.ac.cam.acr31.oggdecoder.OggPacketReceiver;
import uk.ac.cam.acr31.oggdecoder.OggStream;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioTrack;
import de.jarnbjo.vorbis.IdentificationHeader;
import de.jarnbjo.vorbis.VorbisStream;

public class PushVorbisPacketReceiver implements OggPacketReceiver {

	private ByteArrayOutputStream currentPacket;
	private VorbisStream vorbisStream = new VorbisStream(
			VorbisStream.LITTLE_ENDIAN);
	private AudioTrack mAudioTrack;
	private boolean audioStreamStarted = false;

	@Override
	public boolean needsData() {
		return true;
	}

	@Override
	public void packetData(long granulePosition, OggStream stream, int length,
			boolean isStart, boolean isEnd) throws EndOfStreamException,
			InvalidStreamException {
		if (isStart) {
			currentPacket = new ByteArrayOutputStream();
		}
		byte[] data = new byte[length];
		stream.read(data, 0, length);
		try {
			currentPacket.write(data);
		} catch (IOException e) {
			throw new ImpossibleExceptionError(e);
		}
		if (isEnd) {
			try {
				byte[] result = vorbisStream.processPacket(currentPacket
						.toByteArray());
				if (result != null) {
					if (!audioStreamStarted) {
						IdentificationHeader ih = vorbisStream
								.getIdentificationHeader();
						int channelConfig = ih.getChannels() == 1 ? AudioFormat.CHANNEL_CONFIGURATION_MONO
								: AudioFormat.CHANNEL_CONFIGURATION_STEREO;

						int minBufSize = AudioTrack.getMinBufferSize(ih
								.getSampleRate(), channelConfig,
								AudioFormat.ENCODING_PCM_16BIT);
						int timedBufSize = ih.getSampleRate()
								* ih.getChannels() * 2;
						int bufSize = Math.max(timedBufSize, minBufSize);

						mAudioTrack = new AudioTrack(AudioManager.STREAM_MUSIC,
								ih.getSampleRate(), channelConfig,
								AudioFormat.ENCODING_PCM_16BIT, bufSize,
								AudioTrack.MODE_STREAM);
						mAudioTrack.play();
						audioStreamStarted = true;
					}
					writeFully(result, 0, result.length);
				}
			} catch (IOException e) {
				throw new InvalidStreamException(e);
			} catch (ArrayIndexOutOfBoundsException e) {
				// gets thrown in audio decode sometimes. What to do?
			}
		}
	}

	private void writeFully(byte[] data, int off, int length) {
		while (length > 0) {
			int read = mAudioTrack.write(data, off, length);
			length -= read;
			off += read;
		}
	}

	@Override
	public void close() {
		mAudioTrack.stop();
		mAudioTrack.release();
	}
}
