#!/usr/bin/perl
#
# Set up the router for bandwidth tests
#
# Note: puts all entries in *first* or *last* position in tables
#


use strict;
use NF2::RegressLib;
use NF2::PacketLib;
use Getopt::Long;

# Update the following lines appropriately
# == BEGIN ==
use reg_defines_reference_router;
my $ROUTE_TABLE_LEN = 32;
my $ARP_TABLE_LEN = 32;
# == END ==


my $NUM_IFACE = 4;
my $pos = 0;

my $iface = 0;
my $first;
my $last;
my $help;

unless ( GetOptions ( "iface=i" => \$iface,
		      "first" => \$first,
		      "last" => \$last,
		      "help" => \$help,
		     )
	 and ($help eq '')

       ) { usage(); exit 1 }

# Verify that the iface is valid
if ($iface < 0 || $iface >= $NUM_IFACE) {
	print STDERR "$0: Error: the interface must be between 0 and " . ($NUM_IFACE - 1) . "\n\n";
	usage();
	exit 1;
}

# Verify that only one of first and last is specified
if (!defined($first) && !defined($last)) {
	print STDERR "$0: Error: must specify either first or last\n\n";
	usage();
}
elsif (defined($first) && defined($last)) {
	print STDERR "$0: Error: can only specify one of first and last\n\n";
	usage();
}

# Enable testing on nf2c0 to allow register reads/writes
my @interfaces = ("nf2c0");
nftest_init(\@ARGV,\@interfaces,);

# Set the mac addresses
my $offset = ROUTER_OP_LUT_MAC_1_HI_REG() - ROUTER_OP_LUT_MAC_0_HI_REG();
for (my $i = 0; $i < $NUM_IFACE; $i++) {
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_MAC_0_HI_REG() + $offset * $i, 0x0011);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_MAC_0_LO_REG() + $offset * $i, 0x22334400 + $i * 16 + $i);
}

# Clear tha routing table
for (my $i = 0; $i < $ROUTE_TABLE_LEN; $i++) {
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_IP_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_MASK_REG(), 0xffffffff);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_NEXT_HOP_IP_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_OUTPUT_PORT_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_WR_ADDR_REG(), $i);
}

# Clear tha ARP table
for (my $i = 0; $i < $ARP_TABLE_LEN; $i++) {
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_MAC_HI_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_MAC_LO_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_NEXT_HOP_IP_REG(), 0x0);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_WR_ADDR_REG(), $i);
}

# Set up the routing table
my $ip_addr = (192 << 24) + (168 << 16) + (100 << 8) + 0;
my $subnet_mask = (255 << 24) + (255 << 16) + (255 << 8) + 0;
my $next_hop = (192 << 24) + (168 << 16) + (0 << 8) + 1;
my $pos = defined($first) ? 0 : $ROUTE_TABLE_LEN - $NUM_IFACE;
for (my $i = 0; $i < $NUM_IFACE; $i++) {
	my $offset = defined($first) ? ($iface + $i) % 4 : ($iface + 1 + $i) % 4;
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_IP_REG(), $ip_addr + ($offset << 8));
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_MASK_REG(), $subnet_mask);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_NEXT_HOP_IP_REG(), $next_hop);
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_ENTRY_OUTPUT_PORT_REG(), 0x01 << ($offset * 2));
	nftest_regwrite('nf2c0', ROUTER_OP_LUT_ROUTE_TABLE_WR_ADDR_REG(), $pos + $i);
}

# Set up the arp table
$pos = defined($first) ? 0 : $ARP_TABLE_LEN - 1;
nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_MAC_HI_REG(), 0x00aa);
nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_MAC_LO_REG(), 0xbbccdd00);
nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_ENTRY_NEXT_HOP_IP_REG(), $next_hop);
nftest_regwrite('nf2c0', ROUTER_OP_LUT_ARP_TABLE_WR_ADDR_REG(), $pos);

exit 0;

#####################################################################
#  Functions
#####################################################################

sub usage {
  (my $cmd = $0) =~ s/.*\///;
  print <<"HERE1";
NAME
   $cmd - setup the router for performance testing

SYNOPSIS
   $cmd [--iface <iface_num>]
        [--first] [--last] 

   $cmd --help  - show detailed help

HERE1

  return unless ($help);
  print <<"HERE";

HERE
}
