//////////////////////////////////////////////////////////////////////////////
// vim:set shiftwidth=3 softtabstop=3 expandtab:
// $Id: u_board.v 1 2009-10-08 18:23:02Z root $
//
// Module: c_board.v
// Project: NF2 Control board
// Description: Instantiates active components on Control board.
//              Used for verification of control board.
//
///////////////////////////////////////////////////////////////////////////////

`timescale 1 ns/100 ps
`define sb200
  
module u_board  (
		   // CPCI I/Os.
		   inout [31:0] AD,
		   inout  [3:0] CBE,
		   inout        PAR,
		   inout        FRAME_N,
		   inout        TRDY_N,
		   inout        IRDY_N,
		   inout        STOP_N,
		   inout        DEVSEL_N,
		   output       INTR_A,
		   input        RST_N,
                                                                                
		   // PCI ports from pci_top as generated by CoreGen
		   input        IDSEL,
		   inout        PERR_N,
		   inout        SERR_N,
		   output       REQ_N,
		   input        GNT_N,
		   input        PCLK,     // PCI clock 33MHz
		   
		   // --- UNET I/Os
		   // Ethernets
		   output [3:0] rgmii_0_tx_d,
		   output       rgmii_0_tx_ctl, // 1 = tx_d is valid
		   output       rgmii_0_tx_clk, // used in 1000 mode only
		   input        rgmii_0_rx_clk, // used in 10/100/1000 modes
		   input  [3:0] rgmii_0_rx_d,
		   input        rgmii_0_rx_ctl,
		   
		   output [3:0] rgmii_1_tx_d,
		   output       rgmii_1_tx_ctl, // 1 = tx_d is valid
		   output       rgmii_1_tx_clk, // used in 1000 mode only
		   input        rgmii_1_rx_clk, // used in 10/100/1000 modes
		   input  [3:0] rgmii_1_rx_d,
		   input        rgmii_1_rx_ctl,
		   
		   output [3:0] rgmii_2_tx_d,
		   output       rgmii_2_tx_ctl, // 1 = tx_d is valid
		   output       rgmii_2_tx_clk, // used in 1000 mode only
		   input        rgmii_2_rx_clk, // used in 10/100/1000 modes
		   input  [3:0] rgmii_2_rx_d,
		   input        rgmii_2_rx_ctl,
		   
		   output [3:0] rgmii_3_tx_d,
		   output       rgmii_3_tx_ctl, // 1 = tx_d is valid
		   output       rgmii_3_tx_clk, // used in 1000 mode only
		   input        rgmii_3_rx_clk, // used in 10/100/1000 modes
                   input  [3:0] rgmii_3_rx_d,
                   input        rgmii_3_rx_ctl,
                   
                   // --- Serial Pins
                   output  serial_TXP_0,
                   output  serial_TXN_0,
                   input   serial_RXP_0,
                   input   serial_RXN_0,
                 
                   output  serial_TXP_1,
                   output  serial_TXN_1,
                   input   serial_RXP_1,
                   input   serial_RXN_1,
		 
                   output phy_mdc,    // Serial control interface to Quad PHY (clock)
                   inout  phy_mdio,   // Serial control interface to Quad PHY (tri data)
                   
                   input gtx_clk,     // 125MHz
                   input mii_tx_clk,   // 25MHz   (used in 10/100 mode)

                   input host32_is_active // to tell us when the cpci is ready
    );

   wire clk; // Core system clock
   reg [1:0] clk_gen;
   reg clk_62_5; // 62.5MHz clock
   reg clk_ddr_200; // 200MHz clock

   // CPCI <-> CNET signals with simulated delays
   wire [26:0] #1 cpci_addr;
   tri  [31:0] #1 cpci_data;
   wire        #1 cpci_rd_wr_L;
   wire        #1 cpci_req;
   wire        #1 cpci_wr_rdy;
   wire        #1 cpci_rd_rdy;

   // CPCI <-> CNET signals (DMA) with simulated delays
   wire [1:0] #1 dma_op_code_req;
   wire [3:0] #1 dma_op_queue_id;
   wire [1:0] #1 dma_op_code_ack;
     
   // DMA data and flow control 
   wire #1 dma_vld_n2c;
   wire #1 dma_vld_c2n;
   wire [31:0] #1 dma_data;
   wire #1     dma_q_nearly_full_c2n;
   wire #1     dma_q_nearly_full_n2c;
      
   // CNET <-> SRAM with simulated delays
   wire [19:0] #1 sram1_addr;
   wire [35:0] #1 sram1_data;
   wire [3:0]  #1 sram1_bw;
   wire        #1 sram1_we;
   wire        #1 sram1_zz;

   wire [19:0] #1 sram2_addr;
   wire [35:0] #1 sram2_data;
   wire [3:0]  #1 sram2_bw;
   wire        #1 sram2_we;
   wire        #1 sram2_zz;

   wire        ddr2_odt0;
   wire        ddr2_clk0;
   wire        ddr2_clk0b;
   wire        ddr2_clk1;
   wire        ddr2_clk1b;
   wire        ddr2_cke;
   wire        ddr2_csb;
   wire        ddr2_rasb;
   wire        ddr2_casb;
   wire        ddr2_web;
   wire [3:0]  ddr2_dm;
   wire [1:0]  ddr2_ba;
   wire [12:0] ddr2_addr;
   wire [31:0] ddr2_dq;
   wire [3:0]  ddr2_dqs;
   wire [3:0]  ddr2_dqs_n;
   wire        ddr2_rst_dqs_div;


   wire [7:0]    rp_data;
   // Debug signals
   wire [28:0]   cpci_debug_data;
   wire [1:0]    cpci_debug_clk;

   wire 	 cnet_clk_sel;

   // Allow reprogramming
   wire          cpci_allow_reprogram;

   wire [3:0] 	 cpci_id;

   // CPCI reprogramming signals
   wire     cpci_rp_done;
   wire     cpci_rp_init_b;
   wire     cpci_rp_cclk;

   wire      cpci_rp_en;
   wire      cpci_rp_prog_b;
   wire      cpci_rp_din;
   
cpci_top cpci_top(
            .AD (AD),
            .CBE (CBE),
            .PAR (PAR),
            .FRAME_N (FRAME_N),
            .TRDY_N (TRDY_N),
            .IRDY_N (IRDY_N),
            .STOP_N (STOP_N),
            .DEVSEL_N (DEVSEL_N),
            .IDSEL (IDSEL),
            .INTR_A (INTR_A),
            .PERR_N (PERR_N),
            .SERR_N (SERR_N),
            .REQ_N (REQ_N),
            .GNT_N (GNT_N),
            .RST_N (RST_N),
            .PCLK (PCLK),
            .nclk (clk_62_5),
            .cnet_reset (cnet_reset),
            .phy_int_b (phy_int_b),
            .cpci_rd_wr_L (cpci_rd_wr_L),
            .cpci_req (cpci_req),
            .cpci_addr (cpci_addr),
            .cpci_data (cpci_data),
            .cpci_wr_rdy (cpci_wr_rdy),
            .cpci_rd_rdy (cpci_rd_rdy),
            .cnet_err (nf2_err),

	    //DMA intfc 
            // --- CPCI DMA handshake signals 
   	    //output:
	    .cpci_dma_op_code_req ( dma_op_code_req ),//[1:0] 
	    .cpci_dma_op_queue_id ( dma_op_queue_id ),//[3:0] 

	    //input:
	    .cpci_dma_op_code_ack ( dma_op_code_ack ),//[1:0] 
   
	    // DMA data and flow control 
	    //output:
	    .cpci_dma_vld_c2n ( dma_vld_c2n ),
	    .cpci_dma_q_nearly_full_c2n ( dma_q_nearly_full_c2n ),
	  
	    //input: 
	    .cpci_dma_vld_n2c ( dma_vld_n2c ),
	    .cpci_dma_q_nearly_full_n2c ( dma_q_nearly_full_n2c ), 
		  
	    //inout: 
	    .cpci_dma_data ( dma_data ),//[DMA_DATA_WIDTH-1:0] 
		  
            .rp_cclk (rp_cclk),
            .rp_prog_b (rp_prog_b),
            .rp_init_b (rp_init_b),
            .rp_cs_b (rp_cs_b),
            .rp_rdwr_b (rp_rdwr_b),
            .rp_data (rp_data),
            .rp_done (rp_done),
            .cpci_led (cpci_led),
            .cpci_debug_data (cpci_debug_data),
            .cpci_debug_clk (cpci_debug_clk),
            .allow_reprog (cpci_allow_reprog),
	    .cnet_clk_sel   (cnet_clk_sel),
	    .cpci_jmpr           (1'b1),
	    .cpci_id             (cpci_id)
          );

   nf2_top nf2_top (

        //-- RGMII interfaces for 4 MACs.
        .rgmii_0_txd              (rgmii_0_tx_d),
        .rgmii_0_tx_ctl           (rgmii_0_tx_ctl),
        .rgmii_0_txc              (rgmii_0_tx_clk),
        .rgmii_0_rxd              (rgmii_0_rx_d),
        .rgmii_0_rx_ctl           (rgmii_0_rx_ctl),
        .rgmii_0_rxc              (rgmii_0_rx_clk),

        .rgmii_1_txd              (rgmii_1_tx_d),
        .rgmii_1_tx_ctl           (rgmii_1_tx_ctl),
        .rgmii_1_txc              (rgmii_1_tx_clk),
        .rgmii_1_rxd              (rgmii_1_rx_d),
        .rgmii_1_rx_ctl           (rgmii_1_rx_ctl),
        .rgmii_1_rxc              (rgmii_1_rx_clk),

        .rgmii_2_txd              (rgmii_2_tx_d),
        .rgmii_2_tx_ctl           (rgmii_2_tx_ctl),
        .rgmii_2_txc              (rgmii_2_tx_clk),
        .rgmii_2_rxd              (rgmii_2_rx_d),
        .rgmii_2_rx_ctl           (rgmii_2_rx_ctl),
        .rgmii_2_rxc              (rgmii_2_rx_clk),

        .rgmii_3_txd              (rgmii_3_tx_d),
        .rgmii_3_tx_ctl           (rgmii_3_tx_ctl),
        .rgmii_3_txc              (rgmii_3_tx_clk),
        .rgmii_3_rxd              (rgmii_3_rx_d),
        .rgmii_3_rx_ctl           (rgmii_3_rx_ctl),
        .rgmii_3_rxc              (rgmii_3_rx_clk),

        .gtx_clk                  (clk),      // common TX clk reference 125MHz.


        // DDR signals

        .ddr_clk_200              (clk_ddr_200),
        .ddr_clk_200b             (~clk_ddr_200),

        .ddr2_odt0         (ddr2_odt0),
        .ddr2_clk0         (ddr2_clk0), 
        .ddr2_clk0b        (ddr2_clk0b), 
        .ddr2_clk1         (ddr2_clk1), 
        .ddr2_clk1b        (ddr2_clk1b), 
        .ddr2_cke          (ddr2_cke),
        .ddr2_csb          (ddr2_csb),
        .ddr2_rasb         (ddr2_rasb),
        .ddr2_casb         (ddr2_casb),
        .ddr2_web          (ddr2_web),
        .ddr2_dm           (ddr2_dm),
        .ddr2_ba           (ddr2_ba),
        .ddr2_address      (ddr2_addr), 
        .ddr2_dq           (ddr2_dq),
        .ddr2_dqs          (ddr2_dqs),
        .ddr2_dqs_n        (ddr2_dqs_n),
        .ddr2_rst_dqs_div_in    (ddr2_rst_dqs_div),
        .ddr2_rst_dqs_div_out   (ddr2_rst_dqs_div),		

     	
        // CPCI interface and clock

        .cpci_clk                 (clk_62_5),   // 62.5 MHz
        .cpci_rd_wr_L             (cpci_rd_wr_L),
        .cpci_req                 (cpci_req),
        .cpci_addr                (cpci_addr),
        .cpci_data                (cpci_data),
        .cpci_wr_rdy              (cpci_wr_rdy),
        .cpci_rd_rdy              (cpci_rd_rdy),
        .nf2_err                  (nf2_err),  // was cnet_err

        // ---- SRAM 1

        .sram1_addr               (sram1_addr),
        .sram1_data               (sram1_data),
        .sram1_we                 (sram1_we),
        .sram1_bw                 (sram1_bw),
        .sram1_zz                 (sram1_zz),
        
        // ---- SRAM 2

        .sram2_addr               (sram2_addr),
        .sram2_data               (sram2_data),
        .sram2_we                 (sram2_we),
        .sram2_bw                 (sram2_bw),
        .sram2_zz                 (sram2_zz),

        // --- CPCI DMA

        // --- CPCI DMA handshake signals
        // output:
        .dma_op_code_ack ( dma_op_code_ack ),

        // input: 
        .dma_op_code_req ( dma_op_code_req ),
        .dma_op_queue_id ( dma_op_queue_id ),
     
        // DMA data and flow control 
        // input:
        .dma_vld_c2n ( dma_vld_c2n ),
	.dma_q_nearly_full_c2n ( dma_q_nearly_full_c2n ),

        // output: 
        .dma_vld_n2c ( dma_vld_n2c ),
	.dma_q_nearly_full_n2c ( dma_q_nearly_full_n2c ), 
		    
	//inout: 
        .dma_data ( dma_data ),

        // --- CPCI Debug Data
        .cpci_debug_data          (cpci_debug_data),
     	
        // ---  MDC/MDIO PHY control signals

        .phy_mdc                  (phy_mdc),
        .phy_mdio                 (phy_mdio),
        .serial_TXP_0        (serial_TXP_0),
	.serial_TXN_0        (serial_TXN_0),
        .serial_RXP_0        (serial_RXP_0),
        .serial_RXN_0        (serial_RXN_0),
        .serial_TXP_1        (serial_TXP_1),
        .serial_TXN_1        (serial_TXN_1),
        .serial_RXP_1        (serial_RXP_1),
        .serial_RXN_1        (serial_RXN_1),


        //--- CPCI reprogramming interface
        .cpci_rp_done     (cpci_rp_done),
        .cpci_rp_init_b   (cpci_rp_init_b),
        .cpci_rp_cclk     (cpci_rp_cclk),

        .cpci_rp_en       (cpci_rp_en),
        .cpci_rp_prog_b   (cpci_rp_prog_b),
        .cpci_rp_din      (cpci_rp_din),


        //--- Debug bus (goes to LA connector)

        .debug_led                (debug_led),
        .debug_data               (),
        .debug_clk                (),
        

        // core clock - must also be same as sram clock

        .core_clk                 (clk),
        
        // Misc
        
        .nf2_reset                (cnet_reset || (~host32_is_active))    // driven by CPCI

    );



   // ---- CPCI reprogramming interface
   cpci_program_iface cpci_program_iface 
   (
      .cpci_rp_done     (cpci_rp_done),
      .cpci_rp_init_b   (cpci_rp_init_b),
      .cpci_rp_cclk     (cpci_rp_cclk),

      .cpci_rp_en       (cpci_rp_en),
      .cpci_rp_prog_b   (cpci_rp_prog_b),
      .cpci_rp_din      (cpci_rp_din)
   );



   // ---- SRAM 1
   // add delays to avoid simulation errors.
   
   reg          sram1_we_del;
   always @* #2 sram1_we_del = sram1_we;
   reg [3:0]    sram1_bw_del;
   always @* #2 sram1_bw_del = sram1_bw;
   reg [19:0]   sram1_addr_del;
   always @* #2 sram1_addr_del = sram1_addr;

   cy7c1370 sram1 (
                  .d      (sram1_data),
                  .clk    (clk),
                  .a      (sram1_addr_del[18:0]),
                  .bws    (sram1_bw_del),
                  .we_b   (sram1_we_del),
                  .adv_lb (1'b0),
                  .ce1b   (1'b0),
                  .ce2    (1'b1),
                  .ce3b   (1'b0),
                  .oeb    (1'b0),
                  .cenb   (1'b0),
                  .mode   (1'b0)   // dont care cos we dont burst
                  );
                                                                                           
   // ---- SRAM 2
   // add delays to avoid simulation errors.
   
   reg          sram2_we_del;
   always @* #2 sram2_we_del = sram2_we;
   reg [3:0]    sram2_bw_del;
   always @* #2 sram2_bw_del = sram2_bw;
   reg [19:0]   sram2_addr_del;
   always @* #2 sram2_addr_del = sram2_addr;

   cy7c1370 sram2 (
                  .d      (sram2_data),
                  .clk    (clk),
                  .a      (sram2_addr_del[18:0]),
                  .bws    (sram2_bw_del),
                  .we_b   (sram2_we_del),
                  .adv_lb (1'b0),
                  .ce1b   (1'b0),
                  .ce2    (1'b1),
                  .ce3b   (1'b0),
                  .oeb    (1'b0),
                  .cenb   (1'b0),
                  .mode   (1'b0)   // dont care cos we dont burst
                  );

   // ---- DDR
   ddr2 dram1 (
               .ck      (ddr2_clk0),
               .ck_n    (ddr2_clk0b),
               .cke     (ddr2_cke),
               .cs_n    (ddr2_csb),
               .ras_n   (ddr2_rasb),
               .cas_n   (ddr2_casb),
               .we_n    (ddr2_web),
               .dm_rdqs (ddr2_dm[1:0]),
               .ba      (ddr2_ba),
               .addr    (ddr2_addr),
               .dq      (ddr2_dq[15:0]),
               .dqs     (ddr2_dqs[1:0]),
               .dqs_n   (ddr2_dqs_n[1:0]),
               .rdqs_n  (),
               .odt     (ddr2_odt0)
               );

   ddr2 dram2 (
               .ck      (ddr2_clk1),
               .ck_n    (ddr2_clk1b),
               .cke     (ddr2_cke),
               .cs_n    (ddr2_csb),
               .ras_n   (ddr2_rasb),
               .cas_n   (ddr2_casb),
               .we_n    (ddr2_web),
               .dm_rdqs (ddr2_dm[3:2]),
               .ba      (ddr2_ba),
               .addr    (ddr2_addr),
               .dq      (ddr2_dq[31:16]),
               .dqs     (ddr2_dqs[3:2]),
               .dqs_n   (ddr2_dqs_n[3:2]),
               .rdqs_n  (),
               .odt     (ddr2_odt0)
               );


   always #4 clk_gen = clk_gen + 'h1;         //  125MHz or 62.5Mhz
   initial begin clk_gen = 0; end
   assign clk = cnet_clk_sel ? clk_gen[0] : clk_gen[1]; 

   always #8 clk_62_5 = !clk_62_5;              //  62.5MHz
   initial begin clk_62_5 = 0; end

   always #2.5 clk_ddr_200 = !clk_ddr_200;      //  200MHz
   initial begin clk_ddr_200 = 0; end

   // Respond to sleep requests on the SRAM
   always @(posedge clk)
   begin
      if (sram1_zz)
              $display($time, " WARNING: Unable to simulate sleep mode on SRAM1: %m");
      if (sram2_zz)
              $display($time, " WARNING: Unable to simulate sleep mode on SRAM2: %m");
   end

   // Simulate the LEDs
   reg debug_led_d1, cpci_led_d1;

   always @(posedge clk)
   begin
      debug_led_d1 <= debug_led;
      cpci_led_d1 <= cpci_led;
   end

   always @(posedge clk)
   begin
      if (debug_led != debug_led_d1)
         $display($time, " INFO: Debug LED: %b", debug_led);
      if (cpci_led != cpci_led_d1)
         $display($time, " INFO: CPCI LED: %b", cpci_led);
   end
endmodule // c_board

