#!/usr/bin/perl -w

#
# Common run script that runs simulations and processes the output
#
# $Id: run.pl 1 2009-10-08 18:23:02Z root $
#

use strict;
use NF2::TeamCity;
use NF2::Base;
use Getopt::Long;
use Cwd;
use File::Copy;

# Local variables that should be overridden by the local config script
my %config = (
	'test_desc' => undef,
	'test_num' => 1,
	'opts' => '',
	'extra_files' => '',
	'extra_checks' => '',
	'log' => 'my_sim.log',
	'finish' => 1000000,
);

# Configuration file
my $configFile = 'config.txt';

# Finish time file
my $finishFile = 'config.sim';

# Name of the script to create packets
my $makePkts = 'make_pkts.pl';

# Location of modelsim.ini file
my $modelsimIni = '../vsim_beh/modelsim.ini';

# Files to remove before running the simulation
my @rmFiles = ('PASS', 'FAIL', 'GUI');

# Work out the test name
my $dir = cwd;
$dir =~ s/^.*\///;
my $test = "sim.$dir";

# Check that the correct environment variables are set
check_NF2_vars_set();

# Test directory
$ENV{'NF2_DESIGN_DIR'} =~ /.*\/([^\/]+)/;
my $testDir = $ENV{'NF2_WORK_DIR'} . "/verif/$1";

# Program to perform packet comparison
my $compare = $ENV{'NF2_ROOT'} . '/bin/nf2_compare.pl';



# Parse the command line arguments
my $sim = '';   	# Which simulator to use
my $dump = 0; 		# Dump the output
my $gui = 0;		# Run the GUI
my $ci = '';		# Continuous integration program
my $citest = '';	# Continuous integration test name

unless ( GetOptions ( "dump" => \$dump,
		      "gui" => \$gui,
		      "sim=s" => \$sim,
		      "ci=s" => \$ci,
		      "citest=s" => \$citest,
		     )
       ) { usage(); exit 1; }



# Verify that a simulator has been set
if ($sim eq '' || ($sim ne 'vsim' && $sim ne 'vcs')) {
	print "Unkown simulator \"$sim\". Supported simulators: vcs vsim\n";
	exit 1;
}

# Verify that the continuous integration program is correct if set
if ($ci ne '' && $ci ne 'teamcity') {
	print "Unkown continuous integration \"$ci\". Supported CI programs: teamcity\n";
	exit 1;
}
if ($ci ne '' && $citest eq '') {
	print "The name of the test was not specified in 'citest'\n";
	exit 1;
}
tcDisableOutput if ($ci ne 'teamcity');
if ($ci eq 'teamcity') {
	$test = $citest . tcGetTestSeparator . $test;
}

# -------  Print a test started message  -------
# After this point CI messages should be printed
tcTestStarted($test);

my $good = 1;

# Verify that the verilog files have been compiled
$good &= &checkSimCompiled if $good;

# Read the configuration
$good &= &readConfig if $good;

# Attempt to run the actual test
if ($good) {
	print "--- Running test.\n";
	print "$config{'test_desc'}\n";
}

# Generate the packets
if ($good) {
	print "--- Generating packets...\n";

	# Run the script
	my $makePktsOut;
	$makePktsOut = `perl $makePkts 2>&1`;
	print $makePktsOut;

	# Verify the return code
	if ($? != 0) {
		print "--- Test failed ($dir) - $makePkts broke!.\n";
		tcTestFailed($test, '$makePkts broke', $makePktsOut);
		$good = 0;
	}
}

# Create the finish time file
#
# Note: always run the simulation but only run it for 1 time step if 
# things are not good since the reg_defines.h is generated from it
$config{'finish'} = 1 if (!$good);
$good &= &createFinishFile;

# Run the simulation (again, always try to run since
# the reg_defines.h file is generated from the output)
$good &= &runSim($good);

# Validate the output
$good &= &validateOutput if $good;

# Write a success/fail file and return an exit code as appropriate
tcTestFinished($test);
if ($gui) {
	system('touch GUI');
	exit 0;
}
elsif ($good) {
	system('touch PASS');
	exit 0;
}
else {
	system('touch FAIL');
	exit 1;
}

########################################

#########################################################
# usage
#   print usage information
sub usage {
  my $cmd = $0;
  $cmd =~ s/.*\///;
  print <<"HERE1";
NAME
   $cmd - Run a simulation. (Should be invoked by the nf21_run_test.pl command.)

SYNOPSIS
   $cmd 
        [--sim <vsim|vcs>] 
	[--ci <teamcity>]
        [--gui]
        [--dump]

   $cmd --help  - show detailed help

HERE1

##  return unless ($help);
##  print < < "HERE";
##sub usage {

}


#########################################################
# readConfig
#   read the configuration file
sub readConfig {
	my $ok = 1;

	print "--- Reading configuration file\n";

	# Verify that the config file exists
	if (! -f $configFile) {
		&printError("configuration file '$configFile' does not exist");
		return 0;
	}

	# Open and read the configuration file
	if (open CONFIG, $configFile) {
		while (<CONFIG>) {
			chomp;

			# Remove comments
			s/#.*//;

			# Kill as much white space as possible at beginning
			s/^\s*//;

			# Skip blanks
			next if /^$/;

			# Work out the components
			/^(\w+)\s*=\s*((\S.*)?\S)?\s*/;
			my ($key, $val) = ($1, $2);
			$val = '' if (!defined($val));

			# Set the value in the hash
			$config{$key} = $val;

			print "Seen: $key => $val\n";
		}
		close CONFIG;
	}
	else {
		&printError("Unable to open configuration file '$configFile'");
		return 0;
	}

	# Verify that all necessary variables have been set
	foreach my $key (keys %config) {
		if (!defined($config{$key})) {
			&printError("'$key' not defined in configuration file");
			$ok = 0;
		}
	}

	return $ok;
}


#########################################################
# validateOutput
#   validate the output of the tests
sub validateOutput {
	if (!$gui) {
		print "--- Simulation is complete. Validating the output.\n";

		# Check the log for errors
		my $logErrors = `grep -i error $config{'log'} | grep -v -i ERROR_DETECT`;
		print $logErrors;
		if ($? >> 8 != 1) {
			print "--- Test failed ($dir) - see $config{'log'} for errors.\n";
			tcTestFailed($test, 'Errors seen in simulation output', $logErrors);
			return 0;
		}

		# Run any additional tests on the output
		if ($config{'extra_checks'} ne '') {
			$logErrors = `$config{'extra_checks'} --log $config{'log'}`;
			print $logErrors;

			if ($?) {
				print "--- Test failed ($dir) - see $config{'log'} for errors.\n";
				tcTestFailed($test, 'Errors seen in simulation output', $logErrors);
				return 0;
			}
		}


		# Compare the simulation output with the expected output
		my $compareOutput = `$compare`;
		print $compareOutput;
		if ($?) {
			print "--- Test failed ($dir) - expected and seen data differs.\n";
			tcTestFailed($test, 'Expected and seen data differs', $compareOutput);
			return 0;
		}

		print "--- Test PASSED ($dir) \n";
		unlink('test.dump');
	}
	else {
		print "--- Simulation is complete. Cannot evaluate correctness due to GUI mode.\n";
	}

	return 1;
}


#########################################################
# runSim
#   run the simulation
sub runSim {
	my $good = shift;

	my $ok = 1;

	print "--- Running the simulation (takes a while). Logging to $config{'log'}\n";
	unlink($config{'log'}, @rmFiles);

	if ($sim eq 'vcs') {
		print "--- Running my_sim\n";

		if (system("$testDir/my_sim > $config{'log'} 2>&1") != 0) {
			if ($good) {
				print "--- Test Failed.\n";
				tcTestFailed($test, 'Error when running simulator', '');
			}
			$ok = 0;
		}
	}
	elsif ($sim eq 'vsim') {
		print "--- Running vsim\n";

		# Work out if there are any extra files to process
		if ($dump) {
			$config{'extra_files'} .= ' dump';
		}

		# Check if we should invoke the gui
		my $cmd;
		if ($gui) {
			$config{'opts'} .= "";
			$cmd = 'view object; view wave;';
		}
		else {
			# default finish time is 1000000ns. 
			# the config.txt overrides the finish time.
			$config{'opts'} .= " -c -l $config{'log'}";
			$cmd = "run -all";
		}

		# Check if we need to disable the DRAM debug information
		if (defined($ENV{'NF2_NO_DRAM_DEBUG'})) {
			$config{'opts'} .= " -g/testbench/u_board/dram1/DEBUG=0 -g/testbench/u_board/dram2/DEBUG=0";
		}

		# Set the modelsim environment variable if the modelsim.ini file exists
		if (-f $modelsimIni) {
			$ENV{'MODELSIM'} = $modelsimIni;
		}

		if (system("vsim $config{'opts'} -voptargs=\"+acc\" testbench glbl $config{'extra_files'} -do \"${cmd}\"") != 0) {
			if ($good) {
				print "--- Test Failed.\n";
				tcTestFailed($test, 'Error when running simulator', '');
			}
			$ok = 0;
		}
	} 

	return $ok;
}


#########################################################
# checkSimCompiled 
#   check that the simulator has been compiled
sub checkSimCompiled {
	if ($sim eq 'vcs') {
		if ( ! -x "$testDir/my_sim" ) {

			&printError("Cannot find executable my_sim at $testDir");
			return 0;
		}
	}
	else {
		if ( ! -d "${testDir}/vsim_beh" ) {
			&printError("Cannot find directory vsim_beh at $testDir");
			return 0;
		}
	}

	# Everything is good
	return 1;
}


#########################################################
# printError
#   prints an error including through all of the defined CIs
sub printError {
	my $err = shift;

	warn "Error: $err\n";
	tcTestFailed($test, $err, '');
}


#########################################################
# createFinishFile 
#   create the file that instructs the simulator when to finish
sub createFinishFile {
	if (open FINISH, ">$finishFile") {
		print FINISH "FINISH=$config{'finish'}\n";
		close FINISH;
	}
	else {
		&printError("Unable to open finish time file '$finishFile' for writing");
		return 0;
	}

}
