#!/usr/bin/perl -w

#
# Perl script to identify whether a project has changed
# $Id: is_project_unchanged.pl 1 2009-10-08 18:23:02Z root $
#
# Returns 0 if the buildfile is newer than all dependent files
# Returns non-zero if the buildfile is older or non-existant
#

use strict;
use NF2::Base;
use File::Find;

my $project;

my $buildMtime;

my $projectDir = 'projects';
my $synthDir = 'synth';
my $includeFile = 'include/lib_modules.txt';
my $verLibDir = 'lib/verilog';
my $libDir = 'lib';

my $buildOkFile = 'BUILD_GOOD';

my $changed = 0;

my @defaultModules = ('utils', 'common');
my @verLibDirsToSearch = ('src', 'synth');
my @nonVerLibs = ('Perl5');

# check vars are set.
check_NF2_vars_set();

my $nf2_root = $ENV{'NF2_ROOT'};

# Process arguments
&processArgs();

# Get the build mtime
&getBuildMtime();

# Attempt to find changed files in the project directory
&findChangesIn("$nf2_root/$projectDir/$project");


# Search the verilog libraries for changes
&findModuleChanges();

# Search the various important library directories for changes
&findLibChanges();

exit $changed;

#########################################################
sub processArgs {
	if ($#ARGV != 0) {
		die "Must provide the project name as the only argument";
	}

	$project = $ARGV[0];
}


#
# Get the modification time of the build ok file (if it exists
# and if a bit file exists)
#
sub getBuildMtime {
	my $projSynth = "$nf2_root/$projectDir/$project/$synthDir";
	my $warned = 0;

	# Check to see if the bitfile exists
	if (-d $projSynth) {
		# Verify we have a bitfiles
		if (glob("$projSynth/*.bit")) {
			my $file = "$nf2_root/$projectDir/$project/$buildOkFile";
			if (-r $file) {
				$buildMtime = (stat($file))[9];
			}
		}
	}

	if (!defined($buildMtime)) {
		# No bitfiles/buildfiles -- so return 1
		exit 1;
	}
}


#
# function to skip subversion directories
#
sub skipSubversion {
	my @ret;

	foreach my $fileOrDir (@_) {
		push @ret, $fileOrDir if ($fileOrDir ne '.svn');
	}

	return @ret;
}


#
# Identify changed files in the project directory
#
sub identifyChangedFiles {
	# Skip directories
	if (! -d $File::Find::name) {
		# Get the mtime and compare
		my $mtime = (stat($File::Find::name))[9];
		if ($mtime > $buildMtime) {
			print $File::Find::name . "\n";
			$changed = 1;
		}
	}
}


#
# Find changes in the Verilog modules
#
sub findModuleChanges {
	my $moduleFile = "$nf2_root/$projectDir/$project/$includeFile";

	# Verify that a module file exists
	if (-r $moduleFile) {
		# Read in the module file
		my @modules = @defaultModules;
		push @modules, split(/\n/, `cat $moduleFile`);

		foreach my $module (@modules) {
			foreach my $dir (@verLibDirsToSearch) {
				my $path = "$nf2_root/$verLibDir/$module/$dir";
				if (-d $path) {
					findChangesIn($path);
				}
			}
		}
	}
}


#
# Find changes in non-Verilog libraries
#
sub findLibChanges {
	foreach my $dir (@nonVerLibs) {
		my $path = "$nf2_root/$libDir/$dir";
		if (-d $path) {
			findChangesIn($path);
		}
	}
}


#
# Find changed files in a certain directory
#
sub findChangesIn {
	my $dir = shift;

	find(
		{

			wanted => \&identifyChangedFiles, 
			preprocess => \&skipSubversion
		}, 
		$dir);
}

