#!/usr/bin/perl -w

#
# $Id: build.pl 1 2009-10-08 18:23:02Z root $
#
# Script to build all hardware and software and run all tests
#
# Hopefully the output will be in a format that TeamCity can parse
#

use strict;
use Cwd;
$|++;

# Location of work and config directories
# - relative to NF2_ROOT
# - if NF2_ROOT is the work dir then use '.' -- don't leave blank
my $workDir = 'builders/work';
my $confDir = 'builders/conf';
my $progDir = 'builders/teamcity';

# Location of contrib library
my $contribURL = "http://netfpga.org/netfpga_lib.tar.gz";
my $contribFile = "netfpga_lib.tar.gz";
my $contribExtractSuccess = 'contrib.extracted';
my $contribMD5File = 'contrib.md5';

# Location of projects
my $projects = "projects";

# Location of verif
my $verif = "verif";

# Location of synth
my $synth = "synth";

# Location of sw
my $sw = "sw";

# Location of regress
my $regress = "regress";

# Simulation command
my $simCmd = "bin/nf2_run_test.pl";

# Regression test command
my $regressCmd = "bin/nf2_regress_test.pl";

# Fetch memory models cmd
my $fetchMemModelsCmd = "$progDir/fetch_mem_models.pl";

# File containing list of software projects to build
my $swToBuild = "build_software.txt";

# File containing list of projects to build
my $projToBuild = "build_projects.txt";

# Files to create if builds pass/fail
my $buildOkFile = 'BUILD_GOOD';
my $buildFailFile = 'BUILD_FAILED';

# Check for project changes command
my $isProjectUnchangedCmd = "$progDir/is_project_unchanged.pl";

# Work out the NF2_ROOT
my $nf2_root = getcwd;
$nf2_root =~ s/^(.*\/NF2)\/.*$/$1/;

# Set appropriate environment variables
$ENV{'NF2_ROOT'} = $nf2_root;
$ENV{'NF2_WORK_DIR'} = "$nf2_root/$workDir";
if (defined($ENV{'PERL5LIB'})) {
	$ENV{'PERL5LIB'} = $ENV{'PERL5LIB'} . ':' . $nf2_root . '/lib/Perl5';
}
else {
	$ENV{'PERL5LIB'} = $nf2_root . '/lib/Perl5';
}
if (defined($ENV{'PYTHONPATH'})) {
	$ENV{'PYTHONPATH'} = $ENV{'PYTHONPATH'} . ':' . $nf2_root . '/lib/python';
}
else {
	$ENV{'PYTHONPATH'} = $nf2_root . '/lib/python';
}
$ENV{'NF2_NO_DRAM_DEBUG'} = '1';

chdir $nf2_root;

# Import the TeamCity library (need to update the Perl library
# path first)
push @INC, $nf2_root . '/lib/Perl5';
require NF2::TeamCity;
import NF2::TeamCity;

# Update the root as reported to team city
tcSetRoot($nf2_root);

# Create the work directory if necessary
&createWorkDir;

# Fetch the contributed library
&fetchContrib;

# Fetch the memory models
system($fetchMemModelsCmd);

# Attempt to build the sources
my $ret;
$ret = tcRunMake('top.level.build', 'make', '.');

exit 1 if ($ret != 0);

# Build all software projects
&buildSoftware;

# Run all simulations
my @projects;
&readProjects;
&buildProjects;

exit;

##############################################################################
#   Functions
##############################################################################

########################################
#
# Fetch the contributed libraries
# and extract to their correct location
#
########################################
sub fetchContrib {

	my $suite = 'fetch.3rd.party.libs';
	my $ok = 1;
	my $test;

	tcTestSuiteStarted($suite);

	# Perform the wget
	if ($ok) {
		$test = $suite . tcGetTestSeparator() . 'wget';
		tcTestStarted($test);

		my $wgetResult = `wget -O $workDir/$contribFile $contribURL 2>&1`;
		if ($? != 0) {
			$ok = 0;
			tcTestFailed($test, "Unable to fetch contributed libraries", "URL: $contribURL");
			
			unlink "$workDir/$contribExtractSuccess" if (-f "$workDir/$contribExtractSuccess");

		}
		tcTestFinished($test);
	}

	# Verify if we need to extract the source
	my $needExtract = 1;
	if ($ok) {
		# Calculate the MD5 checksum of the new file
		my $newMD5 = `md5sum $workDir/$contribFile`;
		$newMD5 = (split /\s+/, $newMD5)[0];

		# Check to see if we've extracted previously
		if (-f "$workDir/$contribExtractSuccess") {
			# Check to see if we have an MD5 checksum
			if (-f "$workDir/$contribMD5File") {
				my $oldMD5 = `cat $workDir/$contribMD5File`;
				chomp $oldMD5;

				# Check to see if the sums are identical
				$needExtract = 0 if ($oldMD5 eq $newMD5);
			}
			else {
				# Remove the file
				unlink "$workDir/$contribExtractSuccess";
			}
		}

		# Write the new MD5 sum
		open MD5FILE, ">$workDir/$contribMD5File";
		print MD5FILE "$newMD5\n";
		close MD5FILE;
	}

	# Extract the source
	if ($ok) {
		$test = $suite . tcGetTestSeparator() . 'extract';
		tcTestStarted($test);

		if ($needExtract) {

			chdir '..';
			system("tar -xzf NF2/$workDir/$contribFile");
			chdir 'NF2';

			if ($? != 0) {
				$ok = 0;
				tcTestFailed($test, "Error extracting files from archive", "Source: $contribFile");

			}
			else {
				`touch $workDir/$contribExtractSuccess`;
			}

			tcTestFinished($test);
		}
		else {
			tcTestIgnored($test, 'Contributed libraries unchanged');
			tcTestFinished($test);
		}
	}

	tcTestSuiteFinished($suite);

	# Check to see if everything succeeded
	if (!$ok) {
		exit 1;
	}
}


#
# Create the work directory if it doesn't already exist
#
sub createWorkDir {
	if (! -d $workDir) {
		my @path = split(/\//, $workDir);

		my $pathSoFar = '';
		foreach my $dir (@path) {
			mkdir "${pathSoFar}${dir}" if (! -d "${pathSoFar}${dir}");
			$pathSoFar .= "$dir/";
		}
	}
}


#
# Build all of the software components
#
sub buildSoftware {
	my $suite = 'software';

	my $ok = 1;

	tcTestSuiteStarted($suite);

	# Check to see if we have the configuraiton file of software to build
	if (-f "$confDir/$swToBuild") {
		my @sw;

		# Read the file into memory first to avoid warnings
		# with fork and closing of files
		open TO_BUILD, "$confDir/$swToBuild";
		while (<TO_BUILD>) {
			chomp;

			# Ignore comments
			s/#.*//;

			# Strip leading/trailing white space
			s/^\s+//;
			s/\s+$//;

			# Ignore blank lines
			next if /^$/;

			# Store the entry
			push @sw, $_;
		}
		close TO_BUILD;

		# Process the actual entries
		foreach my $line (@sw) {
			# Split the line up into the components
			my ($test, $path, $cmd) = split /:/, $line;

			my $ret = tcRunMake('', $suite.tcGetTestSeparator().$test, $path, $cmd);
			
			$ok &= ($ret == 0);
		}
	}
	else {
		print "Could not find the software to build file: '$confDir/$swToBuild'\n";

	}

	# Record the start of the suite
	tcTestSuiteFinished($suite);

	return $ok;
}


#
# Read the list of projects
#
sub readProjects {
	my $suite = 'read.projects';

	my $ok = 1;

	tcTestSuiteStarted($suite);

	# Check to see if we have the configuraiton file of projects to build
	if (-f "$confDir/$projToBuild") {
		# Read the file into memory first to avoid warnings
		# with fork and closing of files
		open TO_BUILD, "$confDir/$projToBuild";
		while (<TO_BUILD>) {
			chomp;

			# Ignore comments
			s/#.*//;

			# Strip leading/trailing white space
			s/^\s+//;
			s/\s+$//;

			# Ignore blank lines
			next if /^$/;

			# Store the entry
			push @projects, $_;
		}
		close TO_BUILD;
	}
	else {
		print "Could not find the projects to build file: '$confDir/$projToBuild'\n";

	}

	# Record the start of the suite
	tcTestSuiteFinished($suite);

	return $ok;
}


#
# Build the projects
#
sub buildProjects {
	my $suite = 'build.projects';

	my $ok = 1;

	tcTestSuiteStarted($suite);

	# Process the actual entries
	foreach my $line (@projects) {
		# Split the line up into the components
		my ($proj, $path, $options) = split /:/, $line;
		my $test = $suite. tcGetTestSeparator() . $proj;

		# Process the options if any exist
		my %opts;
		if (defined($options)) {
			my @opts = split(/,/, $options);
			foreach my $opt (@opts) {
				$opts{$opt} = 1;
			}
		}

		# Run simulation, synthesis and regressions for the project if possible
		tcTestStarted($test);

		# Check for changes to the project
		if (&isChanged($path)) {
			my $simOk = 1;
			my $synthOk = 1;
			my $swOk = 1;
			my $regressOk = 1;

			# Remove the good/fail files
			unlink("$projects/$path/$buildOkFile", "$projects/$path/$buildFailFile");

			# Always run sim and synth, but only run regress if synth passes
			$simOk &= runSim($proj, $path, $test) if (!defined($opts{'no_sim'}));;
			$synthOk &= runSynth($proj, $path, $test);
			$swOk &= runSWBuild($proj, $path, $test);
			$regressOk &= runRegress($proj, $path, $test) if ($synthOk && $swOk);

			# Verify the output
			my $projOk = $simOk & $synthOk & $regressOk;
			if ($projOk) {
				open RESULT, ">$projects/$path/$buildOkFile";
				close RESULT;
			}
			else {
				open RESULT, ">$projects/$path/$buildFailFile";
				close RESULT;

				tcTestFailed($test, 'Project build failed', 
					'Simulation, synthesis and/or regressions failed');
			}

			$ok &= $projOk;
		}
		else {
			tcTestIgnored($test, 'Source files unchanged');
		}

		tcTestFinished($test);
	}

	# Record the start of the suite
	tcTestSuiteFinished($suite);

	return $ok;
}


#
# Check if a project is changed
#
sub isChanged {
	my $path = shift;

	# Run synthesis for the project if possible
	if (-x $isProjectUnchangedCmd) {
		$ENV{'NF2_DESIGN_DIR'} = "$nf2_root/$projects/$path";

		my $changes = `$isProjectUnchangedCmd $path`;
		if ($? == 0) {
			return 0;
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}


#
# Run simlations for a single project
#
sub runSim {
	my $proj = shift;
	my $path = shift;
	my $test = shift;

	my $ok = 1;

	# Adjust the test name
	$test .= tcGetTestSeparator() . 'simulation';

	# Run simulation for the project if possible
	tcTestStarted($test);
	if (-d "$projects/$path/$verif") {
		my $PERL5LIB = $ENV{'PERL5LIB'};

		$ENV{'NF2_DESIGN_DIR'} = "$nf2_root/$projects/$path";
		$ENV{'PERL5LIB'} .= ":$nf2_root/$projects/$path/lib/Perl5";

		my $ret = tcRunProg("$projects/$path/$verif", 
				    "$nf2_root/$simCmd --ci teamcity --citest '$test'");

		$ENV{'PERL5LIB'} = $PERL5LIB;

		$ok &= ($ret == 0);
	}
	else {
		print "No simulations defined for project '$proj': $projects/$path/$verif\n";
		tcTestIgnored($test, "No simulations defined for project '$proj': $projects/$path/$verif");
	}
	tcTestFinished($test);

	return $ok;
}


#
# Run synthesis for a single project
#
sub runSynth {
	my $proj = shift;
	my $path = shift;
	my $test = shift;

	my $ok = 1;

	# Adjust the test name
	$test .= tcGetTestSeparator() . 'synth';

	# Run synthesis for the project if possible
	tcTestStarted($test);
	if (-d "$projects/$path/$synth") {
		my $PERL5LIB = $ENV{'PERL5LIB'};

		$ENV{'NF2_DESIGN_DIR'} = "$nf2_root/$projects/$path";
		$ENV{'PERL5LIB'} .= ":$nf2_root/$projects/$path/lib/Perl5";
		$ENV{'USE_SMARTGUIDE'} = "0";

		my $ret = tcRunMake('', $test . tcGetTestSeparator() . 'make', "$projects/$path/$synth");
		$ENV{'PERL5LIB'} = $PERL5LIB;


		$ok &= ($ret == 0);
	}
	else {
		print "No synthesis defined for project '$proj': $projects/$path/$synth\n";
		tcTestIgnored($test, "No synthesis defined for project '$proj': $projects/$path/$synth");
	}
	tcTestFinished($test);

	return $ok;
}

#
# Run software build for a single project
#
sub runSWBuild {
	my $proj = shift;
	my $path = shift;
	my $test = shift;

	my $ok = 1;

	# Adjust the test name
	$test .= tcGetTestSeparator() . 'sw';

	# Run synthesis for the project if possible
	tcTestStarted($test);
	if (-f "$projects/$path/$sw/Makefile") {
		my $PERL5LIB = $ENV{'PERL5LIB'};

		$ENV{'NF2_DESIGN_DIR'} = "$nf2_root/$projects/$path";
		$ENV{'PERL5LIB'} .= ":$nf2_root/$projects/$path/lib/Perl5";

		my $ret = tcRunMake('', $test . tcGetTestSeparator() . 'make', "$projects/$path/$sw");

		$ENV{'PERL5LIB'} = $PERL5LIB;

		$ok &= ($ret == 0);
	}
	else {
		print "No software (or Makefile) defined for project '$proj': $projects/$path/$sw/Makefile\n";
		tcTestIgnored($test, "No software (or Makefile) defined for project '$proj': $projects/$path/$sw/Makefile");
	}
	tcTestFinished($test);

	return $ok;
}

#
# Run regression tests for a single project
#
sub runRegress {
	my $proj = shift;
	my $path = shift;
	my $test = shift;

	my $ok = 1;

	# Adjust the test name
	$test .= tcGetTestSeparator() . 'regress';

	# Run regression tests for the project if possible
	tcTestStarted($test);
	if (-d "$projects/$path/$regress") {
		my $PERL5LIB = $ENV{'PERL5LIB'};

		$ENV{'NF2_DESIGN_DIR'} = "$nf2_root/$projects/$path";
		$ENV{'PERL5LIB'} .= ":$nf2_root/$projects/$path/lib/Perl5";

		my $ret = tcRunProg("$projects/$path/$regress",
				    "sudo $nf2_root/$regressCmd --project '$path' --ci teamcity --citest '$test'");

		$ENV{'PERL5LIB'} = $PERL5LIB;

		$ok &= ($ret == 0);
	}
	else {
		print "No regression tests defined for project '$proj': $projects/$path/$regress\n";
		tcTestIgnored($test, "No regression tests defined for project '$proj': $projects/$path/$regress");
	}
	tcTestFinished($test);

	return $ok;
}
