#include <avr/io.h>
#include <avr/interrupt.h>
#include <inttypes.h>

#include <usart.h>
#include <delay_3686.h>

#define incf() PORTB &= ~_BV(PB1); PORTB |= _BV(PB2); PORTB |= _BV(PB0);
#define stepincf() incf(); waitms(5); offf(); waitms(50);
#define decf() PORTB |= _BV(PB1); PORTB &= ~_BV(PB2); PORTB |= _BV(PB0);
#define stepdecf() decf(); waitms(5); offf(); waitms(50);
#define offf() PORTB &= ~_BV(PB0); PORTB &= ~_BV(PB1); PORTB &= ~_BV(PB2);

#define onz() if(dirz) { PORTD &= ~_BV(PD3); PORTD |= _BV(PD4); } else { PORTD |= _BV(PD3); PORTD &= ~_BV(PD4); } PORTD |= _BV(PD2);
#define incz() dirz = 1; onz();
#define stepincz() incz(); waitms(5); offz(); waitms(100);
#define decz() dirz = 0; onz();
#define stepdecz() decz(); waitms(5); offz(); waitms(100);
#define offz() PORTD &= ~_BV(PD2); PORTD &= ~_BV(PD3); PORTD &= ~_BV(PD4);

#define getz() ADCSRA |= (1<<ADIF); ADMUX = 0x42; ADCSRA = 0xD5; while(!(ADCSRA & (1<<ADIF))) ; atodz = ADCL; atodzh = ADCH; atodzh = atodzh << 8; atodz = atodz | atodzh;
#define getf() ADCSRA |= (1<<ADIF); ADMUX = 0x41; ADCSRA = 0xD5; while(!(ADCSRA & (1<<ADIF))) ; atodf = ADCL; atodfh = ADCH; atodfh = atodfh << 8; atodf = atodf | atodfh;

#define onred() PORTD |= _BV(PD6);
#define offred() PORTD &= ~_BV(PD6);
#define ongreen() PORTD |= _BV(PD7);
#define offgreen() PORTD &= ~_BV(PD7);

#define timer2on() TCNT2 = 0; TCCR2 |= (1<<CS22 | 1<<CS20); timeron = 1;	// use CLKio / 64
#define timer2off() TCCR2 = 0; timeron = 0;

volatile int16_t atodz, atodzh, atodf, atodfh;
int16_t cal;
uint8_t in;
int16_t targetz, targetz2, targetf, targetf2;
uint8_t movez, movef, timeron, dirz, ontime;

ISR (TIMER2_OVF_vect) {					// Timer 1 overflow
	onz();
}

ISR (TIMER2_COMP_vect) {
	offz();
}

// catch all interrupt handler
ISR (__vector_default) {
}

inline void set_up_interrupts(void) {

	TCCR2 = 0;
	OCR2 = ontime;
	TIMSK |= (1<<TOIE2);		// overflow interrupt
	TIMSK |= (1<<OCIE2);		// comparator interrupt
	
	SREG |= 0x80 ;			// Turn on interrupt	
	
}

void calibrate() {
	getz();
	cal = 0;
	decz();
	while(1) {
		wait4ms(255);
		if(cal == atodz)
			break;
		cal = atodz;
		getz();
	}
	offz();
	USART_Transmitint16(atodz);
	USART_Transmit('-');
	cal = 32767;
	incz();
	while(1) {
		wait4ms(255);
		if(cal == atodz)
			break;
		cal = atodz;
		getz();
	}
	offz();
	USART_Transmitint16(atodz);
	USART_Transmit('/');
	getf();
	cal = 0;
	decf();
	while(1) {
		wait4ms(255);
		if(cal == atodf)
			break;
		cal = atodf;
		getf();
	}
	offf();
	USART_Transmitint16(atodf);
	USART_Transmit('-');
	cal = 32767;
	incf();
	while(1) {
		wait4ms(255);
		if(cal == atodf)
			break;
		cal = atodf;
		getf();
	}
	offf();
	USART_Transmitint16(atodf);
}

void main(void)
{
	/* INITIALIZE */

	/* OUTPUTS TO LED */
	DDRD|= _BV(PD6);
	DDRD|= _BV(PD7);
	
	onred();
	wait4ms(255);
	ongreen();
	wait4ms(255);
	
	/* OUTPUTS TO L293 */
	DDRB |= _BV(PB0);
	DDRB |= _BV(PB1);
	DDRB |= _BV(PB2);
	DDRD |= _BV(PD2);
	DDRD |= _BV(PD3);
	DDRD |= _BV(PD4);
	
	set_up_interrupts();

	/* TURN ALL MOTORS OFF */
	PORTB &= ~_BV(PB0);
	PORTB &= ~_BV(PB1);
	PORTB &= ~_BV(PB2);
	PORTD &= ~_BV(PD2);
	PORTD &= ~_BV(PD3);
	PORTD &= ~_BV(PD4);

	/* OUTPUT TO TX */
	DDRD |= _BV(PD1);
	/* TX OFF */
	PORTD &= ~_BV(PD1);
	
	USART_Init(23);
	/* TEST */
	UCSRA |= (1<<RXC);
	/* TEST2 */
	USART_TransmitString("\r\f");
	
	offred();
	offgreen();

	targetz = 500;
	targetf = 500;
	movez = 0;
	movef = 0;
	
	timeron = 0;
	dirz = 0;
	ontime = 255;

	while(1) {

		getz();
		getf();

		/* new instruction? */
		if(UCSRA & (1<<RXC)) {
			in = UDR;
			USART_Transmit(in);
			/*
			C    : calibrate (get range)
			T    : test (return 0)
			
			D    : get focus target
			E    : get focus position
			Fxxx : set focus target
			
			*    : increment focus
			/    : decrement focus
			
			X    : get zoom target
			Y    : get zoom position
			Zxxx : set zoom target
			
			+    : increment zoom
			-    : decrement zoom
			*/
			if(in == 'Z') {
				in = USART_Receive();
                                USART_Transmit(in);
                                targetz = in - '0';
                                in = USART_Receive();
                                USART_Transmit(in);
                                targetz = targetz * 10 + in - '0';
                                in = USART_Receive();
                                USART_Transmit(in);
                                targetz = targetz * 10 + in - '0';
				if(atodz != targetz) {
					if(atodz < targetz) {
						targetz2 = targetz - 20;
					} else {
						targetz2 = targetz + 20;
					}
					movez = 1;
				}
			} else if(in == 'Y') {
				USART_Transmitint16(atodz);
			} else if(in == 'X') {
				USART_Transmitint16(targetz);
			} else if(in == 'F') {
				in = USART_Receive();
				USART_Transmit(in);
				targetf = in - '0';
				in = USART_Receive();
                                USART_Transmit(in);
				targetf = targetf * 10 + in - '0';
				in = USART_Receive();
                                USART_Transmit(in);
				targetf = targetf * 10 + in - '0';
				if(atodf != targetf) {
					if(atodf < targetf) {
						targetf2 = targetf - 20;
					} else {
						targetf2 = targetf + 20;
					}
					movef = 1;
				}
			} else if(in == 'E') {
				USART_Transmitint16(atodf);
			} else if(in == 'D') {
				USART_Transmitint16(targetf);
			} else if(in == 'T') {
				USART_TransmitUint8(0);
			} else if(in == 'C') {
				calibrate();
			} else if(in == '+') {
				stepincz();
			}  else if(in == '-') {
				stepdecz();
			} else if(in == '*') {
				stepincf();
			} else if(in == '/') {
				stepdecf();
			} else if(in == 'A') {
				timer2on();
			} else if(in == 'S') {
				timer2off();
			} else if(in == 'U') {
				USART_TransmitUint8(ontime);
			} else if(in == 'V') {
				in = USART_Receive();
                                USART_Transmit(in);
                                ontime = in - '0';
                                in = USART_Receive();
                                USART_Transmit(in);
                                ontime = ontime * 10 + in - '0';
                                in = USART_Receive();
                                USART_Transmit(in);
                                ontime = ontime * 10 + in - '0';
				OCR2 = ontime;
			}
		}

		if(movef) {
			/* RED LED ON */
			onred();
			if(atodf < targetf) {
				if(atodf < targetf2) {
					incf();
				} else {
					stepincf();
				}
			} else if(atodf > targetf) {
				if(atodf > targetf2) {
					decf();
				} else {
					stepdecf();
				}
			} else {
				offf();
				movef = 0;
			}
		} else {
			offred();
		}

		if(movez) {
			/* GREEN LED ON */
			ongreen();
			if(atodz < targetz) {
				if(atodz < targetz2) {
					dirz = 1;
					if(ontime < 255) {
						if(!timeron)
							timer2on();
					} else {
						timer2off()
						onz();
					}
				} else {
					timer2off();
					stepincz();
				}
			} else if(atodz > targetz) {
				if(atodz > targetz2) {
					dirz = 0;
					if(ontime < 255) {
						if(!timeron)
							timer2on();
					} else {
						timer2off();
						onz();
					}
				} else {
					timer2off();
					stepdecz();
				}
			} else {
				offz();
				timer2off();
				movez = 0;
			}
		} else {
			/* GREEN LED OFF */
			offgreen();
		}

	}
}
