/**********************************************************
 *  This work by Richard Leivers (rjl67@cam.ac.uk)        * 
 *  is licenced under the Creative Commons                *
 *  Attribution-Non-Commercial-Share Alike 2.0 UK:        *
 *  England & Wales License.                              *
 *                                                        *
 *  To view a copy of this licence, visit                 *
 *  http://creativecommons.org/licenses/by-nc-sa/2.0/uk/  *
 *  or send a letter to Creative Commons,                 *
 *  171 Second Street, Suite 300, San Francisco,          *
 *  California 94105, USA.                                *
 *                                                        *
 **********************************************************	
 *	Manchester Baby simulator in Java  	*
 * 	Richard Leivers - July 2009        	*
 * 										*
 *	MAIN PROCESSOR MODULE				*
 ****************************************/


import javax.swing.JPanel;

public class BabyProc {
	
	// Enumeration of the instruction types (sub is repeated) 
	private enum fType {JMP, JRP, LDN, STO, SUB, SUB5, TEST, STOP}
	
	// Store references to memory and JPanel (to refresh after every cycle)
	private Memory MEM;
	private JPanel panel;
	
	// 32-bit PROCESSOR REGISTERS
	int ACC = 0; // Accumulator
	int CI = 0; // Line 1 of Control Register, holds program count
	int PI = 0; // Line 2 of Control Register, holds currently executing instruction
	
	// Go is true until STOP hit
	boolean go = true;
	
	// Constructor just initialises memory and JPanel fields
	public BabyProc (Memory mem, JPanel panel) {
		MEM = mem;
		this.panel = panel;
	}
	
	// Run method used to start processor on mem
	public int run() throws Exception {
		
		// Keep a count of cycles executed
		int count = 0;
		
		while (go) {
			
			// Add a pause to make simulation more realistic and increase count
			Thread.sleep (2);
			count++;
			
			/// MAIN PROCESSOR STAGES ///
			
			// 1 :: Increment CI
			CI++;
			
			// 2 :: Load instruction into PI and decode
			PI = MEM.getMemLine(CI & 31); // get instruction from memory - only first 5 bits are used
			int LINE = PI & 31; // first 5 bits are LINE NUMBER
			fType FUNCTION = fType.values()[((PI >>> 13) & 7)]; // function should be from enum as above

			
			// 3 :: Execute instruction
			
			// Extra fake registers to store updated values and address
			int newCI = 0, newACC = 0, newMEMLine = 0, newMEMLineAddr = 0;
			// Flags to indicate which registers have been updated
			boolean uCI = false, uACC = false, uMEM = false;
			
			// Switch based on function type
			switch (FUNCTION) {
			// JMP sets CR to the contents of LINE
			case JMP : 	newCI = MEM.getMemLine(LINE);
						uCI = true;
						break;
			// JRP adds the contents of LINE to CR
			case JRP :  newCI = (CI + MEM.getMemLine(LINE));
						uCI = true;
						break;
			// LDN gets the contents of LINE, negated, into ACC
			case LDN :	newACC = - MEM.getMemLine(LINE);
						uACC = true;
						break;
			// STO stores the contents of ACC into LINE
			case STO :  newMEMLineAddr = LINE;
						newMEMLine = ACC;
						uMEM = true;
						break;
			// SUB subtracts the contents of LINE from ACC putting the result in ACC
			case SUB : // both 001 and 101 are SUB
			case SUB5 : newACC = ACC - MEM.getMemLine(LINE);
						uACC = true;
						break;
			// TEST if the contents of ACC are less than zero, add 1 to CR
			case TEST:	if (ACC < 0) {
							newCI = (CI+1);
							uCI = true;
						}
						break;
			// STOP halts the machine
			case STOP:  go = false;
						break;
			}
		
			
			// 4 :: Write-back - update mem & registers
			// If update flag set, then update real registers / memory location from temps
			if (uCI) CI = newCI;
			if (uACC) ACC = newACC;
			if (uMEM) MEM.writeLine(newMEMLineAddr, newMEMLine);
			
			
			// Redraw memory display and output execution message
			panel.repaint();
			System.out.println("Instruction executed: " + FUNCTION.toString() + " " + LINE + " ACC: " + ACC + " CI: " + CI);
		}
		
		return count;
	}
}
