/*module char(
	input	[4:0]	in,
	output	[7:0]	out
);

	assign out = in < 10 ? 48 + in : 87 + in;

endmodule




module hex(
	input	[31:0]	in,
	output	[63:0]	out
);

	char(in[31:28], out[63:56]);
	char(in[27:24], out[55:48]);
	char(in[23:20], out[47:40]);
	char(in[19:16], out[39:32]);
	char(in[15:12], out[31:24]);
	char(in[11: 8], out[23:16]);
	char(in[ 7: 4], out[15: 8]);
	char(in[ 3: 0], out[ 7: 0]);

endmodule*/




module subsystemMMU(
	input			clk,
	input			reset_n,
	output	[31:0]	address,
	output	[1:0]	byteenable,
	input			irq,
	input	[5:0]	irqnumber,
	output			write_n,
	output			read_n,
	input			waitrequest,
	output	[15:0]	writedata,
	input	[15:0]	readdata,

	output [22:0]	ava_address_from_the_scarab,
	output [1:0]	ava_byteenable_from_the_scarab,
	output			ava_read_from_the_scarab,
	input [15:0]	ava_readdata_to_the_scarab,
	input			ava_waitrequest_to_the_scarab,
	output			ava_write_from_the_scarab,
	output [15:0]	ava_writedata_from_the_scarab,
	
	output			vgaCharacterSet,
	output reg	[39:0]	vgaWriteData,
	output reg			vgaWriteReq,
	input			vgaWriteFull,
	output reg	[11:0]	tbWriteAddress,
	output reg	[14:0]	tbWriteData,
	output reg			tbWriteReq,
	
	input			debug,
	input			break,
	
	output			LCD_RW,
	output			LCD_EN,
	output			LCD_RS,
	output	[7:0]	LCD_DATA,
	output			LCD_ON,
	output			LCD_BLON,
	
	output	[6:0]	HEX0,HEX1,HEX2,HEX3,HEX4,HEX5,HEX6,HEX7,
	output	[17:0]	LEDR
	);


//------------------------------------------------------------------------------
//				DEBUGGING OUTPUT
//------------------------------------------------------------------------------
//	wire [63:0] topleft, topright, bottomleft, bottomright;
	
//	hex( mipsPC, topleft );
//	hex( mipsAddress, topright );
//	hex( mipsWriteData, bottomleft );
//	hex( mipsReadData, bottomright );
//	LCD_CONTROLLER(clk, {topleft, topright}, {bottomleft, bottomright}, LCD_RW, LCD_EN, LCD_RS, LCD_DATA, LCD_ON, LCD_BLON);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//				THE MIPS PROCESSOR
//------------------------------------------------------------------------------
	wire mipsWrite_, mipsRead_, mipsMem16_, mipsMem8_, mipsZeroFill_;
	wire [31:0] mipsPC, mipsAddress_, mipsWriteData_;

	malwa_malwa malwa(
		.clk(clk),
		.reset(!reset_n),
		
		.i_stall(tpmViolation || cacheIStall),
		.d_stall(d_stall),

		.irq(irq),
		
		.pc(mipsPC),
		.instrF(cacheInstr),

		.memwrite(mipsWrite_),
		.memread(mipsRead_),
		.mem16(mipsMem16_),
		.mem8(mipsMem8_),
		.memzerofill(mipsZeroFill_),
		.memaddress(mipsAddress_),
		.memwritedata(mipsWriteData_),
		.memreaddata(d_readdata)
	);

//------------------------------------------------------------------------------
//				THE SCARAB CACHE
//------------------------------------------------------------------------------
	wire cacheIStall;
	wire cacheDStall;
	wire [31:0] cacheInstr, cacheReadData;
	wire [22:0] cacheAddress;
	
	wire problem_a,problem_b;
	
	assign LEDR[15:14]={problem_a,problem_b};
	
	cache cache(
		.clk(clk),
		.reset(!reset_n),
		
		.i_pc_(mipsPC),
		.i_instr(cacheInstr),

		.i_stall(cacheIStall),
		.d_stall(cacheDStall),

		.d_doread_(mipsRead_ && cacheHit_ && !tpmViolation),
		.d_dowrite_(mipsWrite_ && cacheHit_ && !tpmViolation),
		.d_address_(mipsAddress_[22:0]),
		.d_mem16_(mipsMem16_),
		.d_mem8_(mipsMem8_),
		.d_writedata_(mipsWriteData_),
		.d_readdata(cacheReadData),

		.ava_read(ava_read_from_the_scarab),
		.ava_write(ava_write_from_the_scarab),
		.ava_byteenable(ava_byteenable_from_the_scarab),
		.ava_address(cacheAddress),
		.ava_readdata(ava_readdata_to_the_scarab),
		.ava_writedata(ava_writedata_from_the_scarab),
		.ava_waitrequest(ava_waitrequest_to_the_scarab),
		
		.problem_a(problem_a),
		.problem_b(problem_b)
	);
	
	assign ava_address_from_the_scarab={cacheAddress[22], cacheAddress[20:9], cacheAddress[21], cacheAddress[8:0]};


//------------------------------------------------------------------------------
//				THE TRUSTED PLATFORM MODULE ;)
//------------------------------------------------------------------------------
	wire [31:0] tpmInstr;
	cache_data thetpm(
			.address_a(mipsPC[11:2]),
			.clock_a(clk),
			.clock_b(clk),
			.wren_a(0),
			.wren_b(0),
			.q_a(tpmInstr)
	);
	reg tpmViolation;
	reg [31:0] tpmPC;
	always @(posedge clk)
	begin
		if( !reset_n )
		begin
			tpmViolation <= 0;
			tpmPC <= 0;
		end
		else
		begin
			if( tpmPC < 4096 && !cacheIStall && cacheInstr != tpmInstr && (tpmPC[11:2] != 262 || cacheInstr != 32'h0C00_0400) )
				tpmViolation <= 1;
			
			tpmPC <= mipsPC;
		end
	end
	
	assign LEDR[0] = tpmViolation;
	assign LEDR[16] = problem_a || problem_b || tpmViolation;

//------------------------------------------------------------------------------
//				THE AVALON BUS
//------------------------------------------------------------------------------
	wire cacheHit_		= (mipsAddress_ >= 32'h0000_0000 && mipsAddress_ <= 32'h007F_FFFF);

	wire d_stall=cacheHit ? cacheDStall : avalonStall;

	reg [31:0] d_readdata;
	
	/////////////////////
	//Register inputs to bus
	reg mipsWrite, mipsRead, mipsMem32, mipsMem16, mipsMem8, mipsZeroFill;
	reg [31:0] mipsAddress, mipsWriteData;
	reg cacheHit;
	
	always @(posedge clk)
	if(!reset_n)
	begin
		mipsWrite <= 0;
		mipsRead <= 0;
		mipsMem32<= 0;
		mipsMem16 <= 0;
		mipsMem8 <= 0;
		mipsZeroFill <= 0;
		mipsAddress <= 0;
		mipsWriteData <= 0;
		cacheHit <= 1'b1;
	end
	else if(!d_stall)
	begin
		mipsWrite <= mipsWrite_;
		mipsRead <= mipsRead_;
		mipsMem32<=  	cacheHit_
						&& !mipsMem8_
						&& !mipsMem16_
						;
		mipsMem16 <= mipsMem16_;
		mipsMem8 <= mipsMem8_;
		mipsZeroFill <= mipsZeroFill_;
		mipsAddress <= mipsAddress_;
		mipsWriteData <= mipsWriteData_;
		cacheHit <= cacheHit_;
	end
	/////////////////////

	wire avalonStall;
	
	wire avalonHit			= !mipsAddress[31] && !cacheHit;
	
	wire avalonTwoStages	= mipsMem32;
	
	reg avalonBegunSecondStage;
	reg [15:0] avalonSavedFirstStage;
	always @(posedge clk)
	begin
		if( avalonHit && (mipsRead || mipsWrite) && avalonTwoStages && !waitrequest )
		begin
			avalonBegunSecondStage <= !avalonBegunSecondStage;
			
			if( !avalonBegunSecondStage )
				avalonSavedFirstStage <= readdata;
		end
	end
	
	//CHECKME: ENDIANNESS :)
	assign address		=	{
								(mipsAddress >= 32'h0000_0000 && mipsAddress <= 32'h007F_FFFF)	?	{9'b000000000, mipsAddress[22], mipsAddress[20:9], mipsAddress[21], mipsAddress[8:2]}
																								:	mipsAddress[31:2]
							, mipsMem32 ? avalonBegunSecondStage : mipsAddress[1]
							, 1'b0
							};
	assign byteenable	= mipsMem8	? (mipsAddress[0] ? 2'b01 : 2'b10)
									: 2'b11;
	assign write_n		= !avalonHit || !mipsWrite;
	assign read_n		= !avalonHit || !mipsRead;
	assign writedata	= mipsMem32	? (avalonBegunSecondStage ? mipsWriteData[15:0] : mipsWriteData[31:16])
						: mipsMem8	? {mipsWriteData[7:0], mipsWriteData[7:0]}
									: mipsWriteData[15:0]
						;
	
	always @(posedge clk)
	if(!d_stall)
	begin
		d_readdata <= avalonHit ?
						  (
							  mipsMem32	? {avalonSavedFirstStage, readdata}
							: mipsMem8	? (mipsAddress[0]	? {{24{!mipsZeroFill && readdata[7]}}, readdata[7:0]}
															: {{24{!mipsZeroFill && readdata[15]}}, readdata[15:8]}
										  )
							: {{16{!mipsZeroFill && readdata[15]}}, readdata}
						  )
						: cacheHit ?
						  (
							  mipsMem32	? cacheReadData
							: mipsMem8	? {{24{!mipsZeroFill && cacheReadData[7]}}, cacheReadData[7:0]}
							: {{16{!mipsZeroFill && cacheReadData[15]}}, cacheReadData[15:0]}
						  )
						: (
							  mipsAddress==32'hF000_0000	? {26'b0,irqnumber}
							: 32'hCCCC_CCCC
						  )
						;
	end
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
//				THE STALL LOGIC
//------------------------------------------------------------------------------
	reg broke;
	assign avalonStall	= 	(
								mipsAddress >= 32'h9000_0000 && mipsAddress <= 32'h9007_e9d7
								&& mipsWrite
								&& vgaWriteFull
							)
							||
							(
								avalonHit
								&& (mipsRead || mipsWrite)
								&& (waitrequest || (avalonTwoStages && !avalonBegunSecondStage))
							)
							||
							(
								debug && (!broke || break)
							)
						;
	always @(posedge clk)
	begin
		broke <= break;
	end
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
//				THE VGA SUBSYSTEM
//------------------------------------------------------------------------------
	wire [17:0] vgaAddress;
	assign vgaAddress = (mipsAddress - 32'h9000_0000) >> 1;
	
	always @(posedge clk)
	begin
		vgaWriteReq <= 0;
		tbWriteReq <= 0;
		
		if( mipsAddress >= 32'h9000_0000 && mipsAddress <= 32'h9007_e9d7 && mipsWrite && !vgaWriteFull )
		begin
			vgaWriteReq <= 1;
			vgaWriteData <= {vgaAddress, 6'b000000, mipsWriteData[15:0]};
		end
		
		if( mipsAddress >= 32'ha000_0000 && mipsAddress <= 32'ha000_12bf && mipsWrite )
		begin
			tbWriteReq <= 1;
			tbWriteAddress <= (mipsAddress - 32'ha000_0000) >> 1;
			tbWriteData <= mipsWriteData[14:0];
		end
	end
//------------------------------------------------------------------------------


endmodule
