package reducer;

/*
 *   Copyright (C) A C Norman, April 2000
 *   Permission is given to use this code in any way you want for
 *   purposes directly concerned with the University of Cambridge taught
 *   courses in Computer Science and associated study. Anybody who
 *   wants to re-distribute this code or use it commercially should
 *   check with ACN (acn1@cam.ac.uk) first.
 */

// For each combinator that I want to support I will derive a
// sub-class of Graph and define the recipe for performing a reduction
// step.

class SCombinator extends Combinator
{

SCombinator()
{
    name = "S";
}

boolean makeHeadNormal()
{
  // the following code is quite short (which is good) but it deserves some
  // explanation. When "current" is the "S" combinator one usually expects
  // that "trail" will refer to a chain that make sit possible to find the
  // arguments that S had. Suppose there were 3 of them, so we are
  // reducing S f g x [which must give f x (g x)]. Then the graph will
  // be in the following state
  //
  //                   old trail
  //                       ^
  //                       |
  //                     [ .   . ]
  //                      ^     \
  //                     /       x
  //                  [ .   . ]
  //                   ^     \
  //                  /       g
  //  trail -----> [ .   . ]
  //                      \
  //  current --> S        f
  //
  // Observe a chain of backwards pointers from trail through the
  // left components of the application cells up to the old trail.
  //
  // Notionally the first thing to do is to unwind the pointer
  // reversing by three steps to get
  //
  //  trail ------> old trail
  //
  //  current -------->  [ .   . ]*
  //                      /     \
  //                     /       x
  //                  [ .   . ]
  //                   /     \
  //                  /       g
  //               [ .   . ]
  //               /      \
  //              S        f
  //
  // where now the pointers down the leftmost spine point downwards
  // again. It is now of course easy to access f, g and x. To perform the
  // "S" operation the application node labelled "*" above is over-written
  // to hold ((f x) (g x)), ie
  //
  //  trail ------> old trail
  //
  //  current -------->  [ .   . ]*             (this bit left around)
  //                      /     \                    [ .   . ]
  //                     /       [ .   . ]            /     \
  //                    /          |    \            /       g
  //                 [ .   . ]     g     x        [ .   . ]
  //                  /     \                      /     \
  //                 f       x                    S       f
  //
  // where the two nodes for (f x) and (g x) are newly created. Finally
  // we know from this construction that "current" can now move down
  // (at least) two steps, and so it makes sense to do that at once,
  // leaving
  //
  //                   old trail
  //                       ^
  //                       |
  //                     [ .   . ]*
  //                      ^     \
  //                     /       [ .   . ]
  //                    /          |    \
  //  trail -------> [ .   . ]     g     x
  //                        \
  //  current -----> f       x
  //
  // One further complication arises. If "S" had not bene provided with
  // three arguments it can not be reduced. This is easily detected
  // because in such cases the trail of back-pointers will terminate in
  // a null before the full structure as above is found.
  // Folding these stages together leads to the following code!
    if (trail == null) return false;
    Application trail1 = (Application)trail.function;
    if (trail1 == null) return false;
    Application trail2 = (Application)trail1.function;
    if (trail2 == null) return false;
    // Now we know we have 3 arguments
    trail1.function = trail;
    trail.function = current; // repair ((S f) g)
    current = trail.argument; // new current is f
    trail = new Application(trail2, trail2.argument); // for (f x)
    trail2.argument = 
        new Application(trail1.argument, trail2.argument); // (g x)
    // All done!!
    return true;
}

}



