/* (C) Cambridge University Computer Laboratory, 2002
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <pthread.h>
#include <string.h>
#include <unistd.h>

#include "stm.h"
#include "stm_test.h"

/*
 * Configuration settings. 
 *
 * Each one of THREADS will perform ITERATIONS get/put pairs taking
 * a value from an input buffer and storing value+1 into an output
 * buffer.  VALUES (<=THREADS) values are introduced SPACING apart
 * at the start and we expect that many suitably-incremented values
 * to remain at the end.
 */

#ifndef EXTERNAL_STM_CONFIG
#define THREADS  4
#define VALUES   (THREADS / 2)
#endif

#define ITERATIONS 10000
#define SPACING (((float) THREADS)/VALUES)

/***********************************************************************/

#define TIMEVAL_TO_MILLIS(tv) \
  ((((long long) (tv.tv_usec)) + (((long long) (tv.tv_sec)) * 1000000)) / 1000)

#define panic(_x)				\
if (1) {					\
  printf _x;					\
  printf ("\n");                                \
  exit(1);					\
}

typedef enum { FALSE = 0,
	       TRUE = 1 } bool_t;

/*
 * Each pair of full/value entries provides a single-cell buffer.
 * 
 *    full[i] == TRUE   => value[i] contains a value
 *
 * and 
 *
 *    full[i] == FALSE  => value[i] is empty
 */

static volatile word_t full[THREADS];
static volatile word_t value[THREADS];

/*
 * Put an item in the cell managed by full/value, blocking until this
 * can be done.
 */

static void put_item (STM_ENV *stm, 
		      volatile word_t *full_ptr, 
		      volatile word_t *value_ptr,
		      int item)
{
  bool_t done;

  done = FALSE;
  while (!done)
    {
      bool_t was_full;
      STMStartTransaction (stm);
      was_full = STMReadValue (stm, (addr_t) full_ptr);
      if (was_full) {
	STMWait (stm);
      } else {
	STMWriteValue (stm, (addr_t) value_ptr, item);
	STMWriteValue (stm, (addr_t) full_ptr, TRUE);
	if (STMCommitTransaction (stm)) {
	  done = TRUE;
	} 
      }
    }
}

/*
 * Extract an item from the cell managed by full/value, blocking until
 * this can be done.  
 */

static int get_item (STM_ENV *stm,
		     volatile word_t *full_ptr,
		     volatile word_t *value_ptr)
{
  bool_t done;
  int result;

  done = FALSE;
  while (!done)
    {
      bool_t was_full;
      STMStartTransaction (stm);
      was_full = STMReadValue (stm, (addr_t) full_ptr);
      if (!was_full) {
	STMWait (stm);
      } else {
	result = STMReadValue (stm, (addr_t) value_ptr);
	STMWriteValue (stm, (addr_t) full_ptr, FALSE);
	if (STMCommitTransaction (stm)) {
	  done = TRUE;
	}
      }
    }

  return result;
}

extern void snapshot(void *s);

static void *thread (void *dummy)
{
  int           from_cell;
  int           to_cell;
  STM_ENV       st;
  volatile word_t *full_in;
  volatile word_t *value_in;
  volatile word_t *full_out;
  volatile word_t *value_out;
  int           i;

  from_cell = (int) dummy;
  to_cell = (from_cell + 1) % THREADS;

  full_in = &full[from_cell];
  value_in = &value[from_cell];
  full_out = &full[to_cell];
  value_out = &value[to_cell];

  STMAddThread (&st);

  for (i = 0; i < ITERATIONS; i++)
    {
      int v;
      v = get_item (&st, full_in, value_in);
      put_item (&st, full_out, value_out, v + 1);
    }

  STMRemoveThread (&st);

  return NULL;
}

/***********************************************************************/

int main (int argc, char **argv)
{
  pthread_t thrs[THREADS];
  struct timeval ts;
  struct timeval te;
  int i;
  int total_increment;
  int num_final_values;

  pthread_setconcurrency(THREADS);
  STMInit ();

  /* Initialise cells: all empty except for the first */

  for (i = 0; i < THREADS; i ++)
    {
      full[i] = FALSE;
      value[i] = (int) 0xdeadbeef;
    }

  for (i = 0; i < VALUES; i ++)
    {
      full[(int)(i * SPACING)] = TRUE;
      value[(int)(i * SPACING)] = 0;
    }

  gettimeofday (&ts, NULL);

  /* Start threads */

  for (i = 0; i < THREADS; i ++)
    {
      pthread_create (&thrs[i], NULL, thread, (void *) i);
    }

  /* Wait for threads to finish */

  for (i = 0; i < THREADS; i ++)
    {
      pthread_join (thrs[i], NULL);
    }

  gettimeofday (&te, NULL);

  /* Summarise results */

  for (i = 0; i < THREADS; i ++)
    {
      if (full[i]) 
	{
	  fprintf(stderr, "%d ", value[i]);
	}
      else
	{
	  fprintf(stderr, ". ");
	}
    }
  fprintf (stderr, "\n");

  fprintf (stderr, "threads = %d ", THREADS);
  fprintf (stderr, " elapsed = %lld ", 
	   TIMEVAL_TO_MILLIS (te) - TIMEVAL_TO_MILLIS (ts));
  fprintf (stderr, "\n");

  /* Check status */

  num_final_values = 0;
  total_increment = 0;
  for (i = 0; i < THREADS; i ++)
    {
      if (full[i]) 
	{
	  num_final_values ++;
	  total_increment += value[i];
	}
    }

  if (num_final_values != VALUES || total_increment != ITERATIONS * THREADS)
    {
      panic (("Unexpected end condition %d %d\n",
	      num_final_values, total_increment));
    }

  exit(0);
}
