/* (C) Cambridge University Computer Laboratory, 2002
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <sched.h>
#include <alloca.h>
#include <pthread.h>
#include <sys/types.h>
#include "env_common.h"

/*
 * Data types
 */

typedef void * volatile   *addr_t;
typedef unsigned long long word_t;
typedef unsigned long long uintptr_t;

typedef pthread_mutex_t STMMutex_t;
typedef pthread_cond_t STMCond_t;

#define OPT_NEED_BOOL_T 1
#define STM_DONE_TYPE_DEFINITIONS

/*
 * Mutex & CV operations
 */

#define STMMutexInit(m) pthread_mutex_init(m, NULL)
#define STMMutexLock(m) pthread_mutex_lock(m)
#define STMMutexUnlock(m) pthread_mutex_unlock(m)
#define STMMutexDestroy(m) pthread_mutex_destroy(m)

#define STMCondInit(c) pthread_cond_init(c, NULL)
#define STMCondWait(c,m) pthread_cond_wait(c,m)
#define STMCondSignal(c) pthread_cond_signal(c)
#define STMCondDestroy(c) pthread_cond_destroy(c)

/*
 * CAS operations
 */

#define CAS32(_a, _o, _n)					\
({ __typeof__(_o) __o = _o;					\
   __asm__ __volatile__("mov ar.ccv=%0 ;;" :: "rO" (_o));	\
   __asm__ __volatile__("cmpxchg4.acq %0=[%1],%2,ar.ccv ;; "	\
       : "=r" (__o)						\
       :  "r"(_a), "r"(_n) : "memory");				\
   __o;								\
})

#define CAS64(_a, _o, _n)					\
({ __typeof__(_o) __o = _o;					\
   __asm__ __volatile__("mov ar.ccv=%0 ;;" :: "rO" (_o));	\
   __asm__ __volatile__("cmpxchg8.acq %0=[%1],%2,ar.ccv ;; "	\
       : "=r" (__o)						\
       :  "r"(_a), "r"(_n) : "memory");				\
   __o;								\
})

static int CASInt (volatile int *a, int o, int n)
{
  return CAS32 (a, o, n);
}

static void * CASPtr (void * volatile *a, void *o, void *n)
{
  return CAS64 (a, o, n);
}

#define OPT_CASPTRINT_AVAILABLE 0

/*
 * Explicitly-managed heap
 */

#define OPT_MALLOC_HEAP_AVAILABLE 1
#define REQ_NATIVE_ALIGNMENT 8
#define mallocBytes(_x) malloc(_x)
#define freeMallocObj(_x) free(_x)

/*
 * Garbage-collected heap
 */

#define OPT_GC_HEAP_AVAILABLE 0

/*
 * Memory consistency barriers.
 */

#define MB()  __asm__ __volatile__ (";; mf ;; " : : : "memory")
#define WMB() MB()
#define RMB() MB()

/*
 * Processor cycle counter access.
 */

typedef unsigned long long tick_t;

#define RDTICK() \
    ({ tick_t __t; __asm__ __volatile__ ("mov %0=ar.itc ;;" : "=rO" (__t)); __t; })

/*
 * Environemnt structure definition.
 */

typedef struct { void *st; } ExtSTMState_t;

#define STM_ENV ExtSTMState_t
#define GET_ENV_STATE(_e) ((_e) -> st)
#define SET_ENV_STATE(_e,_s) if (1) { (_e) -> st =  (_s); }
#define STM_DONE_ENV_DEFINITION

/*
 * Assertions
 */

#define stm_panic(_x)				\
if (1) {					\
  printf _x;					\
  printf ("\n");                                \
  exit(1);					\
}

#ifdef DEBUG
#define stm_assert(_x) if (!(_x)) {			\
  fprintf (stderr, "Assertion failed %s %d (%s)\n",	\
	   __FILE__,					\
	   __LINE__,					\
	   # _x);					\
  exit(1);						\
}
#else
#define stm_assert(_x) /* Nothing */
#endif

/*
 * Inlining
 */

#define NO_INLINE() /* To do */

