;;; jde-run.el --- runs the Java app in the current buffer.

;; Author: Paul Kinnucan <pkinnucan@iname.com>
;; Maintainer: Paul Kinnucan
;; Version 1.2
;; Keywords: tools, processes

;; Copyright (C) 1997 Free Software Foundation, Inc.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.


;;;###autoload
(defvar jde-run-mode-hook nil
  "*List of hook functions run by `jde-run-mode' (see `run-hooks').")

;;;###autoload
(defvar jde-run-window-height nil
  "*Number of lines in a jde-run window.  If nil, use Emacs default.")

;;;###autoload
(defvar jde-run-buffer-name-function nil
  "Function to compute the name of a jde-run buffer.
The function receives one argument, the name of the major mode of the
jde-run buffer.  It should return a string.
nil means compute the name with `(concat \"*\" (downcase major-mode) \"*\")'.")

;;;###autoload
(defvar jde-run-finish-function nil
  "*Function to call when a java process finishes.
It is called with two arguments: the jde-run buffer, and a string
describing how the process finished.")

(defvar jde-run-last-buffer nil
  "The most recent jde-run buffer.
A buffer becomes most recent when its java app is started.")

(defvar jde-run-in-progress nil
  "List of jde-run processes now running.")
(or (assq 'jde-run-in-progress minor-mode-alist)
    (setq minor-mode-alist (cons '(jde-run-in-progress " Running Java App")
				 minor-mode-alist)))

(defvar jde-run-exit-message-function nil "\
If non-nil, called when a java process dies to return a status message.
This should be a function of three arguments: process status, exit status,
and exit message; it returns a cons (MESSAGE . MODELINE) of the strings to
write into the Java app output buffer, and to put in its mode line.")

(defun bashify-command (cmd)
  "If bash is the current shell, returns CMD enclosed in quotes.
This satisfies bash requirement that the -c command line option
be enclosed in qotes."
  (if (and (eq system-type 'windows-nt)
	   (or (string-match "bash" shell-file-name)
	       (string-match "bash" (getenv "SHELL"))))	  
      (concat "\"" cmd "\"")
    cmd))

;;;###autoload
(defun jde-run ()
  "Run the Java app corresponding to the Java source file in the current buffer,
with output going to the buffer `*jde-run*'.

To run more than one Java app at once, start one and rename the
\`*jde-run*' buffer to some other name with \\[rename-buffer].
Then start the next one.

The name used for the buffer is actually whatever is returned by
the function in `jde-run-buffer-name-function', so you can set that
to a function that generates a unique name."
  (interactive)
  (jde-run-internal (file-name-sans-extension (buffer-name))))

(defun jde-run-internal (app-name &optional name-of-mode buffer-name-function)
  "Run Java app (low level interface).
APP-NAME is the name of the class file (minus the .class extension) of
the app to be run. NAME-OF-MODE is the name to display as the major mode in the
output buffer. BUFFER-NAME-FUNCTION is a function that supplies for the
output buffer.

Returns the app output buffer created."
  (interactive)
  (let ((command (concat "java " app-name))
	outbuf)
    (if (string= shell-file-name "bash")
	(setq command (concat "\"" command "\"")))
;dont    (save-excursion
    (or name-of-mode
	(setq name-of-mode "Run Java"))
    (setq outbuf
	  (get-buffer-create
	   (funcall (or buffer-name-function jde-run-buffer-name-function
			(function (lambda (mode)
				    (concat "*" (downcase mode) "*"))))
		    name-of-mode)))
    (let ((run-proc (get-buffer-process (current-buffer))))
      (if run-proc
	  (if (or (not (eq (process-status run-proc) 'run))
		  (yes-or-no-p
		   (format "A %s process is running; kill it? "
			   name-of-mode)))
	      (condition-case ()
		  (progn
		    (interrupt-process run-proc)
		    (sit-for 1)
		    (delete-process run-proc))
		(error nil))
	    (error "Cannot have two processes in `%s' at once"
		   (buffer-name))
	    )))
; DONT					; )
    (let ((thisdir default-directory)
	  outwin)
; DONT					;      (save-excursion
      ;; Clear out the compilation buffer and make it writable.
      ;; Change its default-directory to the directory where the compilation
      ;; will happen, and insert a `cd' command to indicate this.
      (setq buffer-read-only nil)
      (buffer-disable-undo (current-buffer))
      (erase-buffer)
      (buffer-enable-undo (current-buffer))
      (setq default-directory thisdir)
      (insert "cd " thisdir "\n java " app-name "\n")
      (set-buffer-modified-p nil)
					;)
      ;; If we're already in the output buffer, go to the end
      ;; of the buffer, so point will track the compilation output.
      (if (eq outbuf (current-buffer))
	  (goto-char (point-max)))
      ;; Pop up the output buffer.
					;      (save-excursion
      (goto-char (point-max))
      ;; Make it so the next C-x ` will use this buffer. 
      (setq jde-run-last-buffer outbuf)
      (jde-run-mode outbuf)
      (setq default-directory thisdir)
      (set-buffer outbuf)
      (setq outwin (get-buffer-window outbuf))
      (set-window-start outwin (point-min))
      (setq mode-name name-of-mode)
      (or (eq outwin (selected-window))
	  (set-window-point outwin (point-min)))
      (jde-run-set-window-height outwin)
      ;; Start the java app.
      (if (fboundp 'start-process)
	  (let* ((process-environment (cons "EMACS=t" process-environment))
		 (proc (start-process-shell-command (downcase mode-name)
						    outbuf
						    command)))
	    (set-process-sentinel proc 'jde-run-sentinel)
	    (set-process-filter proc 'jde-run-filter)
	    (set-marker (process-mark proc) (point) outbuf)
	    (setq jde-run-in-progress
		  (cons proc jde-run-in-progress)))
	;; No asynchronous processes available.
	(message "Running `%s'..." java-app)
	;; Fake modeline display as if `start-process' were run.
	(setq mode-line-process ":run")
	(force-mode-line-update)
	(sit-for 0)                   ;force redisplay
	(let ((status (call-process shell-file-name nil outbuf nil "-c"
				    command)))
	  (cond ((numberp status)
		 (jde-run-handle-exit 'exit status
				      (if (zerop status)
					  "finished\n"
					(format "exited abnormally with code %d\n" status))))
		((stringp status)
		 (jde-run-handle-exit 'signal status
				      (concat status "\n")))
		(t
		 (jde-run-handle-exit 'bizarre status status))))
	(message "Executing `%s'...done" java-app)))))
					; )
	    
;; Set the height of WINDOW according to jde-run-window-height.
(defun jde-run-set-window-height (window)
  (and jde-run-window-height
       (= (window-width window) (frame-width (window-frame window)))
       ;; If window is alone in its frame, aside from a minibuffer,
       ;; don't change its height.
       (not (eq window (frame-root-window (window-frame window))))
       ;; This save-excursion prevents us from changing the current buffer,
       ;; which might not be the same as the selected window's buffer.
       (save-excursion
	 (let ((w (selected-window)))
	   (unwind-protect
	       (progn
		 (select-window window)
		 (enlarge-window (- jde-run-window-height
				    (window-height))))
	     (select-window w))))))

;;;###autoload
(defun jde-run-mode ()
  "Major mode for jde-run log buffers.
To kill the app, type \\[kill-app].

Runs `jde-run-mode-hook' with `run-hooks' (which see)."
  (interactive)
  (kill-all-local-variables)
  (use-local-map jde-run-mode-map)
  (setq major-mode 'jde-run-mode
	mode-name "Run Java")
  (run-hooks 'jde-run-mode-hook))							
	
;; Write msg in the current buffer and hack its mode-line-process.
(defun jde-run-handle-exit (process-status exit-status msg)
  (let ((buffer-read-only nil)
	(status (if jde-run-exit-message-function
		    (funcall jde-run-exit-message-function
			     process-status exit-status msg)
		  (cons msg exit-status)))
	(omax (point-max))
	(opoint (point)))
    ;; Record where we put the message, so we can ignore it
    ;; later on.
    (goto-char omax)
    (insert ?\n mode-name " " (car status))
    (forward-char -1)
    (insert " at " (substring (current-time-string) 0 19))
    (forward-char 1)
    (setq mode-line-process (format ":%s [%s]" process-status (cdr status)))
    ;; Force mode line redisplay soon.
    (force-mode-line-update)
    (if (and opoint (< opoint omax))
	(goto-char opoint))
    (if jde-run-finish-function
	(funcall jde-run-finish-function (current-buffer) msg))))

;; Called when Java app process changes state.
(defun jde-run-sentinel (proc msg)
  "Sentinel for Java app buffers."
  (let ((buffer (process-buffer proc)))
    (if (memq (process-status proc) '(signal exit))
	(progn
	  (if (null (buffer-name buffer))
	      ;; buffer killed
	      (set-process-buffer proc nil)
	    (let ((obuf (current-buffer)))
	      ;; save-excursion isn't the right thing if
	      ;; process-buffer is current-buffer
	      (unwind-protect
		  (progn
		    ;; Write something in the Java app output buffer
		    ;; and hack its mode line.
		    (set-buffer buffer)
		    (jde-run-handle-exit (process-status proc)
					  (process-exit-status proc)
					  msg)
		    ;; Since the buffer and mode line will show that the
		    ;; process is dead, we can delete it now.  Otherwise it
		    ;; will stay around until M-x list-processes.
		    (delete-process proc))
		(set-buffer obuf))))
	  (setq jde-run-in-progress (delq proc jde-run-in-progress))
	  ))))

(defun jde-run-filter (proc string)
  "Process filter for Java app output buffers.
Just inserts the text, but uses `insert-before-markers'."
  (if (buffer-name (process-buffer proc))
      (save-excursion
	(set-buffer (process-buffer proc))
	(let ((buffer-read-only nil))
	  (save-excursion
	    (goto-char (process-mark proc))
	    (insert-before-markers string)
	    (run-hooks 'jde-run-filter-hook)
	    (set-marker (process-mark proc) (point)))))))	

(defun kill-app ()
  "Kill the process made by the \\[jde-run] command."
  (interactive)
  (let ((buffer (jde-run-find-buffer)))
    (if (get-buffer-process buffer)
	(interrupt-process (get-buffer-process buffer))
      (error "The Java app is not running."))))

(defsubst jde-run-buffer-p (buffer)
  (save-excursion
    (set-buffer buffer)
    (eq major-mode 'jde-run-mode)))
  
;; Return a Java app output buffer.
;; If the current buffer is a Java app output buffer, return it.
;; If jde-run-last-buffer is set to a live buffer, use that.
;; Otherwise, look for a jde-run buffer and signal an error
;; if there are none.
(defun jde-run-find-buffer (&optional other-buffer)
  (if (and (not other-buffer)
	   (jde-run-buffer-p (current-buffer)))
      ;; The current buffer is a Java app output buffer.
      (current-buffer)
    (if (and jde-run-last-buffer (buffer-name jde-run-last-buffer)
	     (jde-run-buffer-p jde-run-last-buffer)
	     (or (not other-buffer) (not (eq jde-run-last-buffer
					     (current-buffer)))))
	jde-run-last-buffer
      (let ((buffers (buffer-list)))
	(while (and buffers (or (not (jde-run-buffer-p (car buffers)))
				(and other-buffer
				     (eq (car buffers) (current-buffer)))))
	  (setq buffers (cdr buffers)))
	(if buffers
	    (car buffers)
	  (or (and other-buffer
		   (jde-run-buffer-p (current-buffer))
		   ;; The current buffer is a Java output buffer.
		   (progn
		     (if other-buffer
			 (message "This is the only Java app output buffer."))
		     (current-buffer)))
	      (error "No Java app started!")))))))

(defvar jde-run-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map "\C-c\C-k" 'kill-app)
    ;; Set up the menu-bar
    (define-key map [menu-bar jde-run-menu]
      (cons "Run Java App" (make-sparse-keymap "Run Java App")))

    (define-key map [menu-bar jde-run-menu jde-run-mode-kill-app]
      '("Stop Java App" . kill-app))
    map)
    "Keymap for `jde-run-mode'.")


(provide 'jde-run)

;;; jde-run.el ends here