;;; ljw.el --- a rudimentary Java development environment
;;  Author: Lucian Wischik <ljw1004@cam.ac.uk>, October 1997
;;  Heavily dependent upon JDE, by Paul Kinnucan <paulk@mathworks.com>
;;  
;;; Justification
;;  --------------
;;  The aim of this package is to provide a rudimentary java development
;;  environment for java. The idea is that students learning java can
;;  use the same environment on both Windows and Unix, and that they
;;  can spend their time learning how to program without having to learn
;;  how the development tools work. (In any case, I reckon that the
;;  development tools are kind of primitive at the moment). Also,
;;  this emacs environment means that you can use the latest version
;;  of the JDK from JavaSoft, rather than being stuck with development
;;  environments that lag behind. This development environment does
;;  syntax hiliting via font-lock. It provides no formatting tools.
;;  (I figure that no one uses them anyway). It's main aim is to provide
;;  a convenient 'build' menu suitable for beginners, in the spirit
;;  of the build menus in Borland C++Builder and Visual C++. It is
;;  aimed at beginning-level Java programmers.

;;; Interface Overview
;;  --------------------
;;  Whenever you start editing a .java file, this package kicks in.
;;  Whenever the .java file has focus, there is a 'java' menu
;;  on the menubar with the following options:
;;
;;    Compile buffer
;;    Make project <proj-name>
;;    Run project <proj-name>
;;    ---
;;    First compile error
;;    Prev compile error
;;    Next compile erro
;;    ---
;;    Debug
;;    Run!
;;    Set breakpoint
;;    Clear breakpoint
;;    Step into
;;    Step over
;;    ---
;;    Options
;;
;;  The first notion to understand is that of a 'project'. A project
;;  is identified by its main class: that is, the class that gets
;;  executed. In standalone applications the main class has
;;  a method 'public static void main(String[] args)'. In an applet
;;  the main class 'extends Applet'. If your program consists only
;;  of a single file, then the menu options Compile-buffer and
;;  Make-project do exactly the same thing. If your program consists
;;  of lots of classes in different files, then Make-project will
;;  compile all of them. Run-project first checks that the thing
;;  has been compiled and is up to date, and then it runs it.
;;
;;  The first time you make a project, Emacs will ask you which is
;;  the main class and whether it is an applet or standalone. It
;;  remembers your answers by writing them in two lines at the
;;  start of the file. (If you dislike this behaviour you can
;;  disable it in your .emacs file: see below).
;;
;;  The three compilation-error menu options (first-error, prev-error
;;  and next-error) are pretty much self-explanatory. They only work
;;  when you have just compiled the thing and it had errors in it.
;; 
;;  Debugging is invoked with the Debug menu option. This launches
;;  the debugger, jdb. From then on, you can type debugging commands into
;;  the debugging window. The menu options (run, set- and clear-breakpoint,
;;  step-over and step-into) are simply shortcuts for various debugging
;;  commands.
;;
;;  You can customize the way this package works. This is done with
;;  elisp commands in your .emacs file. It's not difficult!

;;; Program Overview
;;  ------------------
;;  The single line that installs this package is near the end:
;;  (add-hook 'java-mode-hook 'ljw-java-mode-hook)
;;  Whenever Emacs loads a .java file, it executes the function
;;  ljw-java-mode-hook. This turns on font-lock support and
;;  on-the-fly syntax hiliting, and turns on the Java menu.
;;
;;  The general convention is that those settings which appear to the
;;  user to be concerned with the java menu, and java stuff, are
;;  prefixed with 'java-'. Those that are related just to the internal
;;  workings of the package are prefixed with 'ljw-'. The intention is
;;  that the java- prefix is intuitive and easy to use, and that the
;;  ljw- stuff doesn't clutter up the general namespace.
;;
;;  java-compile is done in a chain. It sets off the compilation.
;;  When compilation is done the function ljw-post-compile-buffer gets
;;  called. That (if appropriate) moves the cursor to the first line
;;  which had an error. The same sort of trick is done for
;;  java-make and java-execute.
;;
;;  Most of the stuff is pretty much self-explanatory, and it's too
;;  late for me to go to the effort of writing about it all here.



(require 'compile)
(require 'jde-run)
(require 'jde-db)




;;; GLOBAL VARIABLES

(defvar ljw-is-debug-started nil
  "Flag for whether the current debug session has started")
; This controls whether under debug it lets you 'Run!' or 'Continue'
(defvar ljw-running-xemacs (string-match "XEmacs\\|Lucid" emacs-version)
  "Flag for whether we're running xemacs")



;;; GENERAL UTILITY THINGS

(defun bashify-command (cmd)
  "If bash is the current shell, returns CMD enclosed in quotes.
This satisfies bash requirement that the -c command line option
be enclosed in quotes."
  (if (and (eq system-type 'windows-nt)
	  (or (string-match "bash" shell-file-name)
	      (string-match "bash" (getenv "SHELL"))))
      (concat "\"" cmd "\"") cmd))
(defun trim (c)
  "Removes space characters before and after the string"
  (let ((len (length c)))
    (cond ((string= c "") "")
	  ((string= (substring c 0 1) " ")
	   (trim (substring c 1)))
	  ((string= (substring c (- len 1) len) " ")
	   (trim (substring c 0 (- len 1))))
	  (t c))))
(defun ljw-scan-project-name ()
  "Scans the first two lines of the current buffer to look for a
'// Main class: test' line, which indicates the current project name."
  (save-excursion
    (let* ((len (buffer-size))
	   (text-area (if (equal len 0) ""
			(buffer-substring-no-properties 1 (min len 1024))))
	   (pos-start (string-match "Main class:" text-area)))
      (if pos-start
	  (let ((pos-end (string-match "$" text-area pos-start)))
	    (if pos-end
		(trim (substring text-area (+ pos-start 11) pos-end))
	      nil))
	nil))))
(defun ljw-scan-project-type ()
  "Scans the first few lines of the current buffer to look for a
// Project type: standalone/applet line, which indicates the type
of the current project."
  (save-excursion
    (let* ((len (buffer-size))
	   (text-area (if (equal len 0) ""
			(buffer-substring-no-properties 1 (min len 1024))))
	   (pos-start (string-match "Project type:" text-area)))
      (if pos-start
	  (let ((pos-end (string-match "$" text-area pos-start)))
	    (if pos-end
		(trim (substring text-area (+ pos-start 13) pos-end))
	      nil))
	nil))))
(defun ljw-scan-project-arguments ()
  "Scans the first few lines of the current buffer to look for a
// Arguments: ... line"
  (save-excursion
    (let* ((len (buffer-size))
	   (text-area (if (equal len 0) ""
			(buffer-substring-no-properties 1 (min len 1024))))
	   (pos-start (string-match "Arguments:" text-area)))
      (if pos-start
	  (let ((pos-end (string-match "$" text-area pos-start)))
	    (if pos-end
		(trim (substring text-area (+ pos-start 10) pos-end))
	      nil))
	nil))))
(defun ljw-scan-project-command ()
  "Scans the first few lines of the current buffer to look for a
// Compile command: ... line"
  (save-excursion
    (let* ((len (buffer-size))
	   (text-area (if (equal len 0) ""
			(buffer-substring-no-properties 1 (min len 1024))))
	   (pos-start (string-match "Compile command:" text-area)))
      (if pos-start
	  (let ((pos-end (string-match "$" text-area pos-start)))
	    (if pos-end
		(trim (substring text-area (+ pos-start 16) pos-end))
	      nil))
	nil))))
(defun ljw-query-type ()
  "Queries the user for the type of the thing, in the echo area,
and returns either 'applet' or 'windowapp' or 'console'"
  (interactive)
  (let ((ans nil) (isFirst t))
    (while (not ans)
      (if isFirst (message "Applet, Window-app or Console? (a/w/c)")
	(message (concat "Please answer a,w or c. Is it an applet, "
			 "Window-app or Console application?")))
      (let ((c (read-char)))
	(setq isFirst nil)
	(cond ((or (equal c ?a) (equal c ?A)) (setq ans "applet"))
	      ((or (equal c ?w) (equal c ?W)) (setq ans "windowapp"))
	      ((or (equal c ?c) (equal c ?C)) (setq ans "console")))))
    ans))
(defun ljw-get-project-details ()
  "Returns the name and type of the main class currently in use. In the first
instance, that will be the one taken from the comment lines at the
start of the file (// Main class: and // Project type:). If that's not
there, it will try to use the variables ljw-main-class-default and
ljw-project-type-default that might optionally have been set in .emacs.
And failing that, it will ask the user in the minibuffer and write
the answers as comments. (Also does project arguments)."
  (interactive)
  (save-excursion
    (let ((scanned-name (ljw-scan-project-name))
	  (scanned-type (ljw-scan-project-type))
	  (scanned-args (ljw-scan-project-arguments))
	  (scanned-command (ljw-scan-project-command))
	  (must-insert-class nil)
	  (must-insert-type nil)
	  (must-insert-args nil)
	  (must-insert-command nil)
	  (return-list nil))
      (if (not (or (string= scanned-type "applet")
		   (string= scanned-type "console")
		   (string= scanned-type "windowapp")))
	  (setq scanned-type nil))
      (setq
       return-list
       (list
	;; if there is a scanned name, use it. Otherwise try fallback options
	(cond (scanned-name scanned-name)
	      ((and (boundp 'ljw-main-class-default)
		    ljw-main-class-default
		    (not (string= ljw-main-class-default "")))
	       ljw-main-class-default)
	      (t (let* ((def-class
			  (file-name-sans-extension
			   (file-name-nondirectory buffer-file-name)))
			(new-name
			 (read-string "Which is the main class? " def-class)))
		   (setq must-insert-class new-name)))) ; ret. val. is new-name
	;; likewise for the type
	(cond (scanned-type scanned-type)
	      ((and (boundp 'ljw-project-type-default)
		    ljw-project-type-default
		    (not (string= ljw-project-type-default "")))
	       ljw-project-type-default)
	      (t (let ((new-type (ljw-query-type)))
		   (setq must-insert-type new-type))))
	;; likewise for the arguments
	(cond (scanned-args scanned-args)
	      ((and (boundp 'ljw-arguments-default)
		    ljw-arguments-default)
	       ljw-arguments-default)
	      (t (let ((new-args (read-string "Arguments? " "")))
		   (setq must-insert-args new-args)))) ;; ret.val. is new-args
	;; likewise for the compile command
	(cond (scanned-command scanned-command)
	      ((and (boundp 'ljw-command-default)
		    ljw-command-default)
	       ljw-command-default)
	      (t (let ((new-command
			(read-string "Compile command? "
				     "javac -g -deprecation")))
		   (setq must-insert-command new-command)))))) ;; ret.val.
      (if must-insert-command
	  (save-excursion
	    (goto-char (point-min))
	    (insert (concat "// Compile command: "
			    must-insert-command "\n"))))
      (if must-insert-args
	  (save-excursion
	    (goto-char (point-min))
	    (insert (concat "// Arguments: " must-insert-args "\n"))))
      (if must-insert-type
	  (save-excursion
	    (goto-char (point-min))
	    (insert (concat "// Project type: " must-insert-type "\n"))))
      (if must-insert-class
	  (save-excursion
	    (goto-char (point-min))
	    (insert (concat "// Main class: " must-insert-class "\n"))))
      return-list)))
(defun ljw-scan-package ()
  "Returns the package of the class whose source file resides in the current
buffer. By Paul Kinnucan."
  (save-excursion
    (goto-char (point-min))
    (if (re-search-forward "\\(package\\) *\\([^ ]*\\) *;" (point-max) t)
	(concat (buffer-substring-no-properties
		 (match-beginning 2) (match-end 2)) "."))))
(defun ljw-maybe-save ()
  "If necessary, prompts the user to see whether they want to save
the current bufer."
  ;; if the user had set ljw-disable-autosave in their .emacs,
  ;; then we'll need to prompt. (passed as the first argument to
  ;; save-some-buffers).
  (let ((need-to-prompt (if (boundp 'ljw-disable-autosave)
			    t nil)))
    (if (eq system-type 'windows-nt)
	(let ((temp last-nonmenu-event))
	  ;; next line makes emaks think that it was invoked from minibuffer
	  ;; to avoid a crash
	  (setq last-nonmenu-event t)
	  (save-some-buffers (not need-to-prompt) nil)
	  (setq last-nonmenu-event temp))
      ;; otherwise we can just do it normally
      (save-some-buffers (not need-to-prompt) nil))))
(defun ljw-get-interpret-command ()
  "Returns either the string 'java', or, if the user happened to set
ljw-java-interpreter to something in their .emacs, returns that."
  (if (and (boundp 'ljw-java-interpreter)
	   ljw-java-interpreter
	   (not (string= ljw-java-interpreter "")))
      ljw-java-interpreter
    "java"))
(defun ljw-get-viewer-command ()
  "Returns either the string 'appletviewer', or, if the user happened to set
ljw-java-viewer to something in their .emacs, returns that."
  (if (and (boundp 'ljw-java-viewer)
	   ljw-java-viewer
	   (not (string= ljw-java-viewer "")))
      ljw-java-viewer
    "appletviewer"))
(defun ljw-convert-class-to-classfile (class)
  "Given the name of a class, returns the filename for that class."
  (concat class ".class"))
(defun ljw-convert-class-to-javafile (class)
  "Given the name of a class, returns the java filename for its source."
  (concat class ".java"))
(defun ljw-convert-class-to-htmlfile (class)
  "Given the name of a class, returns the html filename for its html."
  (concat class ".html"))
(defun ljw-convert-javafile-to-class (javafile)
  "Given the name of a java file, returns the main class of it."
  (file-name-sans-extension javafile))
(defun ljw-convert-javafile-to-classfile (javafile)
  "Given the name of a java file, returns the class file for it."
  (ljw-convert-class-to-classfile (ljw-convert-javafile-to-class javafile)))
(defun ljw-maybe-first-error ()
  "Unless the ljw-disable-autoscanerrors flag has been set, this command
goes to the first compilation error."
  (if (not (boundp 'ljw-disable-autoscanerrors))
      (java-first-error)))
(defun ljw-list-from-padded (s)
  (ljw-list-from-padded-internal (trim s)))
(defun ljw-list-from-padded-internal (s)
  (if (string= s "") nil
    (let ((i (string-match " " s)))
      (if i
	  (let ((a (substring s 0 i))
		(ar (ljw-list-from-padded-internal
		     (trim (substring s (+ 1 i))))))
	    (cons a ar))
	(list s)))))
(defun ljw-padded-from-list (sl)
  (ljw-padded-from-list-internal nil sl))
(defun ljw-padded-from-list-internal (mustSpace sl)
  (if sl
      (concat (if mustSpace " " "") (car sl)
	      (ljw-padded-from-list-internal t (cdr sl)))
    ""))

  




;;; CLASSPATH UTILITY THINGS
;;  Classpath is a really horrible environment variable.
;;  Here are some last-ditch attempts to check that it's okay.
;;
;;  The way the CLASSPATH variable works is this.
;;  if it's empty, then .:<java-path>/lib/classes.zip  ends up getting put
;;  into it automatically by the various JDK tools.
;;  But if they are run and they find CLASSPATH non-empty, then they
;;  don't! So, if by mistake you happened to set CLASSPATH to
;;  include only the JDK classes, then it will not end up including
;;  the current directory (or at least, the default-directory in which
;;  you compile and run). The issue is that the javac flag -depend
;;  requires that the various dependencies be accessible via classpath.
;;  So, if you are writing your classes in a temporary working directory,
;;  and your classpath happens not to include '.', and you try to do
;;  a javac -depend (or Java|MakeProject) then it won't work. And it
;;  will give really silly and incomprehensible error messages.
;;  That's why I think it's my responsibility to check.
(defun ljw-maybe-check-classpath ()
  "Unless specified otherwise, this function checks that the classpath
environment variable is properly set. You can disable the check by
setting ljw-disable-classpathcheck in your .emacs."
  (if (boundp 'ljw-disable-classpathcheck) 
      t
    (ljw-check-classpath default-directory)))
(defun ljw-check-classpath (dir)
  "Checks that either the classpath environemnt variable includes the
current directory '.', or that it includes the directory specified as
an argument."
  (interactive "s")
  (let ((classpath (getenv "CLASSPATH"))
	(fulldir (expand-file-name dir)))
    (if (or (not classpath)
	    (string= classpath "")
	    (string= classpath "\.")
	    (string-match "\.:" classpath)
	    (string-match ":\." classpath)
	    (ljw-check-pathinlist fulldir
				  (ljw-tokenize classpath)))
	t
      (progn
	(if (y-or-n-p (concat
		       "Warning: CLASSPATH doesn't contain working dir. "
		       "Continue anyway? "))
	    t
	  (progn (message "") nil))))))
;; ljw-expand is just to expand a single directory name. This is because
;; maybe you have something like ~ljw1004/classes in your CLASSPATH.
;; And emacs thinks it is editing /home/ljw1004/classes/thing.java.
;; So we need to expand each entry in the classpath and then check
;; if they match what emacs thinks is it's current directory.
;; There is one problem. Emacs will expand
;;    .         -> /home/ljw1004/java
;;    ~/classes -> /Nfs/bescot/usr8/ljw1004/java
;; Our solution, as above, was to put lots of checks for '.' explicitly.
(defun ljw-expand (path)
  (file-name-as-directory (expand-file-name (directory-file-name path))))
(defun ljw-tokenize (path)
  "Separates path into a list, figuring out automatically what the
separator is."
  (ljw-tokenize-internal-one path nil))
(defun ljw-tokenize-sans-dot (path)
  "Separates path into a list, removing ."
  (ljw-tokenize-internal-one path t))
(defun ljw-tokenize-internal-one (path removeDot)
  "Separates path into a list. removeDot says whether or not to remove ."
  (if (string-match ";" path)
      (ljw-tokenize-internal-two path ";" removeDot)
    (if (string-match ":\\\\" path)     ; checks for things like D:\
	(ljw-tokenize-internal-two path ";" removeDot)
      (ljw-tokenize-internal-two path ":" removeDot))))
(defun ljw-tokenize-internal-two (path sep removeDot)
  "The function ljw-tokenize takes in a sep-separated list of paths
and returns the same but in list form. It expands each item. Sep will be
either : or ; presumably."
  (if (string= path "")
      nil
    (let ((i (string-match sep path)))
      (if i
	  (let ((a (substring path 0 i)))
	    (if (and (string= a ".") removeDot)
		(ljw-tokenize-internal-two
		 (substring path (+ 1 i)) sep removeDot)
	      (cons (ljw-expand (substring path 0 i))
		    (ljw-tokenize-internal-two
		     (substring path (+ 1 i)) sep removeDot))))
	(if (and (string= path ".") removeDot) nil
	  (list (ljw-expand path)))))))
(defun ljw-check-pathinlist (path list)
  "Given a path, and a list of fully-expanded directory names,
this function checks that the path is somewhere in the list."
  (if (not list) nil
    (if (string= path (car list)) t
      (ljw-check-pathinlist path (cdr list)))))
(defun ljw-frontslashify (p)
  "For every string in list p, replaces backslashes with forward slashes"
  (if (not p) p
    (let* ((hd (car p))
	   (tl (cdr p))
	   (i (string-match "\\\\" hd)))
      (if (not i) (cons hd (ljw-frontslashify tl))
	(let ((ns (concat (substring hd 0 i)
			"/"
			(substring hd (+ i 1)))))
	  (ljw-front-slashify (cons ns tl)))))))
(defun ljw-add-srcdir (p)
  "In the list, if ever there is .../xxx/classes.zip or .../classes or
.../bin, adds a .../src/"
  (interactive)
  (if (not p) p
    (let* ((hd (car p))
	   (tl (cdr p))
	   (hdmark (concat hd "_LJWEND")) ; a marker of the end of string
	   (iR (string-match "/[^/]*/classes.zip/_LJWEND" hdmark))
	   (iCZ (string-match "/classes.zip/_LJWEND" hdmark))
	   (iC (string-match "/classes/_LJWEND" hdmark))
	   (iB (string-match "/bin/_LJWEND" hdmark)))
      (cond
       ((string= hd ".") (cons "" (ljw-add-srcdir tl)))
       (iB (cons (concat (substring hd 0 iB) "/src/") (ljw-add-srcdir tl)))
       (iC (cons (concat (substring hd 0 iB) "/src/") (ljw-add-srcdir tl)))
       (iR (cons (concat (substring hd 0 iR) "/src/")
		 (cons (concat (substring hd 0 iCZ) "/src/")
		       (ljw-add-srcdir tl))))
       (iCZ (cons (concat (substring hd 0 iCZ) "/src/") (ljw-add-srcdir tl)))
       (t (cons hd (ljw-add-srcdir tl)))))))
(defun ljw-ensure-exist (p)
  "In the list p, remove all directories which don't exist"
  (if (not p) p
    (let* ((hd (car p))
	   (tl (cdr p))
	   (dirasf (substring hd 0 -1))
	   (exists (file-accessible-directory-p dirasf)))
      (if exists (cons hd (ljw-ensure-exist tl))
	(ljw-ensure-exist tl)))))
(defun ljw-construct-db-path ()
  "If ljw-db-source-directories has been set, then returns that; if not,
then turns classpath into a list of directories, with slash at their end,
ensuring that wherever there is a .../xxx/classes.zip or .../classes or
.../bin there is also a .../src/"
  (interactive)
  (if (boundp 'ljw-db-source-directories)
      (if (listp ljw-db-source-directories)
	  ljw-db-source-directories
	(ljw-tokenize ljw-db-source-directories))
    (let ((classpath (getenv "CLASSPATH")))
      (if (not (and (boundp 'classpath)
		    classpath
		    (not (string= classpath ""))))
	  (setq classpath (getenv "PATH"))) ; in case classpath was not set
      (append (ljw-ensure-exist
	       (ljw-add-srcdir
		(ljw-frontslashify
		 (ljw-tokenize-sans-dot classpath))))
	      (list "")))))



;;; COMPILE-BUFFER
;;  Menu item. Checks if the current buffer should be saved. Then
;;  compiles the current buffer. Then (using a chain) takes you to the
;;  first compilation error when compilation has finished.
(defun java-compile ()
  "Compiles the current buffer."
  (interactive)
  (let* ((projdetails (ljw-get-project-details))
	 (projname (car projdetails))
	 (projjavafile (ljw-convert-class-to-javafile projname))
	 (projcommand (car (cdr (cdr (cdr projdetails))))))
    (setq compile-command 
	  (bashify-command
	   (concat projcommand
		   " \""
		   (file-name-nondirectory projjavafile)
		   "\"")))
  (ljw-maybe-save)
  (message "Compiling...")
  (setq compilation-finish-function 'ljw-post-compile)
  (compile-internal compile-command "No more errors")))
(defun ljw-post-compile (buf desc)
  (if (string-equal (substring desc 0 5) (substring "finished" 0 5))
      (message "Compiled succesfully")
    (progn (message "Compilation error") (beep) (ljw-maybe-first-error))))



;;; MAKE-PROJECT
;;  Menu item. First checks if the current buffer should be saved. Then
;;  makes the current buffer. (Like compiling it, but with the -depend
;;  flag). Then (using a chain) takes you to the first compilation error.
(defun java-make ()
  "Makes the current project. First checks that the current project
has a name (specified by the line // Main class: ... at the start of
your file), prompting you if not. Then compiles it using the -depend
flag. Uses the ljw-java-compiler variable as the name of the compiler,
or assumes the name 'javac' if that is empty."
  (interactive)
  (ljw-make-internal 'ljw-post-compile))
(defun ljw-make-internal (end-function)
  "Makes the current project. Checks project name and type, and classpath."
  (let* ((projdetails (ljw-get-project-details))
	 (projname (car projdetails))
	 (projjavafile (ljw-convert-class-to-javafile projname))
	 (projtype (car (cdr projdetails)))
	 (projargs (car (cdr (cdr projdetails))))
	 (projcommand (car (cdr (cdr (cdr projdetails)))))
	 (isapplet (string= projtype "applet")))
    (ljw-maybe-save)
    (ljw-update-menu)
    (if (string= projname "") (message "Main class name is invalid!")
      (if (ljw-maybe-check-classpath)
	  (progn
	    (setq compile-command 
		  (bashify-command
		   (concat projcommand
			   " \""
			   (file-name-nondirectory projjavafile)
			   "\"")))
	    (message "Compiling...")
	    (setq compilation-finish-function end-function)
	    (compile-internal compile-command "No more errors"))))))
  


;;; EXECUTE-PROJECT
;;  Menu item. First does the same as make. Then, chaining, it either
;;  takes you to the first compilation error or (hopefully) runs it.
;;  Running it might involve creating a .html file if one isn't already
;;  there.
(defun java-execute ()
  "If necessary, makes the current projects. Then runs it either
stand-alone or as an applet."
  (interactive)
  (ljw-maybe-save)
  (ljw-execute-internal-one nil))
(defun ljw-execute-internal-one (isDebug)
  "The first step in executing a project. Compiles it and then
chains on to the next step; or, if it is already compiled, then
just executes the next step immediately."
  (interactive)
  ;; We check two things here. First, we assume that the current file has
  ;; the name xxx.java and we check that xxx.class is at least as up to
  ;; date. Then we look for the main-class name of this project, MCN,
  ;; and check that MCN.class is at least as up to date as MCN.java.
  (let* ((isUptodate t)    ; is the .class file uptodate?
	 (isTerminal nil)  ; if not, is the fact that it isn't, terminal?
	 (reason "")
	 (projdetails (ljw-get-project-details))
	 (projclass (car projdetails))
	 (projtype (car (cdr projdetails)))
	 (projjavafile (ljw-convert-class-to-javafile projclass))
	 (projclassfile (ljw-convert-class-to-classfile projclass))
	 (isapplet (string= projtype "applet"))
	 (thisjavafile (file-name-nondirectory buffer-file-name))
	 (thisclass (ljw-convert-javafile-to-class thisjavafile))
	 (thisclassfile (ljw-convert-class-to-classfile thisclass))
	 (cancontinue t)
	 (mustrecompile t))
    (ljw-maybe-save)
    (ljw-update-menu)
    (cond ((not (file-exists-p thisclassfile))
	   (progn (setq isUptodate nil)
		  (setq isTerminal t)
		  (setq reason (concat thisclassfile " doesn't exist."))))
	  ((file-newer-than-file-p thisjavafile thisclassfile)
	   (progn (setq isUptodate nil)
		  (setq reason (concat thisjavafile " has changed."))))
	  ((not (file-exists-p projclassfile))
	   (progn (setq isUptodate nil)
		  (setq isTerminal t)
		  (setq reason (concat projclassfile " doesn't exist."))))
	  ((file-newer-than-file-p projjavafile projclassfile)
	   (progn (setq isUptodate nil)
		  (setq reason (concat projjavafile " has changed.")))))
    (if (not isUptodate)
	(if (y-or-n-p (concat reason " Do you want to recompile? "))
		      (setq mustrecompile t)
	  (progn (setq mustrecompile nil)
		 (setq cancontinue (not isTerminal)))))
    ;; that means that, if they don't want to recompile but it wasn't terminal,
    ;; we can continue.
    (setq ljw-execute-remember-projdetails projdetails)
    (setq ljw-execute-remember-isDebug isDebug)
    (setq ljw-execute-remember-scanPackage (ljw-scan-package))
    (if cancontinue
	(if (and mustrecompile (not isUptodate))
	    (ljw-make-internal 'ljw-execute-internal-two)
	  (ljw-execute-internal-two "internal" "finished")))))

(defun ljw-execute-internal-two (buf desc)
  "If the thing is an applet, this command ensures that the corresponding
.html file is present and then appletviews it. If it's a standalone
program, then it just javas it. Both cases use the function
ljw-execute-internal-three to manage the buffers &c. of execution."
  (interactive)
  (if (not (string-equal (substring desc 0 5) (substring "finished" 0 5)))
      (progn (message "Compilation error") (beep) (ljw-maybe-first-error))
    (let* ((isUptodate t)
	   (isTerminal nil)
	   (reason "")
	   (projdetails ljw-execute-remember-projdetails)
	   (projclass (car projdetails))
	   (projtype (car (cdr projdetails)))
	   (projargs (car (cdr (cdr projdetails))))
	   (isapplet (string= projtype "applet"))
	   (isconsole (string= projtype "console"))
	   (projhtmlfile (ljw-convert-class-to-htmlfile projclass))
	   (runcommand (if isapplet (ljw-get-viewer-command)
			 (ljw-get-interpret-command)))
	   (rundata (cons (if isapplet projhtmlfile projclass)
			  (ljw-list-from-padded projargs)))
	   (cancontinue t))
      (cond ((and isapplet (not (file-exists-p projhtmlfile)))
	     (progn (setq isUptodate nil)
		    (setq isTerminal t)
		    (setq reason (concat projhtmlfile " doesn't exist.")))))
      (if (not isUptodate)
	  (if (y-or-n-p (concat reason " Do you want to create it? "))
	      t
	    (setq cancontinue (not isTerminal))))
      (if cancontinue
	  (progn
	    (if (not isUptodate)
		(ljw-generate-htmlfile projclass projhtmlfile))
	    (if ljw-execute-remember-isDebug
		(ljw-debug-internal-three runcommand rundata)
	      (ljw-execute-internal-three runcommand rundata isconsole)))))))

(defun ljw-execute-internal-three (initcmd args isconsole)
  "Runs cmd (first argument) on data-file (second argument). Manages
buffers and stuff. Note that under Windows, emacs will use a pipe rather
than a PTY. This means that ctrl-c doesn't work, ctrl-d does, and that
you cannot use BufferedReaders with buffers larger than 1 character
in your program."
  (interactive "sCommand:\nsArgs:")
  (let* ((name-of-mode "Run Java")
	 (outbuf (get-buffer-create (concat "*" name-of-mode "*")))
	 (outwin nil)
	 (thisdir default-directory)
	 (cmd (if (string= shell-file-name "bash") (concat "\"" initcmd "\"")
		initcmd))
	 (run-proc (get-buffer-process outbuf)))
    (if run-proc
	(if (or (not (eq (process-status run-proc) 'run))
		(yes-or-no-p
		 (format "A %s process is running; kill it? "
			 name-of-mode)))
	    (condition-case ()
		(progn
		  (interrupt-process run-proc)
		  (sit-for 1)
		  (kill-process run-proc)
		  (delete-process run-proc)))
	  (error "Cannot have two processes in `%s' at once"
		 (buffer-name))))
    (set-buffer outbuf)
    (setq buffer-read-only nil)
    (buffer-disable-undo (current-buffer))
    (erase-buffer)
    (buffer-enable-undo (current-buffer))
    (setq default-directory thisdir)
    (insert "cd " thisdir "\n" cmd " " (ljw-padded-from-list args) "\n")
    (set-buffer-modified-p nil)
    (setq outwin (display-buffer outbuf))
    (set-window-start outwin (point-min))
    (set-window-point outwin (point-min))
    (goto-char (point-max))
    (setq jde-run-last-buffer outbuf)
    (message "Running...")
    (setq default-directory thisdir)
    (let* ((process-environment (cons "EMACS=t" process-environment))
	   (process-connection-type t)
	   (mustwarn (eq system-type 'windows-nt))
	   ;; The user may have ljw-execute-in-shell set. If so,
	   ;; we execute in a shell. This has the advantage of letting
	   ;; pipes be understood. It has the disadvantage that, if
	   ;; the thing happens to be running in a pipe rather than a PTY
	   ;; (ie. under Windows), then SIGKILL doesn't work.
	   ;; Applets can only be run as shells. Don't know why.
	   ;; NB. 'proc' is the name of a variable we are binding.
	   ;; NB. that executing it as a pipe requires that the
	   ;; arguments really be separate arguments: we can't rely on
	   ;; the shell parsing them.
	   ;; We'll only bother trying to do it as a shell if the system
	   ;; happens to be Windows. (which doesn't have PTYs).
	   (proc (if (and (not (boundp 'ljw-execute-consoleapps-in-shell))
			  isconsole
			  (eq system-type 'windows-nt))
		     (progn
		       (if mustwarn
			   (message
			    (concat "Running as a console. "
				    "No windowing stuff "
				    "or piping will work.")))
		       (eval
			(cons 'start-process
			      (append (list (downcase name-of-mode)
					    outbuf cmd)
				      args))))
		   (progn
		     (if mustwarn
			 (message
			  (concat
			   "Running as a shell. "
			   "Ctrl-c doesn't work. "
			   "Can only quit from within app.")))
		     (start-process-shell-command
		      (downcase name-of-mode) outbuf
		      (concat cmd " " (ljw-padded-from-list args)))))))
      (setq mode-name name-of-mode)
      (jde-run-mode)
      (comint-mode))))


(defun ljw-generate-htmlfile (class htmlfile)
  "Generates an html file called htmlfile (second argument) which
contains the applet class (first argument)."
  (interactive "s")
  (write-region
   (concat "<html>\n"
	   "<title>"
	   class " test"
	   "</title>\n"
	   "<body>\n"
	   "<hr>\n"
	   "<applet "
	   "code=\"" class "\" "
	   "width=500 height=300>\n"
	   "</applet>\n"
	   "<hr>\n"
	   "</body>\n"
	   "</html>\n")
   nil htmlfile))




;;; ERROR NAVIGATION
;;  Menu items. Takes you to the first compilation error, if there was one,
;;  and to previous, and to next.
;;  The first-error function is part of standard emacs compilation support
;;  in 'compile.el'.
(defun java-first-error ()
  "Moves to first line which had a compilation error or warning.
Only works if you have just completed a compilation."
  (interactive)
  (first-error))
(defun java-next-error ()
  "Moves to the line with the next error or warning.
Only works if you have just completed a compilation."
  (interactive)
  (next-error))
(defun java-prev-error ()
  "Moves to the line with the previous error or warning.
Only works if you have just completed a compilation."
  (interactive)
  (previous-error))









;;; DEBUG
;;  This uses chaining in a similar way to java-execute, in case the
;;  thing needs to be compiled before debugging.
(defun java-debug ()
  (interactive)
  (setq jde-db-source-directories (ljw-construct-db-path))
  (let* ((res (re-search-forward "Compile command:.*$" 1024 t))
	 (cmd (if res (buffer-substring (match-beginning 0) (match-end 0))
		""))
	(isDebug t))
;;    (if (not (string-match "javac_g" cmd)) (setq isDebug nil))
;; I've removed that because I really don't like the javac_g compiler,
;; even if the docs do say that it's the non-optimising one you
;; should be using for debugging. So what!
;; (it printed too many MONITOR-FREED messages)
    (if (not (string-match "-g" cmd)) (setq isDebug nil))
    (if (string-match "-O" cmd) (setq isDebug nil))
    (if (not isDebug)
	(if (y-or-n-p (concat "Do you want to change settings so it "
			      "compiles with debugging information? "))
	    (java-settings-replace "Compile command:"
				   "javac -g -deprecation")))
    (ljw-maybe-save)
    (ljw-execute-internal-one t)))
(defun ljw-debug-internal-three (cmd args)
  "Runs cmd (first argument) on data-file (second argument) within
debugger."
  (setq ljw-is-debug-started nil)
  (ljw-update-menu)
  (save-excursion
    (let ((selwin (selected-window))
	  (projdetails ljw-execute-remember-projdetails)
	  (projclass (car projdetails))
	  (projtype (car (cdr projdetails)))
	  (projjavafile (ljw-convert-class-to-javafile projclass))
	  (isapplet (string= projtype "applet"))
	  (projhtmlfile (ljw-convert-class-to-htmlfile projclass)))
      (ljw-db-internal 
       (if isapplet (concat
		     (ljw-viewer-command)
		     " -debug "
		     ljw-execute-remember-scanPackage
		     projhtmlfile
		     " "
		     (ljw-padded-from-list args))
	 (concat
	  "jdb "
	  ljw-execute-remember-scanPackage
	  projclass
	  " "
	  (ljw-padded-from-list args))))
      (select-window selwin))))





;;; OTHER DEBUGGING COMMANDS
;;  The command ljw-check-isdebugging is to make sure that the debug
;;  buffer is actually running before we actually issue any commands to it.
;;  The rest of the commands simply execute things within the debugger.
;;  Note the global flag ljw-is-debug-started. If this is 't', then
;;  the menu option will be for Continue. If it is nil, then the menu
;;  option will be for Run. So, when you start debugging, it is set to
;;  nil; and when you step in it gets set to Cont. That's because
;;  the debugging command 'run' starts execution from scratch, while
;;  the debugging command 'cont' continues. (And you have to get them
;;  right!)
(defun ljw-check-is-debugging ()
  (if (and (boundp 'gud-comint-buffer)
	   gud-comint-buffer)
      t
    (progn (message "Debugger not currently running") (beep) nil)))
;;
(defun java-run ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd "run"))
	(message cmd)
	(gud-basic-call cmd)
	(setq ljw-is-debug-started t)
	(ljw-update-menu))))
(defun java-cont ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd "cont"))
	(message cmd)
	(gud-basic-call cmd))))
(defun java-set-breakpoint ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd (concat "stop at "
			 (file-name-sans-extension
			  (file-name-nondirectory buffer-file-name))
			 ":"
			 (save-excursion
			   (beginning-of-line)
			   (save-restriction (widen)
					     (1+ (count-lines 1 (point))))))))
	(message cmd)
	(gud-basic-call cmd))))
(defun java-clear-breakpoint ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd (concat "clear "
			 (file-name-sans-extension
			  (file-name-nondirectory buffer-file-name))
			 ":"
			 (save-excursion
			   (beggining-of-line)
			   (save-restriction (widen)
					     (1+ (count-lines 1 (point))))))))
	(message cmd)
	(gud-basic-call cmd))))
(defun java-step-into ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd "step"))
	(message cmd)
	(gud-basic-call cmd)
	(setq ljw-debug-firstrun-p nil)
	(ljw-update-menu))))
(defun java-step-over ()
  (interactive)
  (if (ljw-check-is-debugging)
      (let ((cmd "next"))
	(message cmd)
	(gud-basic-call cmd)
	(setq ljw-debug-firstrun-p nil)
	(ljw-update-menu))))




;;; SETTINGS
;;  These are all just convenient ways to change the text at the top
(defun java-settings-compiler ()
  "Changes the current compile-command."
  (interactive)
  (let ((ans nil) (isFirst t))
    (while (not ans)
      (if isFirst
	  (message "Optimised, Debugging-information or Custom? (o/d/c)")
	(message (concat "Please answer o,d or c. Optimised, debugging "
			 "or custom?")))
      (let ((c (read-char)))
	(setq isFirst nil)
	(cond ((or (equal c ?o) (equal c ?O))
	       (setq ans "javac -O"))
	      ((or (equal c ?d) (equal c ?D))
	       (setq ans "javac -g -deprecation"))
	      ((or (equal c ?c) (equal c ?C))
	       (let* ((spc (ljw-scan-project-command))
		      (def-cmd (if spc spc "javac -g -deprecation"))
		      (s (read-string "Compile command? " def-cmd)))
		 (if (and (boundp 's)
			  s
			  (not (string= s "")))
		     (setq ans s)))))))
    (java-settings-replace "Compile command:" ans)))
(defun java-settings-mainclass ()
  "Changes the current project's main-class."
  (interactive)
  (let ((ans nil)
	(isFirst t)
	(def-class (ljw-scan-project-name)))
    (if (not (and def-class (not (string= def-class ""))))
	(setq def-class
	      (file-name-sans-extension
	       (file-name-nondirectory buffer-file-name))))
    (while (not ans)
      (let ((s (read-string
	      (if isFirst "Which is the main class? "
		"Please enter a class. Which is the main class? ")
	      def-class)))
	(if (and (boundp 's) s (not (string= s "")))
	    (setq ans s))))
    (java-settings-replace "Main class:" ans)))
(defun java-settings-type ()
  "Changes the current project-type."
  (interactive)
  (let ((ans nil) (isFirst t))
    (while (not ans)
      (if isFirst (message "Applet, Window-app or Console? (a/w/c)")
	(message (concat "Please answer a,w or c. Is it an applet, "
			 "Window-app or Console application?")))
      (let ((c (read-char)))
	(setq isFirst nil)
	(cond ((or (equal c ?a) (equal c ?A)) (setq ans "applet"))
	      ((or (equal c ?w) (equal c ?W)) (setq ans "windowapp"))
	      ((or (equal c ?c) (equal c ?C)) (setq ans "console")))))
    (java-settings-replace "Project type:" ans)))
(defun java-settings-args ()
  "Changes the current project arguments."
  (interactive)
  (let* ((def-args (ljw-scan-project-arguments))
	 (ans (read-string "Arguments? " def-args)))
    (java-settings-replace "Arguments:" ans)))
(defun java-settings-replace (key val)
  "Changes some project setting."
  (interactive "sKey? \nsValue? ")
  (save-excursion
    (goto-char (point-min))
    (let* ((regexp (concat (regexp-quote key) ".*$"))
	   (res (re-search-forward regexp 1024 t)))
      (if res
	  (replace-match (concat key " " val) t t)
	(progn
	  (goto-char (point-min))
	  (insert (concat "// " key " " val "\n")))))))





;;; MENU STUFF
;;  The shape menu is essentially constructed as a constant, ljw-menu-map,
;;  immediately below. There is a further function, ljw-update-menu,
;;  called at various times, which may change bits of the menu and
;;  which assigns the appropriate function to each menu item.
;;  (It was done this way so that the shape of the menu could be
;;  planned in advance).


;; The shape of the menu for xemacs:
(if ljw-running-xemacs
    (progn
      (defvar ljw-xemacs-menu
	'("Java"
	  "----")
	"XEmacs19 menu for LJW.")
      (defvar ljw-xemacs-settings-menu
	'("Settings"
	  "----")
	"Xemacs19 menu for LJW settings."))
  ;; The shape of the menu for regular emacs:
  (progn
    (defvar ljw-menu-map (make-sparse-keymap "Java") nil)
    (defvar ljw-settings-map (make-sparse-keymap "Settings"))
    (define-key ljw-settings-map [args] '("Arguments..." . nil))
    (define-key ljw-settings-map [type] '("Project type..." . nil))
    (define-key ljw-settings-map [mainclass] '("Main class..." . nil))
    (define-key ljw-settings-map [compiler] '("Compiler..." . nil))
    (define-key ljw-menu-map [settings] (cons "Settings" ljw-settings-map))
    (define-key ljw-menu-map [separator3] '("--"))
    (define-key ljw-menu-map [debug-stepover] '("Step over" . nil))
    (define-key ljw-menu-map [debug-stepinto] '("Step into" . nil))
    (define-key ljw-menu-map [breakpoint-clear] '("Clear breakpoint" . nil))
    (define-key ljw-menu-map [breakpoint-set] '("Set breakpoint" . nil))
    (define-key ljw-menu-map [debug-run] '("Go!" . nil))
    (define-key ljw-menu-map [debug] '("Debug" . nil))
    (define-key ljw-menu-map [separator2] '("--"))
    (define-key ljw-menu-map [next] '("Next compile error" . nil))
    (define-key ljw-menu-map [prev] '("Prev compile error" . nil))
    (define-key ljw-menu-map [first] '("First compile error" . nil))
    (define-key ljw-menu-map [separator1] '("--"))
    (define-key ljw-menu-map [run] '("Run proj" . nil))
    (define-key ljw-menu-map [make] '("Make proj" . nil))
    (define-key ljw-menu-map [compile] '("Compile buffer" . nil))))




;; functions to update the menu-bar:
(defun ljw-menu-item (txt key)
  "If this is running under Windows, puts the string of the accelerator
key into the menu manually. I don't know why this is needed."
  (if (eq system-type 'windows-nt)
      (concat txt "  " key)
    txt))
(defun ljw-update-menu ()
  "updates the java menu, and assigns functions to each item."
  (interactive)
  (if ljw-running-xemacs
      ;; THE XEMACS VERSION:
      (progn
	(setq ljw-xemacs-settings-menu
	      (list
	       "Settings"
	       (vector "Compiler..." 'java-settings-compiler 't)
	       (vector "Main class..." 'java-settings-mainclass 't)
	       (vector "Project type..." 'java-settings-type 't)
	       (vector "Arguments..." 'java-settings-args 't)))
	(setq ljw-xemacs-menu
	      (list
	       "Java"
	       (vector "Compile buffer" 'java-compile 't)
	       (vector (concat "Make project " (ljw-scan-project-name))
		       'java-make t)
	       (vector (concat "Execute project " (ljw-scan-project-name))
		       'java-execute 't)
	       "----"
	       (vector "First compile error" 'java-first-error 't)
	       (vector "Prev compile error" 'java-prev-error 't)
	       (vector "Next compile error" 'java-next-error 't)
	       "----"
	       (vector "Debug" 'java-debug 't)
	       (if ljw-is-debug-started
		   (vector "Continue" 'java-cont 't)
		 (vector "Go!" 'java-run 't))
	       (vector "Set breakpoint" 'java-set-breakpoint 't)
	       (vector "Clear breakpoint" 'java-clear-breakpoint 't)
	       (vector "Step into" 'java-step-into 't)
	       (vector "Step over" 'java-step-over 't)
	       "----"
	       ljw-xemacs-settings-menu))
	;; Some keyboard shortcuts:
	(define-key java-mode-map [f5] 'java-compile)
	(define-key java-mode-map [f6] 'java-first-error)
	(define-key java-mode-map [f7] 'java-prev-error)
	(define-key java-mode-map [f8] 'java-next-error)
	(define-key java-mode-map [f9] 'java-execute)
	(define-key java-mode-map [f3] 'java-step-into)
	(define-key java-mode-map [f4] 'java-step-over)
	(define-key java-mode-map "\C-c\C-b" 'java-set-breakpoint)
	(define-key java-mode-map "\C-c\C-e" 'java-clear-breakpoint)
	;; and now install the xeamcs menu!
	(if (and (not (memq 'infodoc c-emacs-features))
		 (boundp 'current-menubar)
		 current-menubar)
	    (if (not (assoc "Java" current-menubar))
		(if (fboundp 'add-submenu)
		    (progn
		      (set-buffer-menubar (copy-sequence current-menubar))
		      (add-submenu nil ljw-xemacs-menu))
		  (add-menu nil "Java" ljw-xemacs-menu)))))
    ;; compilation stuf
    ;; THE REGULAR VERSION:
    (progn
      (define-key ljw-menu-map [compile]
	(cons (ljw-menu-item "Compile buffer" "F5")
	      '(. java-compile)))
      (define-key ljw-menu-map [make]
	(cons (concat "Make project " (ljw-scan-project-name))
	      '(. java-make)))
      (define-key ljw-menu-map [run]
	(cons (ljw-menu-item
	       (concat "Execute project " (ljw-scan-project-name)) "F9")
	      '(. java-execute)))
      ;; debug stuff
      (define-key ljw-menu-map [debug-stepover]
	(cons (ljw-menu-item "Step over" "F4")
	      '(. java-step-over)))
      (define-key ljw-menu-map [debug-stepinto]
	(cons (ljw-menu-item "Step into" "F3")
	      '(. java-step-into)))
      (if ljw-is-debug-started
	  (progn
	    (define-key ljw-menu-map [debug-run] '("Continue" . java-cont)))
	(progn
	  (define-key ljw-menu-map [debug-run] '("Go!" . java-run))))
      (define-key ljw-menu-map [breakpoint-set]
	(cons (ljw-menu-item "Set breakpoint" "C-c C-b")
	      '(. java-set-breakpoint)))
      (define-key ljw-menu-map [breakpoint-clear]
	(cons (ljw-menu-item "Clear breakpoint" "C-c C-e")
	      '(. java-clear-breakpoint)))
      (define-key ljw-menu-map [debug] '("Debug" . java-debug))
      ;; error stuff
      (define-key ljw-menu-map [first]
	(cons (ljw-menu-item "First compile error" "F6")
	      '(. java-first-error)))
      (define-key ljw-menu-map [prev]
	(cons (ljw-menu-item "Prev compile error" "F7")
	      '(. java-prev-error)))
      (define-key ljw-menu-map [next]
	(cons (ljw-menu-item "Next compile error" "F8")
	      '(. java-next-error)))
      ;; settings stuff
      (define-key ljw-settings-map [compiler]
	'("Compiler..." . java-settings-compiler))
      (define-key ljw-settings-map [mainclass]
	'("Main class..." . java-settings-mainclass))
      (define-key ljw-settings-map [type]
	'("Project type..." . java-settings-type))
      (define-key ljw-settings-map [args]
	'("Arguments..." . java-settings-args))
      ;; keyboard shortcuts
      (define-key java-mode-map [f5] 'java-compile)
      (define-key java-mode-map [f6] 'java-first-error)
      (define-key java-mode-map [f7] 'java-prev-error)
      (define-key java-mode-map [f8] 'java-next-error)
      (define-key java-mode-map [f9] 'java-execute)
      (define-key java-mode-map [f3] 'java-step-into)
      (define-key java-mode-map [f4] 'java-step-over)
      (define-key java-mode-map "\C-c\C-b" 'java-set-breakpoint)
      (define-key java-mode-map "\C-c\C-e" 'java-clear-breakpoint)
      ;; and now install the menu bar!
      (define-key java-mode-map [menu-bar java]
	(nconc (list "Java") ljw-menu-map)))))
			       


(defun ljw-java-mode-hook ()
 "LJW's java-mode-hook function. This turns on andersl-java-font-lock
  on-the-fly syntax hiliting. It gets called by the java-mode hook."
 (interactive)
 (cond (window-system
	(if (not (boundp 'ljw-disable-autofont))
	    (progn
	      (if (not ljw-running-xemacs) (require 'andersl-java-font-lock))
	      (turn-on-font-lock)))))
 (ljw-update-menu))
(add-hook 'java-mode-hook 'ljw-java-mode-hook)


(provide 'ljw)

;; ljw.el ends here
