structure install = 
struct
local
   open Microsoft.Win32
   open RegTools

   val options = ["/CLR=","/VS=","/u","/?","/help","/silent","/verbose"]

   val arguments = CommandLine.arguments()

   val uninstall = List.exists (fn "/u" => true | _ => false) arguments
  
   val verbose = List.exists (fn "/verbose" => true | _ => false) arguments

   val silent = List.exists (fn "/silent" => true | _ => false) arguments

   val help = List.exists (fn "/?" => true | "/help" => true | _ => false) arguments 
              orelse 
              List.exists (fn arg => not (List.exists (fn option => (String.isPrefix option arg))  options))
                          arguments


   fun echo msg = if verbose orelse help then (print "\n";print msg) else ()

   val _ = if help then
   (print "\ninstall [OPTIONS]\
          \\nOPTIONS:\
	  \\n/CLR=[v1.0|v1.1|v2.0|<version>] install support for SML.NET using given CLR <version>\
	  \\n/VS=[7.0|7.1|8.0|<version>]     install support for SML.NET in given Visual Studio <version>\
          \\n/u                         uninstall installed support\
          \\n[/?|/help]                 show this message\
          \\n/silent                    do not display dialog box\
          \\n/verbose                   show individual steps\
          \\n\
          \\n NOTES:\
	  \\n 0. Your distribution must reside on a path that does not contain spaces, like C:\\smlnet.\
	  \\n 1. You may (un)install alternative configurations by re-running install.\
	  \\n 2. The install.exe sources are available under vs\\[RegTools.sml,install.sml]\
          \\n    (In directory vs, recompile the installer with smlnet @install.smlnet if you need to modify the installer)\
          \\n 3. If unspecified, install.exe will infer /VS and /CLR values\
          \\n    from your registry settings (the highest /VS version wins).\
          \\n    Currently inferred defaults:")
	else ()

   fun findValue(root,keys,value) =
       if iskey(root,keys) then 
  	 (findkey(root,keys))
	 .#GetValue(SOME value)
       else NONE

   fun findValueNames(root,keys) =
       if iskey(root,keys) then 
  	 (findkey(root,keys))
	 .#GetValueNames()
       else NONE

   fun failwith s = if silent 
	            then
	            (print ("\nerror: "^s); 
                     OS.Process.exit(OS.Process.failure))
	            else
	             (System.Windows.Forms.MessageBox.Show
		         ("Installer failed with error:\n"^s,"Install.exe");
	              OS.Process.exit(OS.Process.failure))

    fun system (cmd:string,args:string) = 
	   let 	
	       val p = System.Diagnostics.Process()
	       val psi = System.Diagnostics.ProcessStartInfo()
	       val () = psi.#set_FileName(cmd) 
	       val () = psi.#set_Arguments(args)
               val () = psi.#set_CreateNoWindow(true) 
               val () = psi.#set_UseShellExecute(false) 
	       val () = p.#set_StartInfo(psi)
	   in
	      ( 
               echo ("executing ["^cmd^" "^args^"]");
	       p.#Start();
	       p.#WaitForExit();
	       p.#Refresh();
               let val status = p.#get_ExitCode() 
	       in
		   p.#Dispose(); 
		   if status = 0 then ()
		   else failwith ("\nerror: command ["^cmd^" "^args^"] failed with exit code " ^ Int.toString status);
		   status
	       end
	       ) handle _ =>
	       (failwith ("\nerror: command ["^cmd^"] not found"); 1)
	   end
 
   (* helpers *)

   fun getValues(option) =  
      List.mapPartial(fn optionVal => 
                         if String.isPrefix option optionVal
                         then SOME(String.extract(optionVal,String.size option,NONE))
	                 else NONE) arguments

   fun getVersion (defaults,option,test,desc) =
	let val values = getValues(option)
            val _ = if List.all test values
	            then ()
                    else failwith ("could not find given version of "^desc^" in registry");	    
            val version = List.foldl
		       (fn (key,default) => 
	                if test key
	                then key
	                else default)
	                ""  
	                (defaults@values)
	in  if version = ""
            then failwith ("could not find any installation of "^desc^" in registry")
            else version 
	end	


   val InstallRoot = 
       case findValue(Registry.LocalMachine,
         	      ["SOFTWARE","Microsoft",".NETFramework"],
	              "InstallRoot")
       of SOME (path :> string) => path
       | _ => failwith ("could not find InstallRoot for .NETFramework in registry")

   val _ = echo ("Guessing .NETFramework InstallRoot="^InstallRoot)

   (* guess VS7.1 or VS7.0, if not specified in command line;
      Note that .NETFramework SDK 1.1 makes an incomplete entry under 7.1 to register the debugger, 
      so we use a more refined test that looks for the InstallDir value
   *)

   val VSMajorVersion = 
     getVersion(["7.0","7.1","8.0"],
                "/VS=",
	        fn key => Option.isSome(findValue(Registry.LocalMachine,["SOFTWARE","Microsoft","VisualStudio",key],"InstallDir")),
               "VS");

   val _ = echo ("Guessing VS Major Version, /VS="^VSMajorVersion)

   (* guess CLR v1.1 of v1.0, if not specified in command line *)
	
   val CLRMajorVersion =
     getVersion(case VSMajorVersion of 
	          "7.0" => ["v1.1","v1.0"]  (* for 7.0 prefer v1.0, but use v1.1 if that's all there is *)
	        | "7.1" => ["v1.0","v1.1"]  (* for 7.1 prefer v1.1, but use v1.0 if that's all there is *)
                | "8.0" => ["v1.0","v1.1","v2.0"]
                | _ => ["v1.0","v1.1","v2.0"], (* otherwise use highest of either *)
                "/CLR=", 
	        fn key => iskey(Registry.LocalMachine,["SOFTWARE","Microsoft",".NETFramework","policy",key])
	        ,"CLR");

   val _ = echo ("Guessing CLR Major Version, /CLR="^CLRMajorVersion)



   val InstallDir =  
       case findValue(Registry.LocalMachine,
  		      ["SOFTWARE","Microsoft","VisualStudio",VSMajorVersion],
                      "InstallDir")
       of SOME (path :> string) => path
        | _ => failwith ("could not find an InstallDir of Visual Studio "^VSMajorVersion^" in registry")

   val _ = echo ("Guessing VS InstallDir="^InstallDir)

   val version = 
       case
	let val names = 
	    case findValueNames(Registry.LocalMachine,["SOFTWARE","Microsoft",".NETFramework","policy",CLRMajorVersion])
            of SOME names => names
            |  NONE => failwith ("could not find values for .NETFramework "^CLRMajorVersion^" in registry")

	    fun  getVer (NONE,v) = NONE
	       | getVer (SOME s,v) = case Int.fromString s of NONE => v | SOME _ => SOME s
	in
	   Array.foldl getVer NONE names
	end
	of SOME version => version
         | _ => failwith ("could not find a minor version for .NETFramework "^CLRMajorVersion^" in registry")

   val _ = echo ("Guessing .NETFramework minor version="^version)

   (* exit prematurely on help *)
   val _ = if help then OS.Process.exit(OS.Process.success) else ()

   fun process(entries)=
	(if uninstall
	 then List.app unregister (List.rev entries)
	 else List.app register entries;
	 OS.Process.success)
   	 handle err => OS.Process.failure


   fun writeFile(dst,s) = 
    let val os = TextIO.openOut dst 
    in
        TextIO.output(os,s);
        TextIO.closeOut(os);
        ()
    end handle _ => (echo ("Failed to write file " ^ dst); OS.Process.exit(OS.Process.failure))

   fun checkFile s = 
	if OS.FileSys.access(s,[]) 
	then s 
	else (echo ("file "^s^" is required but does not exist at this location");
	      OS.Process.exit(OS.Process.failure))

   val smlnetpath = getPath("..")

   val () =  
	(* reject smlnetpath with spaces *)
	if not(List.exists (fn #" " => true | _ => false) (String.explode smlnetpath)) 
	then ()
        else (failwith 
	      ("This distribution of SML.NET lives in a path that contains spaces: "^smlnetpath^
	       "\nThe distribution won't work correctly if installed on this path.\
	       \\nPlease move your distribution to a different path that does *not* contain spaces, like C:\\smlnet,\ 
               \\nand re-run the installer from there.\
 	       \\nSorry for the inconvenience."))


   val ToolsDir = checkFile(OS.Path.mkCanonical(OS.Path.concat(OS.Path.getParent(InstallDir),
                                                               "Tools")))
   val xcopy = checkFile(expand "%SystemRoot%\\System32\\xcopy.exe")
   val regsvr32 = checkFile(expand "%SystemRoot%\\System32\\regsvr32.exe")
   val regasm = checkFile(InstallRoot ^ CLRMajorVersion ^ "." ^ version ^ "\\regasm.exe")
   val ngen = checkFile(InstallRoot ^ CLRMajorVersion^ "." ^ version ^ "\\ngen.exe")
   fun deleteDir s = (  echo ("Deleting directory "^s)
	              ; System.IO.Directory.Delete(SOME s,true))
	             handle _ => echo ("error: could not delete directory "^s)
   fun deleteFile s = (  echo ("Deleting file "^s)
	               ; System.IO.File.Delete(SOME s))
                      handle _ => echo ("error: could not delete file "^s)

   fun quote s = "\"" ^ s ^ "\""

   val installedBabelServiceLib = InstallDir^"PrivateAssemblies\\BabelServiceLib.dll"
   val installedSmlBabelService = InstallDir^"PrivateAssemblies\\SmlBabelService.dll"
   val installedMSVCR70 = InstallDir^"msvcr70.dll" (* C runtime, required by SmlBabelPackage *)

   val vcsubdir = if VSMajorVersion >= "8.0" then "VC" else "VC7"

   val smlnetdotbat = OS.Path.concat(ToolsDir,"smlnet.bat")   
   val smlnetscript = "call "^quote(OS.Path.concat(ToolsDir,"vsvars32.bat"))^ "\n"^
                      "set SMLNETPATH="^smlnetpath^"\n"^
                      quote (smlnetpath^"\\bin\\run.x86-win32.exe")
                      ^" @SMLload="^quote (smlnetpath^"\\bin\\smlnet.x86-win32")
                      ^" %*"

   val smlnetcleandotbat = OS.Path.concat(ToolsDir,"smlnetclean.bat")   
   val smlnetcleanscript = "call "^quote(OS.Path.concat(ToolsDir,"vsvars32.bat"))^ "\n" ^
                           "set SMLNETPATH="^smlnetpath^"\n"^
                           "call "^quote(smlnetpath^"\\bin\\clean.bat")


   fun actions()=
	if uninstall
	then (
                echo ("Deleting "^smlnetdotbat^" and " ^ smlnetcleandotbat)
              ; deleteFile(smlnetdotbat)
              ; deleteFile(smlnetcleandotbat)
	      ; echo "Removing wizard files"
	      ; deleteDir(InstallDir^"..\\..\\"^vcsubdir^"\\VCProjectItems\\SML")
	      ; deleteFile(InstallDir^"..\\..\\"^vcsubdir^"\\VCProjectItems\\SML.vsdir")
	      ; deleteDir(InstallDir^"..\\..\\"^vcsubdir^"\\VCWizards\\SML")
	      ; echo "Unregistering SmlBabelPackage.dll"
	      ; system(regsvr32,"/S /u "^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\SmlBabelPackage.dll"))
              ; echo "Unregistering assemblies SmlBabelService and BabelServiceLib"
              ; system(regasm,"/nologo /silent /unregister "^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\SmlBabelService.dll"))
              ; system(ngen,if VSMajorVersion >= "8.0" then 
                                 "uninstall "^quote(installedSmlBabelService)^ " /nologo /silent"
                            else "/nologo /silent /delete "^quote(installedSmlBabelService))
              ; system(ngen,if VSMajorVersion >= "8.0" then 
                                 "uninstall "^quote(installedBabelServiceLib) ^ " /nologo /silent"
                            else "/nologo /silent /delete "^quote(installedBabelServiceLib))
	      ; deleteFile(installedSmlBabelService)
	      ; deleteFile(installedBabelServiceLib)
	      ; deleteFile(installedMSVCR70)
	      ; OS.Process.success
	    )
	else ( 
                echo ("Creating "^smlnetdotbat^" and " ^ smlnetcleandotbat)
              ; writeFile(smlnetdotbat,smlnetscript)
              ; writeFile(smlnetcleandotbat,smlnetcleanscript)
 	      ; echo "Installing wizards"
	      ; system(xcopy,"/Q "
		         ^" /s /y " 
	                 ^quote(smlnetpath^"\\vs\\wizards\\VC7") ^ " "
	                 ^quote(InstallDir^"..\\..\\" ^ vcsubdir))
	      ; echo "Registering SmlBabelPackage.dll with vs"
	      ; system(regsvr32,"/S "^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\SmlBabelPackage.dll"))
              ; echo "Registering assemblies SmlBabelService and BabelServiceLib"
	      ; system(xcopy,"/Q "
		         ^" /s /y " 
	                 ^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\BabelServiceLib.dll") ^ " "
	                 ^quote(InstallDir^"PrivateAssemblies"))
	      ; system(xcopy,"/Q "
		         ^" /s /y " 
	                 ^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\SmlBabelService.dll") ^ " "
	                 ^quote(InstallDir^"PrivateAssemblies"))
	      ; system(xcopy,"/Q "
		         ^" /s /y " 
	                 ^quote(smlnetpath^"\\vs\\babel\\sml\\bin\\msvcr70.dll") ^ " "
	                 ^quote(InstallDir))
              ; system(regasm,"/nologo /silent "^quote(installedSmlBabelService))
              ; echo "Generating native images ..."
              ; system(ngen,if VSMajorVersion >= "8.0" then 
                                 "install "^quote(installedBabelServiceLib) ^ " /nologo /silent "
                            else "/nologo /silent "^quote(installedBabelServiceLib))
              ; system(ngen,if VSMajorVersion >= "8.0" then 
                                 "install "^quote(installedSmlBabelService) ^ " /AppBase:"^quote(InstallDir^"PrivateAssemblies") ^ " /nologo /silent"
                            else "/nologo /silent "^quote(installedSmlBabelService))
	      ; OS.Process.success
	    )

   val SMLEntries =
		[ 
	         (Registry.LocalMachine,
	          CREATE,
		  ["SOFTWARE","SML.NET"],
	           [("SMLNETPATH",STRING smlnetpath)
                   ]
	         )
	        ]

   fun vsroot path = ["SOFTWARE","Microsoft","VisualStudio",VSMajorVersion]@path

   val wizardEntries = 	
		[
	        (Registry.LocalMachine,
	          CREATE,
		  vsroot ["NewProjectTemplates","TemplateDirs","{4137FF4A-F557-42f5-BDAA-E0A736D01E68}"],
	           [
	           ]
	         ),
	         (Registry.LocalMachine,
	          ADD,
		  vsroot ["NewProjectTemplates","TemplateDirs","{4137FF4A-F557-42f5-BDAA-E0A736D01E68}","/1"],
	           [("",STRING "SML.NET Projects"),
	            ("TemplatesDir",STRING (smlnetpath^"\\vs\\wizards\\VS"^VSMajorVersion)),
	            ("SortPriority",DWORD 0x00000014)
	            (* to place in a virtual subfolder, add this value 
	             , ("Folder",
	                 STRING "{DCF2A94A-45B0-11D1-ADBF-00C04FB6BE4C}") 
	            *)
	           ]
	         )
	        ]

   val babelEntries = 
    let 
    val BabelPackageGUID = (* "{221F0EB7-30F7-45FF-AE73-5968B167CF09}"  *)
	                   "{2F5B21F8-4AC2-41C0-BA47-76C4CC441517}"
    val SMLGUID = "{822C7D3E-2ABC-3F13-97E3-1CCFB8A6C9CB}"
    in
        [(Registry.LocalMachine,
	  CREATE,
          vsroot ["Languages","Language Services","SML"],
            [("",STRING SMLGUID),
             ("Package",STRING BabelPackageGUID),
             ("Extensions",STRING ".sml;.fun;.sig;"),
             ("LangResId",DWORD 0x00000000),
             ("RequestStockColors", DWORD 0x00000001),
             ("ShowCompletion", DWORD 0x00000001),
             ("ThreadModel", DWORD 0x00000001),
             ("MatchBraces", DWORD 0x00000001),
             ("QuickInfo", DWORD 0x00000001),
             ("CodeSense", DWORD 0x00000001),
             ("CodeSenseDelay", DWORD 0x000005dc),
             ("CodeSenseFastOnChangeLine", DWORD 0x00000000),
             ("MaxErrorMessages", DWORD 0x00000005),
             ("SortMemberList", DWORD 0x00000001),
             ("ShowMatchingBrace", DWORD 0x00000001),
             ("MatchBracesAtCaret", DWORD 0x00000001)]),
         (Registry.LocalMachine,
	  CREATE,
          vsroot ["Languages","File Extensions",".sml"],
          [("",STRING SMLGUID),
           ("Name",STRING "SML")]),
         (Registry.LocalMachine,
	  CREATE,
          vsroot ["Languages","File Extensions",".fun"],
          [("", STRING SMLGUID),
           ("Name",STRING "SML")]),
         (Registry.LocalMachine,	
	  CREATE,
          vsroot ["Languages","File Extensions",".sig"],
          [("",STRING SMLGUID),
           ("Name",STRING "SML")]),
         (Registry.LocalMachine,
	  CREATE,
          vsroot ["Services",SMLGUID],
          [("",STRING BabelPackageGUID),
           ("Name",STRING "SML")])
        ]
   end	

   val entries = SMLEntries @ wizardEntries @ babelEntries

   fun dialog() = if silent 
	           then OS.Process.success
                   else
                    ((if uninstall
                      then 
	                 System.Windows.Forms.MessageBox.Show
		         ("Uninstalled support for SML.NET in Visual Studio.NET "^VSMajorVersion,"Install.exe")
	              else 
                         System.Windows.Forms.MessageBox.Show
		         ("Installed support for SML.NET in Visual Studio.NET "^VSMajorVersion,"Install.exe"));
	              OS.Process.success)
                    
in

   fun main() = 
	(
 	 actions();
	 echo "Modifying registry settings";
	 process entries;
	 dialog()
	)
end
end









