package skywriting.examples.wordcount;
/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

public class WritableUtils {

	/**
	 * Serializes an integer to a binary stream with zero-compressed encoding.
	 * For -120 <= i <= 127, only one byte is used with the actual value.
	 * For other values of i, the first byte value indicates whether the
	 * integer is positive or negative, and the number of bytes that follow.
	 * If the first byte value v is between -121 and -124, the following integer
	 * is positive, with number of bytes that follow are -(v+120).
	 * If the first byte value v is between -125 and -128, the following integer
	 * is negative, with number of bytes that follow are -(v+124). Bytes are
	 * stored in the high-non-zero-byte-first order.
	 *
	 * @param stream Binary output stream
	 * @param i Integer to be serialized
	 * @throws java.io.IOException 
	 */
		public static void writeVInt(DataOutput stream, int i) throws IOException {
			writeVLong(stream, i);
		}

		/**
		 * Serializes a long to a binary stream with zero-compressed encoding.
		 * For -112 <= i <= 127, only one byte is used with the actual value.
		 * For other values of i, the first byte value indicates whether the
		 * long is positive or negative, and the number of bytes that follow.
		 * If the first byte value v is between -113 and -120, the following long
		 * is positive, with number of bytes that follow are -(v+112).
		 * If the first byte value v is between -121 and -128, the following long
		 * is negative, with number of bytes that follow are -(v+120). Bytes are
		 * stored in the high-non-zero-byte-first order.
		 * 
		 * @param stream Binary output stream
		 * @param i Long to be serialized
		 * @throws java.io.IOException 
		 */
		public static void writeVLong(DataOutput stream, long i) throws IOException {
			if (i >= -112 && i <= 127) {
				stream.writeByte((byte)i);
				return;
			}

			int len = -112;
			if (i < 0) {
				i ^= -1L; // take one's complement'
				len = -120;
			}

			long tmp = i;
			while (tmp != 0) {
				tmp = tmp >> 8;
			len--;
			}

			stream.writeByte((byte)len);

			len = (len < -120) ? -(len + 120) : -(len + 112);

			for (int idx = len; idx != 0; idx--) {
				int shiftbits = (idx - 1) * 8;
				long mask = 0xFFL << shiftbits;
				stream.writeByte((byte)((i & mask) >> shiftbits));
			}
		}
		
		  /**
		   * Reads a zero-compressed encoded long from input stream and returns it.
		   * @param stream Binary input stream
		   * @throws java.io.IOException 
		   * @return deserialized long from stream.
		   */
		  public static long readVLong(DataInput stream) throws IOException {
		    byte firstByte = stream.readByte();
		    int len = decodeVIntSize(firstByte);
		    if (len == 1) {
		      return firstByte;
		    }
		    long i = 0;
		    for (int idx = 0; idx < len-1; idx++) {
		      byte b = stream.readByte();
		      i = i << 8;
		      i = i | (b & 0xFF);
		    }
		    return (isNegativeVInt(firstByte) ? (i ^ -1L) : i);
		  }
		  
		  /**
		   * Parse the first byte of a vint/vlong to determine the number of bytes
		   * @param value the first byte of the vint/vlong
		   * @return the total number of bytes (1 to 9)
		   */
		  public static int decodeVIntSize(byte value) {
		    if (value >= -112) {
		      return 1;
		    } else if (value < -120) {
		      return -119 - value;
		    }
		    return -111 - value;
		  }
		  
		  public static boolean isNegativeVInt(byte value) {
			    return value < -120 || (value >= -112 && value < 0);
			  }

		
		  /**
		   * Reads a zero-compressed encoded integer from input stream and returns it.
		   * @param stream Binary input stream
		   * @throws java.io.IOException 
		   * @return deserialized integer from stream.
		   */
		  public static int readVInt(DataInput stream) throws IOException {
		    return (int) readVLong(stream);
		  }
		  /**
		   * Skip <i>len</i> number of bytes in input stream<i>in</i>
		   * @param in input stream
		   * @param len number of bytes to skip
		   * @throws IOException when skipped less number of bytes
		   */
		  public static void skipFully(DataInput in, int len) throws IOException {
		    int total = 0;
		    int cur = 0;

		    while ((total<len) && ((cur = in.skipBytes(len-total)) > 0)) {
		        total += cur;
		    }

		    if (total<len) {
		      throw new IOException("Not able to skip " + len + " bytes, possibly " +
		                            "due to end of input.");
		    }
		  }
		  
	}
