/*
 * sqlscriptexport.{cc,hh} -- export data records to a SQL script
 * 
 * Marco Canini
 *
 * Copyright (c) 2008-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: sqlscriptexport.cc 2213 2009-05-06 18:03:35Z marco $
 */

#include <click/config.h>
#include <click/error.hh>
#include <click/string.hh>
#include <click/confparse.hh>

#include <iostream>
#include <fstream>
#include <memory>

#include "sqlscriptexport.hh"
#include "fileexport.hh"

CLICK_DECLS

SQLScriptExport::SQLScriptExport(): initialized_(false), os_(0)
{}

SQLScriptExport::~SQLScriptExport()
{
    delete os_;
}

void* SQLScriptExport::cast(const char* n)
{
    if (strcmp(n, "SQLScriptExport") == 0)
        return static_cast<SQLScriptExport*>(this);
    else if (strcmp(n, "DataExport") == 0)
        return static_cast<DataExport*>(this);
    else
        return Element::cast(n);
}

int SQLScriptExport::configure(Vector<String>& conf, ErrorHandler* errh)
{
    Element* next = 0;
    drop_table_ = false;
    
    if (cp_va_kparse(conf, this, errh,
            "FILENAME", cpkP+cpkM, cpFilename, &filename_,
            "TABLENAME", cpkM, cpString, &table_name_,
            "DROP_TABLE", 0, cpBool, &drop_table_,
            "NEXT", cpkM, cpElement, &next,
            cpEnd) < 0)
        return -1;
    
    if (next && !(next_ = static_cast<DataExport*>(static_cast<FileExport*>(next->cast("FileExport")))))
        return errh->error("NEXT must be a FileExport element");

    os_ = new std::ofstream(filename_.c_str());

    if (!os_)
        return errh->error("Unable to open file %s", filename_.c_str());

    return 0;
}

int SQLScriptExport::initialize(ErrorHandler*)
{
    data_filename_ = (static_cast<FileExport*>(next_))->filename(); 
    return 0;
}

void SQLScriptExport::add_column(const Column& col)
{
    if (initialized_)
        ErrorHandler::default_handler()->fatal("Cannot add a Column when SQLScriptExport is already initialized");
    
    cols_.push_back(col);
}

void SQLScriptExport::add_constraint(const Constraint& con)
{
    if (initialized_)
        ErrorHandler::default_handler()->fatal("Cannot add a Constraint when SQLScriptExport is already initialized");
    
    cons_.push_back(con);
}

void SQLScriptExport::write(const String&)
{}

void SQLScriptExport::write_null()
{}

void SQLScriptExport::done()
{
    if (!initialized_)
    {
        create_sql_strings();
        *os_ << table_sql_.c_str() << "\n";
        *os_ << load_sql_.c_str() << "\n";
        os_->close();
        initialized_ = true;
    }
}

void SQLScriptExport::create_sql_strings()
{
    StringAccum sa_c;  // table creation string
    StringAccum sa_ld; // load data string
    
    if (drop_table_)
        sa_c << "DROP TABLE IF EXISTS " << table_name_ << ";\n";
    
    sa_c << "CREATE TABLE " << table_name_ << " (\n";
    
    typedef Vector<DataExport::Column>::const_iterator Iter;
    typedef Vector<DataExport::Constraint>::const_iterator CnIter;
    int idx = 0;
    
    String kStr; // Name of primary key column. NOTE: primary key can be made of only one column!
    int kIdx = 0;
    
    for (CnIter i = cons_.begin(); i != cons_.end(); ++i)
    {
        if ((*i).type == DataExport::Constraint::PRIMARY_KEY)
        {
            kStr = (*i).name;
            break;
        }
    }
    
    for (Iter i = cols_.begin(); i != cols_.end(); ++i)
    {
        String name = normalize_name(i->name); // normalized column name
        String type = get_type(*i); // column type
        
        if (type == "")
            ErrorHandler::default_handler()->fatal("Column %s has empty type", i->name.c_str());

        if (name == kStr)
            kIdx = idx;
        
        sa_c << name << " " << type;
        if (!i->nullable)
            sa_c << " NOT NULL";
        sa_c << ",\n";

        ++idx;
    }

    for (CnIter i = cons_.begin(); i != cons_.end(); ++i)
    {
        sa_c << get_constraint(*i) << ",\n";
    }
    sa_c.pop_back(2);
    sa_c << ");";
    
    sa_ld << "LOAD DATA LOCAL INFILE '" << data_filename_ << "' INTO TABLE " << table_name_;
    sa_ld << " FIELDS TERMINATED BY ' ' IGNORE 1 LINES;";

    table_sql_ = sa_c.take_string();
    
    load_sql_ = sa_ld.take_string();
}

String SQLScriptExport::get_type(const DataExport::Column& col)
{
    const std::type_info& id = *col.type;
    std::size_t size = col.size;

    String type;

    if (id == typeid(bool))
    {
        type = "boolean";
    }
    else if (id == typeid(char) ||
            id == typeid(short) ||
            id == typeid(int) ||
            id == typeid(long) ||
            id == typeid(long long) || id == typeid(unsigned long long))
    {
        if (size <= 8) /* FIXME: the mapping should be DB dependent */
            type = "tinyint";
        else if (size <= 16)
            type = "smallint";
        else if (size <= 32)
            type = "int";
        else
            type = "bigint";
    }
    else if (id == typeid(unsigned char) ||
            id == typeid(unsigned short) ||
            id == typeid(unsigned int) ||
            id == typeid(unsigned long))
    {
        if (size <= 8) /* FIXME: the mapping should be DB dependent */
            type = "tinyint unsigned";
        else if (size <= 16)
            type = "smallint unsigned";
        else if (size <= 32)
            type = "int unsigned";
        else
            type = "bigint unsigned";
    }
    else if (id == typeid(String) || id == typeid(std::string))
    {
        /* FIXME: how to have char instead of varchar? */
        if (size > 0)
        {
            StringAccum sa;
            sa << "varchar(" << size << ")";
            type = sa.take_string();
        }
        else
            type = "text";
    }
    else if (id == typeid(Timestamp))
    {
        type = "double precision";
    }
    else if (id == typeid(float))
    {
        StringAccum sa;
        sa << "float(" << size << ")";
        type = sa.take_string();
    }
    else if (id == typeid(double))
    {
        type = "double precision";
    }
    else if (id == typeid(DataExport::Column::EnumType))
    {
        StringAccum sa;
        sa << "enum(";
        for (DataExport::Column::EnumType::const_iterator it = col.enum_type.begin();
            it != col.enum_type.end(); ++it)
        {
            /* FIXME: needs escaping */
            sa << '\'' << *it << "',";
        }
        sa.pop_back(1);
        sa << ')';
        type = sa.take_string();
    }

    return type;
}

String SQLScriptExport::get_constraint(const DataExport::Constraint& con)
{
    StringAccum sa;
    switch (con.type) {
    case DataExport::Constraint::PRIMARY_KEY:
        sa << "PRIMARY KEY(" << con.name << ')';
        break;
    case DataExport::Constraint::KEY:
        sa << "INDEX(" << con.name << ')';
        break;
    case DataExport::Constraint::UNIQUE_KEY:
        sa << "UNIQUE KEY(" << con.name << ')';
        break;
    default:
        break;
    }
    return sa.take_string();
}

bool SQLScriptExport::is_string(const std::type_info* ti)
{
    return *ti == typeid(String) || *ti == typeid(std::string) ||
        *ti == typeid(DataExport::Column::EnumType);
}

String SQLScriptExport::normalize_name(const String& name)
{
    int len = name.length();
    StringAccum sa;
    
    for (int i=0; i<len; ++i)
    {
        char c = name[i];
        
        if (c != '[' && c != ']')
            sa << c;
        else
            sa << '_';
    }
    
    return sa.take_string();
}

ELEMENT_REQUIRES(userlevel)
EXPORT_ELEMENT(SQLScriptExport)

#include <click/vector.cc>

CLICK_ENDDECLS

