/*
 * sqlexport.{cc,hh} -- export data records to a SQL database
 * 
 * Sergio Mangialardi
 * Marco Canini
 *
 * Copyright (c) 2008-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: sqlexport.cc 2208 2009-05-06 17:59:25Z marco $
 */

#include <click/config.h>
#include <click/error.hh>
#include <click/string.hh>
#include <click/confparse.hh>

#include <iostream>
#include <memory>

#include "sqlexport.hh"

CLICK_DECLS

SQLExport::SQLExport(): initialized_(false), keep_connection_(false),
    append_to_table_(false), log(new FileErrorHandler(stderr, "SQLExport::"))
{}

SQLExport::~SQLExport()
{
    delete log;
}

void* SQLExport::cast(const char* n)
{
    if (strcmp(n, "SQLExport") == 0)
        return static_cast<SQLExport*>(this);
    else if (strcmp(n, "DataExport") == 0)
        return static_cast<DataExport*>(this);
    else
        return Element::cast(n);
}

int SQLExport::configure(Vector<String>& conf, ErrorHandler* errh)
{
    Element* next = 0;
    
    if (cp_va_kparse(conf, this, errh,
            "DRIVER", cpkP+cpkM, cpString, &driver_,
            "URL", cpkP+cpkM, cpString, &url_,
            "TABLENAME", 0, cpString, &table_name_,
            "KEEPCONNECTION", 0, cpBool, &keep_connection_,
            "APPENDTOTABLE", 0, cpBool, &append_to_table_,
            "NEXT", 0, cpElement, &next,
            cpEnd) < 0)
        return -1;
    
    if (driver_ == "")
        return errh->error("Empty DRIVER string");
    
    if (url_ == "")
        return errh->error("Empty URL string");
    
    if (next && !(next_ = static_cast<DataExport*>(next->cast("DataExport"))))
        return errh->error("NEXT must be a DataExport element");
    
    return 0;
}

int SQLExport::initialize(ErrorHandler* errh)
{
    log->message("starting initialization");

    try
    {
        connection_.connect(driver_.c_str(), url_.c_str());
    }
    catch (SQL::SQLException& ex)
    {
        return errh->fatal(ex.what());
    }

    if (!keep_connection_)
        connection_.close();
    
    log->message("initialization success");

    return 0;
}

void SQLExport::add_column(const Column& col)
{
    if (initialized_)
        ErrorHandler::default_handler()->fatal("Cannot add a Column when SQLExport is already initialized");
    
    cols_.push_back(col);
}

void SQLExport::add_constraint(const Constraint& con)
{
    if (initialized_)
        ErrorHandler::default_handler()->fatal("Cannot add a Constraint when SQLExport is already initialized");
    
    cons_.push_back(con);
}

void SQLExport::write(const String& val)
{
    if (!initialized_)
        ErrorHandler::default_handler()->fatal("Cannot write a value when SQLExport is not initialized yet");
    
    vals_.push_back(Value(val));
}

void SQLExport::write_null()
{
    if (!initialized_)
        ErrorHandler::default_handler()->fatal("Cannot write a value when SQLExport is not initialized yet");

    vals_.push_back(Value());
}

void SQLExport::done()
{
    if (!initialized_)
    {
        vals_.reserve(cols_.size());
        create_sql_strings();
        create_table();
        initialized_ = true;
    }
    else
    {
        insert_data();
    }
}

void SQLExport::create_table()
{
    try
    {
        if (!connection_.connected())
            connection_.connect(driver_.c_str(), url_.c_str());

        log->debug(table_sql_.c_str());

        if (append_to_table_)
        {
            SQL::NoThrowSQLCommand cmd(connection_, table_sql_.c_str());
            cmd.execute();
        }
        else
        {
            SQL::SQLCommand cmd(connection_, table_sql_.c_str());
            cmd.execute();
        }
    }
    catch (SQL::SQLException& ex)
    {
        ErrorHandler::default_handler()->fatal("SQLExport::create_table: %s", ex.what());
    }
    
    if (!keep_connection_)
        connection_.close();
}

void SQLExport::insert_data()
{
    insert();
    vals_.clear();

    if (!keep_connection_)
        connection_.close();  
}
    
void SQLExport::insert()
{
    String sql = add_values(insert_sql_);
    
    try
    {
        if (!connection_.connected())
            connection_.connect(driver_.c_str(), url_.c_str());
    }
    catch (SQL::SQLException& ex)
    {
        ErrorHandler::default_handler()->fatal("SQLExport::insert: %s", ex.what());
    }
    
    SQL::NoThrowSQLCommand cmd(connection_, sql.c_str());
    cmd.execute();
    
    if (!cmd.success())
        update();
}

void SQLExport::update()
{
    String sql = add_values(update_sql_);
    
    try
    {
        if (!connection_.connected())
            connection_.connect(driver_.c_str(), url_.c_str());

        SQL::SQLCommand cmd(connection_, sql.c_str());
        cmd.execute();
    }
    catch (SQL::SQLException& ex)
    {
        ErrorHandler::default_handler()->error("SQLExport::update: %s", ex.what());
    }
}

void SQLExport::create_sql_strings()
{
    StringAccum sa_c;  // create table query
    StringAccum sa_i;  // insert query
    StringAccum sa_i1; // column names in insert query
    StringAccum sa_i2; // column values in insert query
    StringAccum sa_u;  // update query
    
    sa_c << "CREATE TABLE " << table_name_ << " (\n";
    sa_i << "INSERT INTO " << table_name_ << "(";
    sa_u << "UPDATE " << table_name_ << " SET\n";
    
    typedef Vector<DataExport::Column>::const_iterator Iter;
    typedef Vector<DataExport::Constraint>::const_iterator CnIter;
    int idx = 0;
    
    String kStr; // Name of primary key column. NOTE: primary key can be made of only one column!
    int kIdx = 0;
    
    for (CnIter i = cons_.begin(); i != cons_.end(); ++i)
    {
        if ((*i).type == DataExport::Constraint::PRIMARY_KEY)
        {
            kStr = (*i).name;
            break;
        }
    }
    
    for (Iter i = cols_.begin(); i != cols_.end(); ++i)
    {
        String name = normalize_name(i->name); // normalized column name
        String type = get_type(*i); // column type
        
        if (type == "")
            ErrorHandler::default_handler()->fatal("Column %s has empty type", i->name.c_str());

        if (name == kStr)
            kIdx = idx;
        
        sa_c << name << " " << type;
        if (!i->nullable)
            sa_c << " NOT NULL";
        sa_c << ",\n";

        sa_i1 << name << ", ";
        
        sa_i2 << '%' << idx << ",\n";

        sa_u << name << " = " << '%' << idx << ",\n";
 
        ++idx;
    }

    sa_i1.pop_back(2);
    sa_i1 << ')';

    sa_i2.pop_back(2);
    sa_i2 << ");\n";

    sa_u.pop_back(2);

    sa_u << "\nWHERE " << kStr << " = " << '%' << kIdx << ';'; 
    sa_i << sa_i1 << "\nVALUES\n(" << sa_i2;
    
    for (CnIter i = cons_.begin(); i != cons_.end(); ++i)
    {
        sa_c << get_constraint(*i) << ",\n";
    }
    sa_c.pop_back(2);
    sa_c << ");";

    table_sql_ = sa_c.take_string();
    insert_sql_ = sa_i.take_string();
    update_sql_ = sa_u.take_string();
}

String SQLExport::add_values(const String& sql)
{
    StringAccum sa;

    int len = sql.length();
    
    for (int i=0; i<len; ++i)
    {
        char c = sql[i];
        const char* start = sql.c_str() + i + 1;
        char* end;
        
        if (c == '%')
        {
            int idx = strtol(start, &end, 10);
            const DataExport::Column& col = cols_[idx];
            if (vals_[idx].null || !is_string(col.type))
            {
                sa << vals_[idx].val;
            } else {
                sa << '\'';
                /* FIXME: needs to escape vals_[idx].val according to driver */
                sa << vals_[idx].val;
                sa << '\'';
            }
            i += (end - start);
        }
        else
            sa << c;
    }
    
    return sa.take_string();
}

String SQLExport::get_type(const DataExport::Column& col)
{
    const std::type_info& id = *col.type;
    std::size_t size = col.size;

    String type;

    if (id == typeid(bool))
    {
        type = "boolean";
    }
    else if (id == typeid(char) ||
            id == typeid(short) ||
            id == typeid(int) ||
            id == typeid(long) ||
            id == typeid(long long) || id == typeid(unsigned long long))
    {
        if (size <= 8) /* FIXME: the mapping should be DB dependent */
            type = "tinyint";
        else if (size <= 16)
            type = "smallint";
        else if (size <= 32)
            type = "int";
        else
            type = "bigint";
    }
    else if (id == typeid(unsigned char) ||
            id == typeid(unsigned short) ||
            id == typeid(unsigned int) ||
            id == typeid(unsigned long))
    {
        if (size <= 8) /* FIXME: the mapping should be DB dependent */
            type = "tinyint unsigned";
        else if (size <= 16)
            type = "smallint unsigned";
        else if (size <= 32)
            type = "int unsigned";
        else
            type = "bigint unsigned";
    }
    else if (id == typeid(String) || id == typeid(std::string))
    {
        /* FIXME: how to have char instead of varchar? */
        if (size > 0)
        {
            StringAccum sa;
            sa << "varchar(" << size << ")";
            type = sa.take_string();
        }
        else
            type = "text";
    }
    else if (id == typeid(Timestamp))
    {
        type = "double precision";
    }
    else if (id == typeid(float))
    {
        StringAccum sa;
        sa << "float(" << size << ")";
        type = sa.take_string();
    }
    else if (id == typeid(double))
    {
        type = "double precision";
    }
    else if (id == typeid(DataExport::Column::EnumType))
    {
        StringAccum sa;
        sa << "enum(";
        for (DataExport::Column::EnumType::const_iterator it = col.enum_type.begin();
            it != col.enum_type.end(); ++it)
        {
            /* FIXME: needs escaping */
            sa << '\'' << *it << "',";
        }
        sa.pop_back(1);
        sa << ')';
        type = sa.take_string();
    }

    return type;
}

String SQLExport::get_constraint(const DataExport::Constraint& con)
{
    StringAccum sa;
    switch (con.type) {
    case DataExport::Constraint::PRIMARY_KEY:
        sa << "PRIMARY KEY(" << con.name << ')';
        break;
    case DataExport::Constraint::KEY:
        sa << "INDEX(" << con.name << ')';
        break;
    case DataExport::Constraint::UNIQUE_KEY:
        sa << "UNIQUE KEY(" << con.name << ')';
        break;
    default:
        break;
    }
    return sa.take_string();
}

bool SQLExport::is_string(const std::type_info* ti)
{
    return *ti == typeid(String) || *ti == typeid(std::string) ||
        *ti == typeid(DataExport::Column::EnumType);
}

String SQLExport::normalize_name(const String& name)
{
    int len = name.length();
    StringAccum sa;
    
    for (int i = 0; i<len; ++i)
    {
        char c = name[i];
        
        if (c != '[' && c != ']')
            sa << c;
        else
            sa << '_';
    }
    
    return sa.take_string();
}

ELEMENT_REQUIRES(userlevel)
EXPORT_ELEMENT(SQLExport)

#include <click/vector.cc>

CLICK_ENDDECLS

