#ifndef CLICK_SQL_HH
#define CLICK_SQL_HH

#include <string>
#include <stdexcept>
#include <memory>
#include <map>

namespace SQL
{
    class RegEntry;
    
    struct SQLConnectionImpl
    {
        virtual ~SQLConnectionImpl() {}
        virtual void close() = 0;
        virtual bool execute_query(const std::string& url, bool throw_on_error) = 0;
    };
        
    typedef SQLConnectionImpl* (*FactoryMethod)(const std::string&);
    
    class Driver
    {
    public:
        static Driver get(const std::string& n)
        {
            return Driver(n);
        }
        
        SQLConnectionImpl* new_instance(const std::string& url);
    private:
        std::string name;
        typedef std::map<std::string, FactoryMethod> FactoryMethodMap;
        static FactoryMethodMap map_;
        friend class RegEntry;
        static void register_class(const std::string& n, FactoryMethod m);
        
        Driver(const std::string& s): name(s) {} 
    };
        
    struct RegEntry 
    {
        RegEntry(const char s[], FactoryMethod f) 
        { 
            Driver::register_class(s, f);
        }
    };

    template <class T, const char s[]> class Registered
    {
    public:
        static SQLConnectionImpl* new_instance(const std::string& url) 
        {
            return new T(url);
        }
    protected:
        Registered()
        {
            const RegEntry& dummy = r;
            (void) dummy;
        }
    private:
        static const RegEntry r;
    };

    template <class T, const char s[]> const RegEntry Registered<T, s>::r(s, Registered<T, s>::new_instance);
    
    #define DRIVER(C, N) \
    char C##Name[] = N; \
    class C: public SQLConnectionImpl, public Registered<C, C##Name>
    
    class SQLException: public std::runtime_error
    {
    public:
        explicit SQLException(const std::string& message): std::runtime_error(message) {}
    };
    
    class SQLConnection
    {
        friend class SQLCommand;
        friend class NoThrowSQLCommand;
    public:
        SQLConnection() {}
        SQLConnection(const std::string& driver, const std::string& url);
        ~SQLConnection();
        void connect(const std::string& driver, const std::string& url);
        void close();
        bool connected() const { return impl_.get(); }
    private:
        SQLConnection(const SQLConnection&);
        SQLConnection& operator=(const SQLConnection&);
        void parse(const std::string& driver, const std::string& url);
        bool execute_query(const std::string& url, bool throw_on_error = true) const;
    private:
        std::auto_ptr<SQLConnectionImpl> impl_;
    };
    
    class SQLCommand
    {
    public:
        explicit SQLCommand(const SQLConnection& conn, const std::string& sql): conn_(conn), sql_(sql) {}
        ~SQLCommand() {}
        void execute();
    private:
        const SQLConnection& conn_;
        std::string sql_;
    };
    
    class NoThrowSQLCommand
    {
    public:
        explicit NoThrowSQLCommand(const SQLConnection& conn, const std::string& sql): conn_(conn), sql_(sql), 
            success_(false) {}
        ~NoThrowSQLCommand() {}
        void execute();
        bool success() { return success_; }
    private:
        const SQLConnection& conn_;
        std::string sql_;
        bool success_;
    };
};

#endif

