/*
 * sql.{cc,hh} -- SQL database connection
 *
 * Marco Canini
 * Sergio Mangialardi
 *
 * Copyright (c) 2007-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: sql.cc 2208 2009-05-06 17:59:25Z marco $
 */

#include <click/config.h>

#include "sql.hh"

CLICK_DECLS

SQL::Driver::FactoryMethodMap SQL::Driver::map_;

SQL::SQLConnectionImpl* SQL::Driver::new_instance(const std::string& url)
{
    FactoryMethodMap::iterator it = map_.find(name);
    
    if(it != map_.end())
        return it->second(url);
    else
        return 0;
}

void SQL::Driver::register_class(const std::string& s, FactoryMethod m)
{
    map_.insert(std::pair<std::string, FactoryMethod>(s, m));
}

SQL::SQLConnection::SQLConnection(const std::string& driver, const std::string& url)
{
    connect(driver, url);
}

SQL::SQLConnection::~SQLConnection()
{
    close();
}

void SQL::SQLConnection::connect(const std::string& driver, const std::string& url)
{
    if (impl_.get())
        throw SQLException("SQL::SQLConnection::connect: already connected.");
    
    parse(driver, url);
}

void SQL::SQLConnection::close()
{
    impl_.reset();
}

void SQL::SQLConnection::parse(const std::string& driver, const std::string& url)
{
    impl_.reset(Driver::get(driver).new_instance(url));
    
    if (!impl_.get())
    {
        std::string msg = "SQL::SQLConnection::connect: \"";
        msg.append(driver).append("\" inexistent database driver.");
        
        throw SQLException(msg);
    }
}

bool SQL::SQLConnection::execute_query(const std::string& sql, bool throw_on_error) const
{
    return impl_->execute_query(sql, throw_on_error);
}

void SQL::SQLCommand::execute()
{
    conn_.execute_query(sql_);
}

void SQL::NoThrowSQLCommand::execute()
{
    success_ = conn_.execute_query(sql_, false);
}

ELEMENT_REQUIRES(userlevel)
ELEMENT_PROVIDES(SQL)

CLICK_ENDDECLS

