#ifndef CLICK_L7_HH
#define CLICK_L7_HH

#include <click/element.hh>
#include <click/ipflowid.hh>
#include <clicknet/tcp.h>

/* from l7-classify */
#include <string>
#include <vector>
#include <sys/types.h>
#include <regex.h>

#include "flowcache.hh"
#include "appmarks.hh"

CLICK_DECLS

/*
=c

L7([I<KEYWORDS>])

=s tnt

classifies TCP/UDP flows using the l7-filter engine

=d

L7 classifies TCP/UDP flows using the l7-filter engine.
Please see http://l7-filter.sf.net for l7-filter documentation.

Keywords are:

=over 8

=item PATTERNS

A space separate string containing the list of pattern names to be used by L7.

=item MAX_PKTS

The maximum number of packets which are used to search for the configured
patterns. Default 10.

=item MAX_BYTES

The maximum number of payload bytes which are used to search for the
configured patterns. Default 2048.

=item PATTERNS_DIR

The directory containing the l7-filter patterns (downloadable from
http://l7-filter.sf.net)

=item USE_APPMARKS

Boolean. Whether the output is an APPMARK value or the l7-filter's pattern
name.

=item FLOWCACHE

Element. A FlowCache element that provides state tracking of TCP/UDP flows.
Default is the upstream FlowCache element.

=a

FlowCache,AppMarks
*/

class L7: public Element, FlowCache::FlowStateHolder
{
public:
    L7(): buffer_count_(10), byte_count_(2048), location_("/etc/l7-protocols"), max_pkt_match_(0) {}

    const char* class_name() const { return "L7"; }
    const char* port_count() const { return PORTS_1_1; }
    const char* processing() const { return AGNOSTIC;  }

    int configure(Vector<String>& conf, ErrorHandler* errh);
    int initialize(ErrorHandler* errh);
    void add_handlers();

    Packet* simple_action(Packet* p);

    void write_header(DataExport& exporter) const;
    void write_flow_state(DataExport& exporter, const BaseFlowState* fs_) const;

    class FlowState;
    
    class Pattern
    {
    public:
        Pattern() {}
        Pattern(const std::string& filename);
        bool matches(char* buffer);
        std::string get_name() const { return name_; }
        AppMark get_mark() const { return mark_; }
    private:
        bool parse_pattern_file(const std::string& filename);
        std::string pre_process(const std::string& pattern);
        bool parse_flags(const std::string& line);
    private:
        std::string name_;
        AppMark mark_;
        std::string pattern_;
        int cflags_; // for regcomp
        int eflags_; // for regexec
        regex_t preg_;//the compiled regex
    };
private:
    bool add_pattern_from_file(const std::string& filename);
    static String read_max_pkt_match(Element*, void*);
    AppMark classify(char* buffer);
private:
    Vector<String> patternNames_;
    std::vector<Pattern> patterns_;
    uint32_t buffer_count_;
    uint32_t byte_count_;
    String filename_;
    String location_;
    uint32_t max_pkt_match_;
    FlowCache* flow_cache_;
    bool use_appmarks_;

    friend class FlowState;
};

class L7::FlowState: public BaseFlowState, public AppMark
{
public:
    uint32_t packet_counter[2];
    uint32_t lengthsofar[2];
    char* buffer[2];

    FlowState();
    ~FlowState();

    void handle_packet(Packet*, L7*);
    void append_to_buffer(const char* app_data, int appdatalen, L7*, char* buffer, uint32_t& lengthsofar);
};

CLICK_ENDDECLS
#endif

