package it.unige.dist.tnt.gtvs;

import it.unige.dist.tnt.sql.ConnectionFactory;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

public final class GtvsResolve implements ResolverListener {

    Logger log = Logger.getLogger(GtvsResolve.class.getName());
	
    @Option(name="-h",usage="MySQL hostname")
    private String hostname = "localhost";
    
    @Option(name="-u",usage="MySQL username")
    private String username = "root";
    
    @Option(name="-p",usage="MySQL password")
    private String password = "";
    
    @Option(name="-d",usage="MySQL database")
    private String dbname = "GTVS";

    @Argument
    private List<String> arguments = new ArrayList<String>();
    
    private final static int NUM_THREADS = 30;
    private final static int NUM_HOSTS = 3000;
	
    private Connection conn;
    private String curTable;
    private long resolved;
	
    private GtvsResolve() {
    }
	
    public void run(String[] args) {
        CmdLineParser parser = new CmdLineParser(this);
        String table = null;

        try {
            // parse the arguments.
            parser.parseArgument(args);

            if(arguments.isEmpty() )
                throw new CmdLineException("Expecting table name");
            table = arguments.get(0);
        } catch( CmdLineException e ) {
            // if there's a problem in the command line,
            // you'll get this exception. this will report
            // an error message.
            System.err.println(e.getMessage());
            System.err.println("java " + this.getClass().getName() + " [options...] <table name>");
            // print the list of available options
            parser.printUsage(System.err);
            System.err.println();

            System.exit(1);
        }


        try {
            conn = ConnectionFactory.getMySQLConnection(hostname, dbname, username, password);
        } catch (SQLException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
            System.exit(1);
        }

        try {
            log.info("Will resolve IPs for table " + table + ".");

            long start = System.currentTimeMillis();

            resolveTable(table);

            long time = (System.currentTimeMillis() - start) / 1000;
            log.info("Resolved " + resolved + " IPs in " + time + " seconds.");
        } catch (SQLException e) {
            e.printStackTrace();
            System.exit(1);
        }

    }

    private void resolveTable(String table) throws SQLException {
        curTable = table;

        Statement stmt = conn.createStatement();
        ResultSet rs = stmt.executeQuery("SELECT COUNT(*) FROM " + table + " WHERE HostName IS NULL");
        rs.next();

        int hosts = rs.getInt(1);
        rs.close();
        log.info("Will attempt to resolve " + hosts + " addresses.");

        ThreadPoolExecutor tpe = new ThreadPoolExecutor(NUM_THREADS, NUM_THREADS, 0L, TimeUnit.MILLISECONDS, new SynchronousQueue<Runnable>(), new ThreadPoolExecutor.CallerRunsPolicy());
        tpe.prestartAllCoreThreads();

        for (int i = 0; i < hosts; i += NUM_HOSTS)
        {
            //log.info("SELECT INET_NTOA(Ip), HostName FROM " + table + " LIMIT " + i + ", " + NUM_HOSTS);
            rs = stmt.executeQuery("SELECT INET_NTOA(Ip), HostName FROM " + table + " LIMIT " + i + ", " + NUM_HOSTS);
            while (rs.next()) {
                if (rs.getString(2) == null) {
                    String address = rs.getString(1);
                    tpe.submit(new DnsJavaResolver(address, this));
                }
            }
            rs.close();
        }

        tpe.shutdown();
    }

    public void notifyResolution(String address, String hostname) {
        Statement stmt;
        try {
            stmt = conn.createStatement();
            stmt.execute("UPDATE " + curTable + " SET HostName = '" + hostname +
                    "' WHERE Ip = INET_ATON('" + address + "')");
            incrementResolved();
        } catch (SQLException e) {
            e.printStackTrace();
        }
    }

    private synchronized void incrementResolved() {
        resolved += 1;
    }

    /**
     * @param args
     */
    public static void main(String[] args) {
        new GtvsResolve().run(args);

    }

}
