/*
 * concaterf.c -- concatenates network traces in ERF format
 * Marco Canini
 *
 * Copyright (c) 2007 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id$
 */

#define _LARGEFILE_SOURCE
#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/param.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#define VERSION "0.1"

typedef struct erf_record {
	uint64_t	ts;
	uint8_t		type;
	uint8_t		flags;
	uint16_t	rlen;
	uint16_t	lctr;
	uint16_t	wlen;
	uint8_t		data[0];
} erf_record_t;

void erf_copy(FILE *trace, FILE *dest, int is_first_trace)
{
	erf_record_t rec;
	size_t sz;
	size_t rec_len;
	uint8_t buf[65520];
	int is_first_rec = 1;
	static uint64_t last_ts = 0;
	while (!feof(trace)) {
		sz = fread(&rec, sizeof(rec), 1, trace);
		if (sz != 1) {
			if (feof(trace))
				break;
			fprintf(stderr, "concaterf: read %d got %d\n", sizeof(rec), sz);
			goto error_read;
		}
		rec_len = ntohs(rec.rlen);
		sz = rec_len - sizeof(rec);
		assert(sz <= sizeof(buf));
		sz = fread(buf, sizeof(uint8_t), sz, trace);
		if (sz != rec_len - sizeof(rec)) {
			fprintf(stderr, "concaterf: read %d got %d\n", rec_len - sizeof(rec), sz);
			goto error_read;
		}
		if (is_first_rec) {
			is_first_rec = 0;
			if (is_first_trace == 0)
				continue;
		} else if (last_ts != 0) {
			if (rec.ts < last_ts) {
				static int warned = 0;
				if (!warned) {
					fprintf(stderr, "\
concaterf: warning, packets are out of order.\n");
				}
				warned = 1;
			}
		}
		last_ts = rec.ts;
		sz = fwrite(&rec, sizeof(rec), 1, dest);
		if (sz != 1)
			goto error_write;
		sz = fwrite(buf, sizeof(uint8_t), rec_len - sizeof(rec), dest);
		if (sz != rec_len - sizeof(rec))
			goto error_write;
	}
	return;
error_read:
	fprintf(stderr, "concaterf: cannot read from file. Truncated trace?\n");
	exit(1);
error_write:
	fprintf(stderr, "concaterf: cannot write to file. Out of disk space?\n");
	exit(1);
}

int main(int argc, char *argv[])
{
	char erffile[PATH_MAX+1];
	char *debug;
	int is_first_trace = 1;
	
	fprintf(stderr, "concaterf " VERSION "\n");
	debug = getenv("CONCATERF_DEBUG");
	if (debug != NULL) {
		fprintf(stderr, "waiting for debugger (%d)\n", getpid());
		sleep(atoi(debug));
	}

	while (fgets(erffile, PATH_MAX, stdin) != NULL) {
		int len = strlen(erffile);
		FILE *trace = NULL;
		int trace_is_pipe = 0;

		if (erffile[len - 1] == '\n') {
			erffile[len - 1] = '\0';
			len--;
		}
		if (debug != NULL)
			fprintf(stderr, "processing file %s\n", erffile);

		if (len > 3 && strcmp(erffile + len - 3, ".gz") == 0) {
			char cmd[16+len];
			sprintf(cmd, "gzip -d -c \"%s\"", erffile);
			trace = popen(cmd, "r");
			trace_is_pipe = 1;
		} else if (len > 4 && strcmp(erffile + len - 4, ".bz2") == 0) {
			char cmd[16+len];
			sprintf(cmd, "bzip2 -d -c \"%s\"", erffile);
			trace = popen(cmd, "r");
			trace_is_pipe = 1;
		} else {
			trace = fopen(erffile, "rb");
		}
		if (trace == NULL) {
			perror("concaterf");
			exit(1);
		}

		erf_copy(trace, stdout, is_first_trace);

		if (trace_is_pipe) {
			pclose(trace);
		} else {
			fclose(trace);
		}
		is_first_trace = 0;
	}
}

