/*
 * concatcap.c -- concatenates tcpdump traces in pcap format
 * Marco Canini
 *
 * Copyright (c) 2007 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id$
 */

#define _LARGEFILE_SOURCE
#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/param.h>
#include <string.h>
#include <stdlib.h>
#include <pcap.h>

#define VERSION "0.3"

#define ZERO_TIME(ptv)(((ptv)->tv_sec == 0) && ((ptv)->tv_usec == 0))

void write_packet(u_char *out, const struct pcap_pkthdr *pkthdr, const u_char *buf)
{
	static struct timeval last_ts = {0, 0};
	/* check for re-ordered packets */
	if (!ZERO_TIME(&last_ts)) {
		if (pkthdr->ts.tv_sec <= last_ts.tv_sec) {
			if (pkthdr->ts.tv_sec < last_ts.tv_sec ||
			    pkthdr->ts.tv_usec < last_ts.tv_usec) {
				static int warned = 0;
				if (!warned) {
					fprintf(stderr, "\
concatcap: warning, packets are out of order.\n");
				}
				warned = 1;
			}
		}
	}
	last_ts = pkthdr->ts;
	pcap_dump(out, pkthdr, buf);
}

int main(int argc, char *argv[])
{
	char pcapfile[PATH_MAX+1];
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_dumper_t *out = NULL;
	char *debug;
	
	fprintf(stderr, "concatcap " VERSION "\n");
	debug = getenv("CONCATCAP_DEBUG");
	if (debug != NULL) {
		fprintf(stderr, "waiting for debugger (%d)\n", getpid());
		sleep(atoi(debug));
	}

	while (fgets(pcapfile, PATH_MAX, stdin) != NULL) {
		pcap_t *pcap;
		int len = strlen(pcapfile);
		FILE *trace = NULL;

		if (pcapfile[len - 1] == '\n') {
			pcapfile[len - 1] = '\0';
			len--;
		}
		if (debug != NULL)
			fprintf(stderr, "processing file %s\n", pcapfile);

		if (len > 3 && strcmp(pcapfile + len - 3, ".gz") == 0) {
			char cmd[16+len];
			sprintf(cmd, "gzip -d -c \"%s\"", pcapfile);
			trace = popen(cmd, "r");
			pcap = pcap_fopen_offline(trace, errbuf);
		} else if (len > 4 && strcmp(pcapfile + len - 4, ".bz2") == 0) {
			char cmd[16+len];
			sprintf(cmd, "bzip2 -d -c \"%s\"", pcapfile);
			trace = popen(cmd, "r");
			pcap = pcap_fopen_offline(trace, errbuf);
		} else {
			pcap = pcap_open_offline(pcapfile, errbuf);
		}
		if (pcap == NULL) {
			fprintf(stderr, "concatcap: %s\n", errbuf);
			exit(1);
		}

		if (out == NULL)
			out = pcap_dump_open(pcap, "-");
		
		if (pcap_loop(pcap, -1, write_packet, (u_char *) out) < 0) {
			fprintf(stderr, "concatcap: %s\n", errbuf);
			exit(1);
		}
		pcap_close(pcap);
	}
	pcap_dump_close(out);
}

