require 'mysql'
require 'getoptlong'
require 'highline'
require 'ipaddr'
require 'net/http'
require 'uri'

require 'gtvs-config'

configcheck = {
 GTVS::Config::PCAP_IDX => File.executable?(GTVS::Config::PCAP_IDX),
 GTVS::Config::TCPDUMP => File.executable?(GTVS::Config::TCPDUMP),
 GTVS::Config::TCPTRACE => File.executable?(GTVS::Config::TCPTRACE)
}

if configcheck.values.index(false) != nil
    $stderr.puts "It appears you have some configuration problems:"
    configcheck.each {|k,v|
        $stderr.puts "#{k} is not executable" if v == false
    }
    exit(1)
end


class IPAddr
    def self.new_i(ip)
        ip = [ip].pack('N')
        return self.new_ntoh(ip)
    end
end

module GTVS

class Options

    attr_reader :host, :database, :dry_run

    def initialize
        @host = Config::DB_HOST
        @user = Config::DB_USER
        @password = Config::DB_PASSWORD
        @database = Config::DB_NAME
        @ask_password = false
        @dry_run = false
    end

    def set_options(opts)
        opts.on("-D", "--database DATABASE", "Database to use.") {|arg|
            @database = arg
        }

        opts.on("-h", "--host HOSTNAME", "Connect to database on this host.") {|arg|
            @host = arg
        }

        opts.on("-p", "--password [PASSWORD]", "Password to use when connecting to server.", "If password is not given it's asked from the tty.") {|arg|
            @password = arg
            @ask_password = true if arg == nil
        }

        opts.on("-u", "--user [USERNAME]", "User for login if not current user.") {|arg|
            @user = arg
        }
        
        opts.on("-n", "--dry-run", "Do not update the database.") {
            @dry_run = true
        }
    end

    def user
        if @user == nil
            @user = `/usr/bin/whoami`.strip
        end
        return @user
    end

    def password
        if @ask_password
            hg = HighLine.new($stdin, $stderr)
            @password = hg.ask('Database password: ') { |q| q.echo = false }
            @ask_password = false
            puts
        end
        return @password
    end

end

class Connection
    def self.get(options)
        db = Mysql.new(options.host, options.user,
            options.password, options.database)
        $stderr.puts 'Connected to database ' + options.database;
        return db
    end
end

class GTVS

    attr_reader :db, :trace, :flows, :flowsAggByDstIp
    attr_reader :flowsAggByDstPortIp, :hostNames
    attr_reader :httpTransactions

    def initialize(db, trace, opts)
        @db = db
        @gtvsUrl = Config::URL
        @gtvsUrl += '/' if @gtvsUrl[-1,1] != '/'
        @gtvsUser = Config::USER
        @gtvsPassword = Config::PASSWORD
        @dry_run = opts.dry_run

        @trace = trace
        @flows = trace + '_Flows'
        @flowsAggByDstIp = trace + '_FlowsAggByDstIp'
        @flowsAggByDstPortIp = trace + '_FlowsAggByDstIpPort'
        @httpTransactions = trace + '_HttpTransactions'
        @hostNames = findHostNamesTable(trace)
        $stderr.puts "Using HostNames table: #{@hostNames}"
        @datapath = nil
    end

    def verify(flowId, gtProto, gtApp)
        #$stderr.puts("aggVerify #{@trace} #{aggId} #{gtProto} #{gtApp}")
        data = {'data[gtProto]'=>gtProto, 'data[gtApp]'=>gtApp}
        post("traces/verify/#{@trace}/#{flowId}", data) unless @dry_run
    end

    def aggVerify(aggId, gtProto, gtApp)
        #$stderr.puts("aggVerify #{@trace} #{aggId} #{gtProto} #{gtApp}")
        data = {'data[gtProto]'=>gtProto, 'data[gtApp]'=>gtApp}
        post("traces/aggVerify/#{@trace}/#{aggId}", data) unless @dry_run
    end

    def datapath
        return @datapath if @datapath != nil
        sql = <<-SQL
SELECT DataPath FROM Traces WHERE Name = '#{@trace}'
        SQL
        @datapath = nil
        @db.query(sql) {|rs|
            @datapath = (rs.fetch_row)[0]
        }
        raise "Undefined datapath." unless @datapath
        return @datapath
    end

    def read_flow(flowId, count = -1)
        require 'pcap'

        sql = <<-SQL
SELECT INET_NTOA(SrcIp), SrcPort, INET_NTOA(DstIp), DstPort, IpProto,
ROUND(FirstPktTs, 6), ROUND(LastPktTs, 6)
FROM #{@flows}
WHERE FlowId = #{flowId}
        SQL
        @db.query(sql) {|rs|
            raise "Undefined flow #{flowId}." if rs.num_rows == 0
            srcIp, srcPort, dstIp, dstPort, ipProto, firstPktTs, lastPktTs = rs.fetch_row
            filter = <<-FILTER
#{ipProto} and ((src port #{srcPort} and dst port #{dstPort}) or
(src port #{dstPort} and dst port #{srcPort}))
            FILTER
            fullcmd = "#{Config::PCAP_IDX} query #{datapath} #{srcIp} #{dstIp} #{ipProto} start #{firstPktTs} end #{lastPktTs}"
            p = IO.popen(fullcmd, 'r')
            $stdin.reopen(p)
            pcap = Pcap::Capture.open_offline('-')
            pcap.setfilter(filter)
            pcap.each_packet(count) {|pkt|
                yield pkt
            }
            pcap.close
            p.close
        }
    end

    def get_flow_count(dstIp, dstPort, l7Mark)
        sql = <<SQL
SELECT COUNT(*)
FROM `#{@flows}` AS `Flow`
WHERE DstIp = #{dstIp}
AND DstPort = #{dstPort}
AND L7Mark = '#{l7Mark}'
SQL
        count = 0
        @db.query(sql) {|rs|
            count = (rs.fetch_row)[0].to_i
        }
        return count
    end

private

    def post(path, form_data)
        url = URI.parse(@gtvsUrl + path)
        req = Net::HTTP::Post.new(url.path)
        req.basic_auth @gtvsUser, @gtvsPassword if @gtvsUser != nil
        req.set_form_data(form_data)
        res = Net::HTTP.new(url.host, url.port).start {|http|
            http.read_timeout = 1200
            http.request(req)
        }
        case res
        when Net::HTTPSuccess, Net::HTTPRedirection
            #puts res.body
        else
            res.error!
        end
    end

    def findHostNamesTable(trace)
        hostNames = trace + '_HostNames'
        found = false
        while !found
            begin
                @db.query("DESCRIBE #{hostNames}") {|rs|
                    found = true
                }
            rescue Mysql::Error
                abort($!.error) if $!.errno != 1146 # Table doesn't exist
                if trace =~ /(.+\d\d\d\d\d\d\d\d)s(\d.+)$/
                    hostNames = $1 + '_HostNames'
                else
                    abort("Missing HostNames table!")
                end
            end
        end
        return hostNames
    end

end

end # module GTVS

