#ifndef CLICK_NETFPGA_HH
#define CLICK_NETFPGA_HH

#include <click/element.hh>
#include <click/error.hh>

#include <unistd.h>

extern "C" {
#include "nf2.h"
#include "nf2util.h"
#include "reg_defines_hostfilter.h"
}

CLICK_DECLS

#define CMD_REMOVE 0x0
#define CMD_INSERT 0x1
#define CMD_RESET 0x2

#define DEFAULT_IFACE   "nf2c0"


class NetFPGA : public Element {
public:

    enum {PHY0 = 0, SW0 = 1, PHY1 = 2, SW1 = 3, PHY2 = 4, SW2 = 5, PHY3 = 6, SW3 = 7};
    static const char * interfaces[];

    NetFPGA();
    ~NetFPGA();
    const char *class_name() const { return "NetFPGA"; }

    inline bool insertIpPort(uint32_t ipaddr, uint32_t l4port, uint8_t protocol, uint8_t appclass)
    {
        appclass = 3;
        //printf("insertIpPort %lx, %d, %d\n", ipaddr, ntohs(l4port), protocol);
        writeReg(&nf2, HOST_WRITE_REG, 0);
        writeReg(&nf2, HOST_KEY0_REG, ntohl(ipaddr));
        writeReg(&nf2, HOST_KEY1_REG, (0x0000FFFF & ntohs(l4port)) | ((protocol << 16) & 0x00FF0000 ));
        writeReg(&nf2, HOST_CMD_REG, ((appclass << 16) & 0x000F0000) | CMD_INSERT);
        writeReg(&nf2, HOST_WRITE_REG, 1);
        writeReg(&nf2, HOST_WRITE_REG, 0);

        return true;
    }

    inline bool removeIpPort(uint32_t ipaddr, uint32_t l4port, uint8_t protocol)
    {
        writeReg(&nf2, HOST_WRITE_REG, 0);
        writeReg(&nf2, HOST_KEY0_REG, ntohl(ipaddr));
        writeReg(&nf2, HOST_KEY1_REG, (0x0000FFFF & ntohs(l4port)) | ((protocol << 16) & 0x00FF0000 ));
        writeReg(&nf2, HOST_CMD_REG, CMD_REMOVE);
        writeReg(&nf2, HOST_WRITE_REG, 1);
        writeReg(&nf2, HOST_WRITE_REG, 0);

        return true;
    }

#define htonll(x) (((uint64_t)(htonl((uint32_t)((x << 32) >> 32))) << 32) | (uint64_t)htonl(((uint32_t)(x >> 32))))

    inline bool insertFlow(uint32_t host0, uint32_t host1, uint32_t port0, uint32_t port1, uint8_t protocol, uint8_t appclass)
    {
        writeReg(&nf2, FILTER_WRITE_REG, 0);
        writeReg(&nf2, FILTER_KEY0_REG, ntohl(host1));
        writeReg(&nf2, FILTER_KEY1_REG, ntohl(host0));
        writeReg(&nf2, FILTER_KEY2_REG, ((protocol<<16) & 0x00FF0000));
        writeReg(&nf2, FILTER_KEY3_REG, ( (ntohs(port0)<<16) | ntohs(port1) ));
        writeReg(&nf2, FILTER_CMD_REG, ((appclass << 16) & 0x000F0000) | CMD_INSERT);
        writeReg(&nf2, FILTER_WRITE_REG, 1);
        writeReg(&nf2, FILTER_WRITE_REG, 0);

        return true;
    }

    inline bool resetFlow(uint32_t host0, uint32_t host1, uint32_t port0, uint32_t port1, uint8_t protocol)
    {
        writeReg(&nf2, FILTER_WRITE_REG, 0);
        writeReg(&nf2, FILTER_KEY0_REG, ntohl(host1));
        writeReg(&nf2, FILTER_KEY1_REG, ntohl(host0));
        writeReg(&nf2, FILTER_KEY2_REG, ((protocol<<16) & 0x00FF0000));
        writeReg(&nf2, FILTER_KEY3_REG, ( (ntohs(port0)<<16) | ntohs(port1) ));
        writeReg(&nf2, FILTER_CMD_REG, CMD_RESET);
        writeReg(&nf2, FILTER_WRITE_REG, 1);
        writeReg(&nf2, FILTER_WRITE_REG, 0);

        return true;
    }

    inline bool removeFlow(uint32_t host0, uint32_t host1, uint32_t port0, uint32_t port1, uint8_t protocol)
    {
        writeReg(&nf2, FILTER_WRITE_REG, 0);
        writeReg(&nf2, FILTER_KEY0_REG, ntohl(host1));
        writeReg(&nf2, FILTER_KEY1_REG, ntohl(host0));
        writeReg(&nf2, FILTER_KEY2_REG, ((protocol<<16) & 0x00FF0000));
        writeReg(&nf2, FILTER_KEY3_REG, ( (ntohs(port0)<<16) | ntohs(port1) ));
        writeReg(&nf2, FILTER_CMD_REG, CMD_REMOVE);
        writeReg(&nf2, FILTER_WRITE_REG, 1);
        writeReg(&nf2, FILTER_WRITE_REG, 0);


        return true;
    }

    bool resetHC () {

        // Stop the pipeline
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 0);
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 1);

        // Here should be a 1 ms wait
        usleep(1000);

        // Reset HC
        writeReg(&nf2, HOST_RESET_REG, 0);
        writeReg(&nf2, HOST_RESET_REG, 1);
        writeReg(&nf2, HOST_RESET_REG, 0);
        
        // Here should be a 1 ms wait
        usleep(1000);

        // Enable pipeline
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 0);

        return true;
    }
    bool resetSFC () {

        // Stop the pipeline
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 0);
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 1);

        // Here should be a 1 ms wait
        usleep(1000);

        // Reset HC
        writeReg(&nf2, FILTER_RESET_REG, 0);
        writeReg(&nf2, FILTER_RESET_REG, 1);
        writeReg(&nf2, FILTER_RESET_REG, 0);
        
        // Here should be a 1 ms wait
        usleep(1000);

        // Enable pipeline
        writeReg(&nf2, DROP_PIPELINE_STOP_REG, 0);


        return true;
    }

    bool setCutOffThreshold(uint32_t cutoff) {
        if (cutoff<7) {
            writeReg(&nf2, FILTER_CUTOFF_REG, cutoff);
            return true;
        } else {
            return false;
        }
    }

    bool classMapping(uint32_t appclass, uint32_t route)
    {
        uint32_t lastrouting;
        uint32_t routing;
        routing = (0xF & route) << (appclass*8);
        //lastrouting = (~(0xF << (applicationclass*4))) & readReg(&nf2, AR_SW_REG);
        readReg(&nf2, AR_SW_REG, &lastrouting);
        lastrouting &= (~(0xF << (appclass*8)));
        writeReg(&nf2, AR_SW_REG, lastrouting | routing);
        return true;
    }

    static NetFPGA* instance()
    {
        if (NetFPGA::_instance == 0)
            ErrorHandler::default_handler()->fatal("NetFPGA was not initialized! "
                "Did you insert it in the config file?");
        return NetFPGA::_instance;
    }

private:

    uint64_t compute_hash (uint32_t host0, uint32_t host1, uint32_t port0, uint32_t port1, uint32_t protocol) {
        uint64_t hash = 0;
        uint64_t word0 = 0;
        uint64_t word1 = 0;
        uint64_t word2 = 0;
        uint64_t word3 = 0;
        int i;

        for (i = 0; i < 4; i++) {
            word0 |= (( (host0>> (8 * i))  + (host1>> (8 * i)) ) & 0xFF) << (8 * i);
        }


        for (i = 0; i < 2; i++) {
            word1 |= (( (host1>> (8 * 2 * i))  + (host1>> (8 * (2 * i + 1))) ) & 0xFF) << (8 * i);
        }

        for (i = 2; i < 4; i++) {
            word1 |= (( (host0>> (8 * 2 * i))  + (host0>> (8 * (2 * i + 1))) ) & 0xFF) << (8 * i);
        }

        word2 =
            ( ( (port0 + protocol) & 0xFFFF) << 16) |
            (    port1             & 0xFFFF);


        word3 =
            ( ( (port0 + (port0 >> 8) ) & 0xFF) << 24) |
            ( ( (port1 + (port1 >> 8) ) & 0xFF) << 16) |
            (   (protocol                   & 0xFF) << 8);

        hash =
            ( (0xFFFFFFFF & (word3 + word1) ) << 32) +
            (  0xFFFFFFFF & (word0 + word2) );

        return hash;
    }

    struct nf2device nf2;

    static NetFPGA* _instance;
};

CLICK_ENDDECLS
#endif

