/*
 * multistagefilter.hh -- a parallel multi-stage filter implementation
 *                        see New Directions in Traffic Measurement and Accounting
 *                        C. Estan et al. - SIGCOMM 2002
 *
 * Marco Canini
 *
 * Copyright (c) 2008-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: multistagefilter.hh 2578 2009-10-05 10:32:45Z marco $
 */

#ifndef MULTISTAGEFILTER_HH_
#define MULTISTAGEFILTER_HH_

#include <stdio.h>
#include <stdlib.h> /* rand() */
#include <assert.h>
#include <vector>
#include <set>

#include "hashing.hh"

template <class T>
class MultiStageFilter
{
public:

    typedef Hashing<T>* hashing_type;

    MultiStageFilter(size_t d_, size_t b_, int th_, hashing_type hashing_) :
        d(d_), b(b_), th(th_), h(hashing_)
    {
        assert(h->size() == d);
        stages = new int*[d];
        for (size_t i = 0; i < d; ++i) {
            stages[i] = new int[b];
        }
        clear();
    }
    
    virtual ~MultiStageFilter()
    {
        delete h;
        
        for (size_t i = 0; i < d; ++i)
            delete[] stages[i];
        delete[] stages;
    }

    virtual bool update(const T& item);

    virtual void clear()
    {
        for (size_t i = 0; i < d; ++i) {
            memset(stages[i], 0, sizeof(int) * b);
        }
    }

    const hashing_type hashing() const { return h; }
    void set_hashing(hashing_type hashing_)
    {
        assert(h->size() == d);
        delete h;
        h = hashing_;
    }

protected:
    size_t d; // number of stages
    size_t b; // number of buckets
    int th; // filter threshold
    int **stages;
    hashing_type h;
};

template <class T>
bool MultiStageFilter<T>::update(const T& item)
{
    size_t above_th = 0;
    std::vector<size_t> idxs = h->hash(item, b);
    for (size_t i = 0; i < d; ++i)
    {
        int *stage = stages[i];
        size_t idx = idxs[i];
        if (stage[idx] < th) {
            ++stage[idx];
        } else {
            ++above_th;
        }
    }
    return above_th == d;
}

#endif /*MULTISTAGEFILTER_HH_*/
