#ifndef CLICK_HOSTCACHE_HH
#define CLICK_HOSTCACHE_HH

#include <click/element.hh>
#include <click/hashtable.hh>
#include <click/dequeue.hh>
#include <click/ipaddress.hh>
#include <clicknet/tcp.h>

#include "hash.hh"
#include "appmark.hh"
#include "flowcache.hh"
#include "flowclassifier.hh"
#include "multistagefilter.hh"
#include "appmarks.hh"
#include "netfpga.hh"

CLICK_DECLS

/*
 * HostCache
 */

class HostCache: public Element, public FlowClassifier::Listener
{
public:
    HostCache();
    ~HostCache();

    const char* class_name() const { return "HostCache"; }
    const char* port_count() const { return "1/2"; }
    const char* processing() const { return "a/ah"; }
    int configure(Vector<String>& conf, ErrorHandler* errh);
    int initialize(ErrorHandler* errh);

    void push(int, Packet*);
    Packet* pull(int);

private:

    enum
    {
        ACT_PASS, ACT_CLASSIFY, ACT_NONE
    };

    int handle_packet(Packet* p);
    void flow_classified(const FlowCache::Flow& flow, const AppMark& mark);
    
    void update();

    class RuleKey
    {
    public:
    };

    class ServiceSubTuple
    {
    public:
        ServiceSubTuple(uint32_t srv_ip, uint16_t srv_port, uint8_t ip_proto) : _srv_ip(srv_ip),
            _srv_port(srv_port), _ip_proto(ip_proto) {}
        ServiceSubTuple(const FlowCache::Flow& flow) : _srv_ip(flow.dst_ip()), _srv_port(flow.dst_port()),
            _ip_proto(flow.ip_proto()) {}
        
        inline uint32_t srv_ip() const { return _srv_ip; }
        inline uint16_t srv_port() const { return _srv_port; }
        inline uint8_t ip_proto() const { return _ip_proto; }
        
        inline hashcode_t hashcode() const
        {
            return hashword(_srv_ip, 0, _srv_port, _ip_proto);
        }

        inline bool operator==(const ServiceSubTuple &b) const
        {
            return srv_port() == b.srv_port() && srv_ip() == b.srv_ip() && ip_proto() == b.ip_proto();
        }

        inline bool operator!=(const ServiceSubTuple &b) const
        {
            return srv_port() != b.srv_port() || srv_ip() != b.srv_ip() || ip_proto() != b.ip_proto();
        }

        struct Hash1
        {
            size_t operator()(const ServiceSubTuple &o, unsigned long initval) const
            {
                return static_cast<size_t>(hashword(o.srv_ip(), o.srv_port(), o.ip_proto(), initval));
            }
        };

        struct Hash2
        {
            size_t operator()(const ServiceSubTuple &o, unsigned long initval) const
            {
                return static_cast<size_t>(hashword(o.srv_port(), o.srv_ip(), o.ip_proto(), initval));
            }
        };
        
    private:
        uint32_t _srv_ip;
        uint16_t _srv_port;
        uint8_t _ip_proto;
    };

    class ServiceKey
    {
    public:
        ServiceKey(uint32_t srv_ip, uint16_t srv_port) : _srv_ip(srv_ip), _srv_port(srv_port) {}
        ServiceKey(const ServiceSubTuple& st) : _srv_ip(st.srv_ip()), _srv_port(st.srv_port()) {}
        
        inline uint32_t srv_ip() const { return _srv_ip; }
        inline uint16_t srv_port() const { return _srv_port; }
        
        inline hashcode_t hashcode() const
        {
            return hashword(_srv_ip, 0, _srv_port, 0);
        }

        inline bool operator==(const ServiceKey &b) const
        {
            return srv_port() == b.srv_port() && srv_ip() == b.srv_ip();
        }

        inline bool operator!=(const ServiceKey &b) const
        {
            return srv_port() != b.srv_port() || srv_ip() != b.srv_ip();
        }

        struct Hash1
        {
            size_t operator()(const ServiceKey &o, unsigned long initval) const
            {
                return static_cast<size_t>(hashword(o.srv_ip(), 0, o.srv_port(), initval));
            }
        };

        struct Hash2
        {
            size_t operator()(const ServiceKey &o, unsigned long initval) const
            {
                return static_cast<size_t>(hashword(o.srv_port(), 0, o.srv_ip(), initval));
            }
        };
        
    private:
        uint32_t _srv_ip;
        uint16_t _srv_port;
    };
    
    class ServiceRule
    {
    public:
        ServiceRule() : _count(0), _utility(0) {}
        ServiceRule(const AppMark &mark, int utility) : _mark(mark), _count(0), _utility(utility) {}
        void apply(Packet *p, int dir)
        {
            SET_APPMARK_ANNO(p, _mark.get_mark());
            // count refers to the number of distinct client IP, port pairs
            // for a quick implementation we use the SYN flag to count that information
            // for TCP we assume that we always see every packet
            if (dir == 0)
            {
                const click_ip* iph = p->ip_header();
                if (iph->ip_p == IP_PROTO_TCP && p->tcp_header()->th_flags & TH_SYN)
                    ++_count;
            }
        }
        void reset() { _count = 0; }
        AppMark mark() const { return _mark; }
        int count() const { return _count; }
        int utility() const { return _utility; }
    private:
        AppMark _mark;
        int _count;
        int _utility;
    };
    
    typedef Pair<ServiceSubTuple, ServiceRule> Service;
    
    class PacketClassifier
    {
    public:
        virtual ~PacketClassifier() {}
        virtual bool classify(Packet *) = 0;
    };
    
    class ServicePacketClassifier : public PacketClassifier
    {
    public:
        ServicePacketClassifier()
        {
            /*ServiceKey k1(IPAddress("150.146.206.11").addr(), htons(80));
            ServiceRule r1(AppMarks::default_instance()->lookup_by_class_name("WWW"), 100);
            _tcp_rules[k1] = r1;*/
        }
        bool classify(Packet *);
        void insert(const ServiceSubTuple& st, const ServiceRule& sr);
        void remove(const ServiceSubTuple& st);
    private:
        typedef HashTable<ServiceKey, ServiceRule> Table;
        Table _tcp_rules, _udp_rules;
    };

    class NetFPGAClassifier : public PacketClassifier
    {
    public:
        NetFPGAClassifier()
        {
            _netfpga = NetFPGA::instance();
        }
        bool classify(Packet *) { return false; }
        void insert(const ServiceSubTuple& st, const ServiceRule& sr)
        {
            uint32_t route = sr.mark().get_class();
            if (route == 1)
                route = 0; // UNKNOWN
            else
            {
                route = 3; // + ((route - 1) % 5);
            }
            _netfpga->insertIpPort(st.srv_ip(), st.srv_port(), st.ip_proto(), route);
        }
        void remove(const ServiceSubTuple& st)
        {
            _netfpga->removeIpPort(st.srv_ip(), st.srv_port(), st.ip_proto());
        }
    private:
        NetFPGA* _netfpga;
    };

    template <class PktCls>
    class ServiceInference
    {
    public:
        static const int MIN_OCCURENCES = 10;
        static const int MAX_DISTINCT_MARKS = 4;
        static const double MIN_HOMOGENEITY = 0.9;
        
        ServiceInference(int max_rules);
        ~ServiceInference();
        
        void push(const FlowCache::Flow& flow, const AppMark& mark);
        void infer();
        
        PacketClassifier* classifier() { return &_pkt_cls; }
        
    private:
        struct Bits
        {
            bool failed;
            AppMark marks[MAX_DISTINCT_MARKS];
            
            Bits()
            {
                failed = false;
            }
        };
        
        typedef HashTable<ServiceSubTuple, uint32_t> Table;
        typedef HashTable<AppMark, Table> TupleMap;
        typedef HashTable<ServiceSubTuple, Bits> TupleSet;
        
        ServiceRule infer(const ServiceSubTuple& st, const AppMark marks[], TupleMap& map);

        int _max_rules;
        int _utility_th; // ambience utility threshold
        
        TupleMap _tcp_map, _udp_map;
        TupleSet _tuples;
        MultiStageFilter<ServiceSubTuple>* _filter;
        
        DEQueue<Service> _rules;
        
        PktCls _pkt_cls;
    };
    
    friend int rules_utility_comp(const void *ap, const void *bp/*, void *user_data*/);
    
    unsigned int _active_sec;
    unsigned int _up_sec;
    
    unsigned int _up_interval;
    unsigned int _hc_size;
    
    FlowClassifier* _flow_cls;

    Vector<PacketClassifier*> _pkt_cls;
    //ServiceInference<ServicePacketClassifier>* _srv_inf;
    ServiceInference<NetFPGAClassifier>* _srv_inf;

    friend int rules_utility_comp(const void*, const void*, void*);
};

CLICK_ENDDECLS

#endif
