/*
 * hostcache.{cc,hh} -- classify TCP/UDP packets according to the causing
 *                      application as identified by a FlowClassifier
 *                      element.
 *
 * Marco Canini
 *
 * Copyright (c) 2008-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: hostcache.cc 2578 2009-10-05 10:32:45Z marco $
 */

#include <click/config.h>
#include <click/error.hh>
#include <click/hashmap.hh>
#include <click/straccum.hh>
#include <click/confparse.hh>
#include <clicknet/ip.h>
#include <clicknet/tcp.h>
#include <clicknet/udp.h>
#include <clicknet/icmp.h>
#include <click/packet_anno.hh>
#include <click/handlercall.hh>
#include <click/elemfilter.hh>

#include <fstream>
#include <iostream>

#include "hostcache.hh"

CLICK_DECLS

HostCache::HostCache() : _flow_cls(0), _srv_inf(0)
{
}

HostCache::~HostCache()
{
    /*for (Vector<PacketClassifier*>::iterator iter = _pkt_cls.begin();
        iter != _pkt_cls.end(); ++iter)
    {
        delete *iter;
    }*/

    delete _srv_inf;
    /*for (Vector<ServiceInference*>::iterator iter = _rule_inf.begin();
        iter != _rule_inf.end(); ++iter)
    {
        delete *iter;
    }*/
}

int HostCache::configure(Vector<String>& conf, ErrorHandler* errh)
{
    _hc_size = 120;
    _up_interval = 15;

    if (cp_va_kparse(conf, this, errh,
            "SIZE", 0, cpUnsigned, &_hc_size,
            "UP_INTERVAL", 0, cpUnsigned, &_up_interval,
            cpEnd) < 0)
        return -1;

    if (_hc_size == 0)
        return errh->error("SIZE must be greater then zero");
    if (_up_interval == 0)
        return errh->error("UP_INTERVAL must be greater then zero");

    return 0;
}

int HostCache::initialize(ErrorHandler* errh)
{
    _active_sec = _up_sec = 0;
    
    /*ServicePacketClassifier* cls = new ServicePacketClassifier();
    _pkt_cls.push_back(cls);*/
    
    //_srv_inf = new ServiceInference<ServicePacketClassifier>(_hc_size);
    _srv_inf = new ServiceInference<NetFPGAClassifier>(_hc_size);
    _pkt_cls.push_back(_srv_inf->classifier());
    
    CastElementFilter filter("FlowClassifier");
    Vector<Element*> elements;
    int ok = router()->downstream_elements(this, 0, &filter, elements);

    if (ok == -1)
        return errh->error("Initialization failure: "
            "missing reference to a FlowClassifier element");

    filter.filter(elements);

    if (elements.size() != 1)
        return errh->error("Initialization failure: "
            "missing reference to a FlowClassifier element");

    _flow_cls = static_cast<FlowClassifier*>(elements[0]->cast("FlowClassifier"));
    _flow_cls->attach_listener(this);

    return 0;
}

void HostCache::push(int, Packet* p)
{
    //int action = handle_packet(p);
    int action = handle_packet(p);

    if (_active_sec >= _up_sec)
        update();

    if (action == ACT_PASS)
        output(0).push(p);
    else if (action == ACT_CLASSIFY)
        output(1).push(p);
}

Packet* HostCache::pull(int)
{
    Packet* p = input(0).pull();

    int action = (p ? handle_packet(p) : ACT_NONE);

    if (_active_sec >= _up_sec)
        update();

    if (action == ACT_PASS)
        return p;
    else if (action == ACT_CLASSIFY)
        output(1).push(p);

    return 0;
}

int HostCache::handle_packet(Packet* p)
{
    _active_sec = p->timestamp_anno().sec();
    // NetFPGA specific
    return ACT_PASS;
    
/*
    const click_ip* iph = p->ip_header();
    // check whether the packet is IP
    if (!iph || (iph->ip_p != IP_PROTO_TCP && iph->ip_p != IP_PROTO_UDP))
        return ACT_PASS;
    
    for (Vector<PacketClassifier*>::iterator iter = _pkt_cls.begin();
        iter != _pkt_cls.end(); ++iter)
    {
        PacketClassifier* cls = *iter;
        if (cls->classify(p))
        {
            return ACT_CLASSIFY;
        }
    }
    return ACT_PASS;
*/
}

void HostCache::flow_classified(const FlowCache::Flow& flow, const AppMark& mark)
{
    /*for (Vector<ServiceInference*>::iterator iter = _rule_inf.begin();
        iter != _rule_inf.end(); ++iter)
    {
        ServiceInference* inf = *iter;
        inf->push(flow, mark);
    }*/
    _srv_inf->push(flow, mark);
}

void HostCache::update()
{
    if (_up_sec)
    {
        /*for (Vector<ServiceInference*>::iterator iter = _rule_inf.begin();
            iter != _rule_inf.end(); ++iter)
        {
            ServiceInference* inf = *iter;
            inf->infer();
        }*/
        _srv_inf->infer();
    }
    _up_sec = _active_sec + _up_interval;
}

bool HostCache::ServicePacketClassifier::classify(Packet *p)
{
    const click_ip* iph = p->ip_header();
    const uint16_t* ports = reinterpret_cast<const uint16_t*>(p->transport_header());
    Table& rules = (iph->ip_p == IP_PROTO_TCP ? _tcp_rules : _udp_rules);
    Table::iterator it;
    
    uint32_t host1 = iph->ip_dst.s_addr;
    uint16_t port1 = ports[1];
    ServiceKey key1(host1, port1);
    
    it = rules.find_prefer(key1);
    if (it != rules.end())
    {
        it->second.apply(p, 0);
        return true;
    }
    
    uint32_t host0 = iph->ip_src.s_addr;
    uint16_t port0 = ports[0];
    ServiceKey key0(host0, port0);
    
    it = rules.find_prefer(key0);
    if (it != rules.end())
    {
        it->second.apply(p, 1);
        return true;
    }
    
    return false;
}

void HostCache::ServicePacketClassifier::insert(const ServiceSubTuple& st, const ServiceRule& sr)
{
    Table& rules = (st.ip_proto() == IP_PROTO_TCP ? _tcp_rules : _udp_rules);
    ServiceKey key(st);
    rules[key] = sr;
}

void HostCache::ServicePacketClassifier::remove(const ServiceSubTuple& st)
{
    Table& rules = (st.ip_proto() == IP_PROTO_TCP ? _tcp_rules : _udp_rules);
    ServiceKey key(st);
    rules.erase(key);
}

template <class PktCls>
HostCache::ServiceInference<PktCls>::ServiceInference(int max_rules) : _max_rules(max_rules), _utility_th(0), _rules(max_rules)
{
    // FIXME: determine stages and buckets from the link speed and error level
    size_t stages = 4, buckets = 32768;
    Hashing<ServiceSubTuple>* hashing = new EnDoubleHashing<ServiceSubTuple, ServiceSubTuple::Hash1, ServiceSubTuple::Hash2>(stages);
    _filter = new MultiStageFilter<ServiceSubTuple>(stages, buckets, MIN_OCCURENCES, hashing);
}

template <class PktCls>
HostCache::ServiceInference<PktCls>::~ServiceInference()
{
    delete _filter;
}

template <class PktCls>
void HostCache::ServiceInference<PktCls>::push(const FlowCache::Flow& flow, const AppMark& mark)
{
    ServiceSubTuple st(flow);
    if (_filter->update(st))
    {
        // ensure we know about this ServiceSubTuple
        typename ServiceInference<PktCls>::TupleSet::iterator it3 = _tuples.find_insert(st);
        Bits& bits = it3->second;
        
        if (bits.failed)
            return;
        
        int i;
        for (i = 0; i < MAX_DISTINCT_MARKS; ++i)
            if (!bits.marks[i].is_valid() || bits.marks[i] == mark)
            {
                bits.marks[i] = mark;
                break;
            }
        if (i == MAX_DISTINCT_MARKS) {
            bits.failed = true;
            // TODO: clear the relevant tables
            return;
        }

        TupleMap& map = (flow.is_tcp() ? _tcp_map : _udp_map);
        TupleMap::iterator it = map.find_insert(mark);
        Table& table = it->second;
        Table::iterator it2 = table.find_insert(st);
        ++(it2->second);
    }
}

int rules_utility_comp(const void *ap, const void *bp, void *)
{
    HostCache::Service* a = (HostCache::Service*) ap;
    HostCache::Service* b = (HostCache::Service*) bp;
    if (a->second.utility() == b->second.utility())
        return 0;
    return (a->second.utility() < b->second.utility()) ? -1 : 1;
}

template <class PktCls>
void HostCache::ServiceInference<PktCls>::infer()
{
    Vector<Service> rules;
    rules.reserve(_max_rules);
    
    for (typename ServiceInference<PktCls>::TupleSet::iterator it = _tuples.begin(); it != _tuples.end(); ++it)
    {
        const ServiceSubTuple& st = it->first;
        Bits& bits = it->second;
        
        if (bits.failed)
            continue;
        
        IPAddress addr(st.srv_ip());
        
#ifdef HOSTCACHE_DEBUG
        std::cerr << "tuple in set: " << addr.s().c_str() << ":" << ntohs(st.srv_port());
        if (st.ip_proto() == IP_PROTO_TCP)
            std::cerr << " TCP";
        else
            std::cerr << " UDP";
        std::cerr << std::endl;
#endif
        
        if (st.ip_proto() == IP_PROTO_TCP)
        {
            Service s(st, infer(st, bits.marks, _tcp_map));
            if (s.second.utility() > _utility_th)
                rules.push_back(s);
        } else
        {
            Service s(st, infer(st, bits.marks, _udp_map));
            if (s.second.utility() > _utility_th)
                rules.push_back(s);
        }
    }
    
    //click_qsort(rules.begin(), rules.size(), sizeof(rules[0]), rules_utility_comp, (void *) &rules);
    click_qsort(rules.begin(), rules.size(), sizeof(Service), rules_utility_comp);
    
    int i = 0;
    if (rules.size() > _max_rules)
        i = rules.size() - _max_rules;
        
    if (_utility_th == 0 && rules.size() > 0)
        _utility_th = rules[i].second.utility();
    
    for (; i < rules.size(); ++i)
    {
        if (_rules.size() == _max_rules)
        {
            Service rs(_rules.front());
            _rules.pop_front();
            ServiceSubTuple& st = rs.first;
            ServiceRule& rule = rs.second;
            
#ifdef HOSTCACHE_DEBUG
            IPAddress addr(st.srv_ip());
            std::cerr << "removed rule: " << addr.s().c_str() << ":" << ntohs(st.srv_port());
            std::cerr << " " << to_string(rule.mark()).c_str() << " " << rule.utility() << std::endl;
#endif
            
            _pkt_cls.remove(st);
        }
        {
            Service& s = rules[i];
            ServiceSubTuple& st = s.first;
            ServiceRule& rule = s.second;
            
#ifdef HOSTCACHE_DEBUG
            IPAddress addr(st.srv_ip());
            std::cerr << "rule: " << addr.s().c_str() << ":" << ntohs(st.srv_port());
            std::cerr << " " << to_string(rule.mark()).c_str() << " " << rule.utility() << std::endl;
#endif
            
            _rules.push_back(s);
            _pkt_cls.insert(st, rule);
        }
    }
    
    for (i = 0; i < _rules.size(); ++i)
    {
        Service& s = _rules[i];
        ServiceSubTuple& st = s.first;
        ServiceRule& rule = s.second;
        
#ifdef HOSTCACHE_DEBUG
        IPAddress addr(st.srv_ip());
        std::cerr << "installed rule: " << addr.s().c_str() << ":" << ntohs(st.srv_port());
        std::cerr << " " << to_string(rule.mark()).c_str() << " " << rule.utility() << std::endl;
#endif
    }
    //_rules.clear();
    
    // clean up
    _tcp_map.clear();
    _udp_map.clear();
    _tuples.clear();
}

template <class PktCls>
HostCache::ServiceRule HostCache::ServiceInference<PktCls>::infer(const ServiceSubTuple& st, const AppMark marks[], TupleMap& map)
{
    double fracs[MAX_DISTINCT_MARKS];
    double count = 0.0;
    double max_frac = 0.0;
    AppMark max_frac_mark;
    
    int i;
    for (i = 0; i < MAX_DISTINCT_MARKS; ++i)
    {
        if (!marks[i].is_valid())
            break;
        TupleMap::iterator it = map.find(marks[i]);
        assert(it != map.end());
        
        const AppMark& mark = it->first;
        Table& table = it->second;
        Table::iterator it2 = table.find(st);
        assert(it2 != table.end());
#ifdef HOSTCACHE_DEBUG
        std::cerr << to_string(mark).c_str() << ": " << it2->second << std::endl;
#endif
        
        fracs[i] = (double) it2->second;
        count += fracs[i];
    }
    
    for (int j = 0; j < i; ++j)
    {
        fracs[j] /= count;
        if (fracs[j] > max_frac)
        {
            max_frac = fracs[j];
            max_frac_mark = marks[j];
        }
    }
    
    if (max_frac < MIN_HOMOGENEITY)
        return ServiceRule();
    
    int utility = (int) (count * (2 * max_frac - 1));
#ifdef HOSTCACHE_DEBUG
    std::cerr << "candidate rule, utility: " << utility << std::endl;
#endif
    
    return ServiceRule(max_frac_mark, utility);
}


ELEMENT_REQUIRES(userlevel)
EXPORT_ELEMENT(HostCache)

#include <click/vector.cc>
CLICK_ENDDECLS
