#ifndef HASHING_HH_
#define HASHING_HH_

#include <stdlib.h> /* rand() */
#include <vector>

template <class T>
class Hashing {
public:
    virtual ~Hashing() {};

    virtual std::vector<size_t> hash(const T& item, size_t m) const = 0;

    virtual size_t size() const = 0;
};

template <class T, class HashFcn1, class HashFcn2>
class EnDoubleHashing: public Hashing<T>
{
public:
    typedef HashFcn1 hasher1;
    typedef HashFcn2 hasher2;

    EnDoubleHashing(size_t k_, unsigned long seed1_ = 0, unsigned long seed2_ = 0) :
        k(k_), seed1(seed1_), seed2(seed2_)
    {
        if (seed1 == 0)
            seed1 = rand();
        if (seed2 == 0)
            seed2 = rand();
    }

    std::vector<size_t> hash(const T& item, size_t m) const {
        size_t x = hash1(item, seed1) % m;
        size_t y = hash2(item, seed2) % m;
        std::vector<size_t> idxs(k);
        idxs[0] = x;
        for (size_t i = 1; i < k - 1; ++i) {
            x = (x + y) % m;
            y = (y + i) % m;
            idxs[i] = x;
        }
        return idxs;
    }

    size_t size() const { return k; }
private:
    size_t k;
    unsigned long seed1;
    unsigned long seed2;
    hasher1 hash1;
    hasher2 hash2;
};

/* Universal hashing,
 * see http://www.cs.panam.edu/~schwellerr/papers/infocom.pdf
 * ARIMA, Holt-Winters, etc. [1], [8]
 */

template <class T, class KeyFcn, size_t KeySize>
struct UniversalHash
{
    static const size_t base = 256; /* this is due to using unsigned char */
    static const size_t key_length = KeySize;

    typedef UniversalHash<T, KeyFcn, KeySize> uhash;
    typedef KeyFcn keyfier;

    UniversalHash() : m(uhash::key_length * uhash::base)
    {
        for (size_t i = 0; i < uhash::key_length * uhash::base; ++i)
            m[i] = ((rand() & 0xFFFF) << 16) | (rand() & 0xFFFF);
        assert(m.size() == uhash::key_length * uhash::base);
    }

    size_t operator()(const T& item) const
    {
        int pos = -1;
        size_t value = 0;
        const unsigned char *key = keyfy(item);
        
        for (size_t i = 0; i < uhash::key_length; ++i) {
            unsigned char c = key[i];
            pos += c + 1;
            assert((size_t) pos < m.size());
            value ^= m[pos];
        }
        return value;
    }

private:
    std::vector<size_t> m;
    keyfier keyfy;    
};

template <class T, class HashFcn>
class KHashing : public Hashing<T>
{
public:
    KHashing(size_t k_) : k(k_)
    {
        for (size_t i = 0; i < k; ++i)
            funcs.push_back(HashFcn());
    }

    std::vector<size_t> hash(const T& item, size_t m) const {
        std::vector<size_t> idxs(k);
        for (size_t i = 0; i < k; ++i)
            idxs[i] = funcs[i](item) % m;
        return idxs;
    }

    size_t size() const { return k; }
private:
    size_t k;
    std::vector<HashFcn> funcs;
};

#endif /*HASHING_HH_*/
