/*
 * crashhandler.{cc,hh} -- launches the debugger when Click receives a SIGSEGV
 *                         signal
 *
 * Sergio Mangialardi
 * Marco Canini
 *
 * Copyright (c) 2008-09 by University of Genova - DIST - TNT laboratory
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the distribution.
 * * Neither the name of University of Genova nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: crashhandler.cc 2577 2009-10-05 10:14:43Z marco $
 */

#include <click/config.h>
#include <click/error.hh>
#include <click/confparse.hh>
#include <click/string.hh>
#include <click/straccum.hh>

#include<signal.h>
#include<unistd.h>
#include<errno.h>
#include <stdio.h>
#include <limits.h>

#include <iostream>

#include "crashhandler.hh"

CLICK_DECLS

namespace
{
    String getExecutablePath(int pid)
    {
        StringAccum sa;
        sa << "/proc/" << pid << "/exe";
        char buf[PATH_MAX];
        int ret = readlink(sa.take_string().c_str(), buf, PATH_MAX);
        
        if (ret == -1)
            return String("");
        
        return String(buf);
    }
}

CrashHandler* CrashHandler::instance_;

CrashHandler::CrashHandler()
{
    instance_ = this;
}

void* CrashHandler::cast(const char* n)
{
    if (strcmp(n, "CrashHandler") == 0)
        return static_cast<CrashHandler*>(this);
    else
        return Element::cast(n);
}

int CrashHandler::configure(Vector<String>& conf, ErrorHandler* errh)
{
    if (cp_va_kparse(conf, this, errh, "DEBUGGER", cpkP+cpkM, cpFilename, &debugger_, cpEnd) < 0)
        return -1;

    return 0;
}

int CrashHandler::initialize(ErrorHandler* errh)
{
    struct sigaction action;
    action.sa_handler = CrashHandler::attach;
    sigemptyset(&(action.sa_mask));
    action.sa_flags = SA_RESETHAND;
    sigaction(SIGSEGV, &action, 0);
    sigaction(SIGABRT, &action, 0);

    (void) errh;

    return 0;
}

void CrashHandler::attach(const int /*sig*/)
{
    instance_->attach();
}

void CrashHandler::attach()
{
    StringAccum sa;
    int pid = getpid();
    sa << debugger_ << " " << getExecutablePath(pid) << " " << pid << "\n";
    int res = system(sa.take_string().c_str());
    if (res == -1) {
        std::cerr << "Error while executing: " << sa.take_string() << std::endl;
    }
}

ELEMENT_REQUIRES(userlevel)
EXPORT_ELEMENT(CrashHandler)

CLICK_ENDDECLS

