/* ****************************************************************************
 * vim:set shiftwidth=2 softtabstop=2 expandtab:
 * $Id: counterdump.c 3207 2007-12-15 01:10:38Z jnaous $
 *
 * Module:  counterdump.c
 * Project: NetFPGA NIC
 * Description: dumps the MAC Rx/Tx counters to stdout
 * Author: Jad Naous
 *
 * Change history:
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include <net/if.h>

#include <signal.h>
#include <sys/time.h>

#include "reg_defines.h"
#include "nf2.h"
#include "nf2util.h"

#define PATHLEN		80

#define DEFAULT_IFACE	"nf2c0"

#define HC_NUM_PKTS_LABELED_REG         0x2000A20
#define SFC_NUM_PKTS_LABELED_REG        0x2000B2C
#define SW0_NUM_PKTS_REG                0x2000C04
#define NONIP_PKTS_REG                  0x2000D10

/* Global vars */
static struct nf2device nf2;
static int port;
static int done;

/* Function declarations */
void dumpCounts();
void processArgs (int , char **);
void usage (void);
void sigHandler(int signum);

int main(int argc, char *argv[])
{
  unsigned pkts[] = {RX_QUEUE_0_NUM_PKTS_STORED_REG,
                     RX_QUEUE_1_NUM_PKTS_STORED_REG,
                     RX_QUEUE_2_NUM_PKTS_STORED_REG,
                     RX_QUEUE_3_NUM_PKTS_STORED_REG};
  unsigned bytes[] = {RX_QUEUE_0_NUM_BYTES_PUSHED_REG,
                      RX_QUEUE_1_NUM_BYTES_PUSHED_REG,
                      RX_QUEUE_2_NUM_BYTES_PUSHED_REG,
                      RX_QUEUE_3_NUM_BYTES_PUSHED_REG};
  nf2.device_name = DEFAULT_IFACE;

  processArgs(argc, argv);

  // Open the interface if possible
  if (check_iface(&nf2))
    {
      exit(1);
    }
  if (openDescriptor(&nf2))
    {
      exit(1);
    }

  signal(SIGINT, sigHandler);
  signal(SIGTERM, sigHandler);

  printf("# ts phy%d_pkt_rate phy%d_bw hc_labeled_rate sfc_labeled_rate sw0_pkt_rate nonip_pkt_rate\n", port, port);
  dumpCounts(pkts[port], bytes[port]);

  closeDescriptor(&nf2);

  return 0;
}

static struct timeval
timeval_diff(struct timeval tv1, struct timeval tv2)
{
    struct timeval diff;

    diff.tv_sec = tv2.tv_sec - tv1.tv_sec;
    diff.tv_usec = tv2.tv_usec - tv1.tv_usec;
    if (diff.tv_usec < 0) {
        diff.tv_sec--;
        diff.tv_usec += 1000000;
    }
    return diff;
}

#define RATE(rate,cnt,prev_cnt) \
    if (cnt >= prev_cnt) \
        rate = ((double) cnt - prev_cnt) / dur; \
    else \
        rate = ((double) cnt + (ULONG_MAX - prev_cnt)) / dur

void dumpCounts(unsigned pkts_reg, unsigned bytes_reg)
{
  unsigned pkts, bytes, prev_pkts, prev_bytes, hc_pkts, sfc_pkts, sw0_pkts, nonip_pkts, prev_hc_pkts, prev_sfc_pkts, prev_sw0_pkts, prev_nonip_pkts;
  struct timeval now, prev, diff;
  double dur, pkt_rate, bw, hc_rate, sfc_rate, sw0_rate, nonip_rate;

  readReg(&nf2, pkts_reg, &prev_pkts);
  readReg(&nf2, bytes_reg, &prev_bytes);
  readReg(&nf2, HC_NUM_PKTS_LABELED_REG, &prev_hc_pkts);
  readReg(&nf2, SFC_NUM_PKTS_LABELED_REG, &prev_sfc_pkts);
  readReg(&nf2, SW0_NUM_PKTS_REG, &prev_sw0_pkts);
  readReg(&nf2, NONIP_PKTS_REG, &prev_nonip_pkts);

  gettimeofday(&prev, NULL);

  while (done == 0)
  {  
    sleep(1);
    readReg(&nf2, pkts_reg, &pkts);
    readReg(&nf2, bytes_reg, &bytes);
    readReg(&nf2, HC_NUM_PKTS_LABELED_REG, &hc_pkts);
    readReg(&nf2, SFC_NUM_PKTS_LABELED_REG, &sfc_pkts);
    readReg(&nf2, SW0_NUM_PKTS_REG, &sw0_pkts);
    readReg(&nf2, NONIP_PKTS_REG, &nonip_pkts);
    gettimeofday(&now, NULL);

    diff = timeval_diff(prev, now);
    dur = diff.tv_sec + ((double) diff.tv_usec / 1000000.0);
    RATE(pkt_rate, pkts, prev_pkts);
    RATE(hc_rate, hc_pkts, prev_hc_pkts);
    RATE(sfc_rate, sfc_pkts, prev_sfc_pkts);
    RATE(sw0_rate, sw0_pkts, prev_sw0_pkts);
    RATE(nonip_rate, nonip_pkts, prev_nonip_pkts);
    RATE(bw, bytes, prev_bytes);
    bw *= 8.0;

    printf("%lu.%lu %f %f %f %f %f %f\n", now.tv_sec, now.tv_usec, pkt_rate, bw, hc_rate, sfc_rate, sw0_rate, nonip_rate);

    prev = now;
    prev_pkts = pkts;
    prev_bytes = bytes;
    prev_hc_pkts = hc_pkts;
    prev_sfc_pkts = sfc_pkts;
    prev_sw0_pkts = sw0_pkts;
    prev_nonip_pkts = nonip_pkts;
  }
}  

void sigHandler(int signum)
{
  done = 1;
}

/* 
 *  Process the arguments.
 */
void processArgs (int argc, char **argv )
{
  char c;

  /* don't want getopt to moan - I can do that just fine thanks! */
  opterr = 0;
	  
  while ((c = getopt (argc, argv, "i:p:h")) != -1)
    {
      switch (c)
	{
	case 'i':	/* interface name */
	  nf2.device_name = optarg;
	  break;
        case 'p':       /* port number */
          port = strtol(optarg, NULL, 10);
          break;
	case '?':
	  if (isprint (optopt))
	    fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	  else
	    fprintf (stderr,
		     "Unknown option character `\\x%x'.\n",
		     optopt);
	case 'h':
	default:
	  usage();
	  exit(1);
	}
    }
}


/*
 *  Describe usage of this program.
 */
void usage (void)
{
  printf("Usage: ./bw <options> \n\n");
  printf("Options: -i <iface> : interface name (default nf2c0)\n");
  printf("         -p <port> : port number (default 0)\n");
  printf("         -h : Print this message and exit.\n");
}
