///////////////////////////////////////////////////////////////////////////////
// vim:set shiftwidth=3 softtabstop=3 expandtab:
// $Id: user_data_path.v 4385 2008-08-05 02:10:01Z grg $
//
// Module: user_data_path.v
// Project: NF2.1
// Description: contains all the user instantiated modules
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

/******************************************************
 * Even numbered ports are IO sinks/sources
 * Odd numbered ports are CPU ports corresponding to
 * IO sinks/sources to rpovide direct access to them
 ******************************************************/
module user_data_path
  #(parameter DATA_WIDTH = 64,
    parameter CTRL_WIDTH=DATA_WIDTH/8,
    parameter UDP_REG_SRC_WIDTH = 2,
    parameter NUM_OUTPUT_QUEUES = 8,
    parameter NUM_INPUT_QUEUES = 8,
    parameter SRAM_DATA_WIDTH = DATA_WIDTH+CTRL_WIDTH,
    parameter SRAM_ADDR_WIDTH = 19)

   (
    input  [DATA_WIDTH-1:0]            in_data_0,
    input  [CTRL_WIDTH-1:0]            in_ctrl_0,
    input                              in_wr_0,
    output                             in_rdy_0,

    input  [DATA_WIDTH-1:0]            in_data_1,
    input  [CTRL_WIDTH-1:0]            in_ctrl_1,
    input                              in_wr_1,
    output                             in_rdy_1,

    input  [DATA_WIDTH-1:0]            in_data_2,
    input  [CTRL_WIDTH-1:0]            in_ctrl_2,
    input                              in_wr_2,
    output                             in_rdy_2,

    input  [DATA_WIDTH-1:0]            in_data_3,
    input  [CTRL_WIDTH-1:0]            in_ctrl_3,
    input                              in_wr_3,
    output                             in_rdy_3,

    input  [DATA_WIDTH-1:0]            in_data_4,
    input  [CTRL_WIDTH-1:0]            in_ctrl_4,
    input                              in_wr_4,
    output                             in_rdy_4,

    input  [DATA_WIDTH-1:0]            in_data_5,
    input  [CTRL_WIDTH-1:0]            in_ctrl_5,
    input                              in_wr_5,
    output                             in_rdy_5,

    input  [DATA_WIDTH-1:0]            in_data_6,
    input  [CTRL_WIDTH-1:0]            in_ctrl_6,
    input                              in_wr_6,
    output                             in_rdy_6,

    input  [DATA_WIDTH-1:0]            in_data_7,
    input  [CTRL_WIDTH-1:0]            in_ctrl_7,
    input                              in_wr_7,
    output                             in_rdy_7,

/****  not used
    // --- Interface to SATA
    input  [DATA_WIDTH-1:0]            in_data_5,
    input  [CTRL_WIDTH-1:0]            in_ctrl_5,
    input                              in_wr_5,
    output                             in_rdy_5,

    // --- Interface to the loopback queue
    input  [DATA_WIDTH-1:0]            in_data_6,
    input  [CTRL_WIDTH-1:0]            in_ctrl_6,
    input                              in_wr_6,
    output                             in_rdy_6,

    // --- Interface to a user queue
    input  [DATA_WIDTH-1:0]            in_data_7,
    input  [CTRL_WIDTH-1:0]            in_ctrl_7,
    input                              in_wr_7,
    output                             in_rdy_7,
*****/

    output  [DATA_WIDTH-1:0]           out_data_0,
    output  [CTRL_WIDTH-1:0]           out_ctrl_0,
    output                             out_wr_0,
    input                              out_rdy_0,

    output  [DATA_WIDTH-1:0]           out_data_1,
    output  [CTRL_WIDTH-1:0]           out_ctrl_1,
    output                             out_wr_1,
    input                              out_rdy_1,

    output  [DATA_WIDTH-1:0]           out_data_2,
    output  [CTRL_WIDTH-1:0]           out_ctrl_2,
    output                             out_wr_2,
    input                              out_rdy_2,

    output  [DATA_WIDTH-1:0]           out_data_3,
    output  [CTRL_WIDTH-1:0]           out_ctrl_3,
    output                             out_wr_3,
    input                              out_rdy_3,

    output  [DATA_WIDTH-1:0]           out_data_4,
    output  [CTRL_WIDTH-1:0]           out_ctrl_4,
    output                             out_wr_4,
    input                              out_rdy_4,

    output  [DATA_WIDTH-1:0]           out_data_5,
    output  [CTRL_WIDTH-1:0]           out_ctrl_5,
    output                             out_wr_5,
    input                              out_rdy_5,

    output  [DATA_WIDTH-1:0]           out_data_6,
    output  [CTRL_WIDTH-1:0]           out_ctrl_6,
    output                             out_wr_6,
    input                              out_rdy_6,

    output  [DATA_WIDTH-1:0]           out_data_7,
    output  [CTRL_WIDTH-1:0]           out_ctrl_7,
    output                             out_wr_7,
    input                              out_rdy_7,

/****  not used
    // --- Interface to SATA
    output  [DATA_WIDTH-1:0]           out_data_5,
    output  [CTRL_WIDTH-1:0]           out_ctrl_5,
    output                             out_wr_5,
    input                              out_rdy_5,

    // --- Interface to the loopback queue
    output  [DATA_WIDTH-1:0]           out_data_6,
    output  [CTRL_WIDTH-1:0]           out_ctrl_6,
    output                             out_wr_6,
    input                              out_rdy_6,

    // --- Interface to a user queue
    output  [DATA_WIDTH-1:0]           out_data_7,
    output  [CTRL_WIDTH-1:0]           out_ctrl_7,
    output                             out_wr_7,
    input                              out_rdy_7,
*****/

     // interface to SRAM
     output [SRAM_ADDR_WIDTH-1:0]       wr_0_addr,
     output                             wr_0_req,
     input                              wr_0_ack,
     output [SRAM_DATA_WIDTH-1:0]       wr_0_data,
     
     input                              rd_0_ack,
     input  [SRAM_DATA_WIDTH-1:0]       rd_0_data,
     input                              rd_0_vld,
     output [SRAM_ADDR_WIDTH-1:0]       rd_0_addr,
     output                             rd_0_req,

     // interface to DRAM
     /* TBD */

     // register interface
     input                              reg_req,
     output                             reg_ack,
     input                              reg_rd_wr_L,
     input [`UDP_REG_ADDR_WIDTH-1:0]    reg_addr,
     output [`CPCI_NF2_DATA_WIDTH-1:0]  reg_rd_data,
     input [`CPCI_NF2_DATA_WIDTH-1:0]   reg_wr_data,
   
     // misc
     input                              reset,
     input                              clk);


   function integer log2;
      input integer number;
      begin
         log2=0;
         while(2**log2<number) begin
            log2=log2+1;
         end
      end
   endfunction // log2

   //---------- Internal parameters -----------

   localparam NUM_IQ_BITS = log2(NUM_INPUT_QUEUES);
   
   localparam IN_ARB_STAGE_NUM = 2;
   localparam OP_LUT_STAGE_NUM = 4;
   localparam OQ_STAGE_NUM     = 6;
   
   //-------- Input arbiter wires/regs ------- 
   wire                             in_arb_in_reg_req;
   wire                             in_arb_in_reg_ack;
   wire                             in_arb_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   in_arb_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  in_arb_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     in_arb_in_reg_src;

   //------- output port lut wires/regs ------
   wire [CTRL_WIDTH-1:0]            op_lut_in_ctrl;
   wire [DATA_WIDTH-1:0]            op_lut_in_data;
   wire                             op_lut_in_wr;
   wire                             op_lut_in_rdy;

   wire                             op_lut_in_reg_req;
   wire                             op_lut_in_reg_ack;
   wire                             op_lut_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   op_lut_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  op_lut_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     op_lut_in_reg_src;

   //------- dropnonip wires/regs ------
   wire [CTRL_WIDTH-1:0]            drop_in_ctrl;
   wire [DATA_WIDTH-1:0]            drop_in_data;
   wire                             drop_in_wr;
   wire                             drop_in_rdy;

   wire                             drop_in_reg_req;
   wire                             drop_in_reg_ack;
   wire                             drop_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   drop_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  drop_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     drop_in_reg_src;


   //------- hc wires/regs ------
   wire [CTRL_WIDTH-1:0]            hc_in_ctrl;
   wire [DATA_WIDTH-1:0]            hc_in_data;
   wire                             hc_in_wr;
   wire                             hc_in_rdy;

   wire                             hc_in_reg_req;
   wire                             hc_in_reg_ack;
   wire                             hc_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   hc_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  hc_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     hc_in_reg_src;

   //------- hc wires/regs ------
   wire [CTRL_WIDTH-1:0]            pf_in_ctrl;
   wire [DATA_WIDTH-1:0]            pf_in_data;
   wire                             pf_in_wr;
   wire                             pf_in_rdy;

   wire                             pf_in_reg_req;
   wire                             pf_in_reg_ack;
   wire                             pf_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   pf_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  pf_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     pf_in_reg_src;

   //------- ar wires/regs ------
   wire [CTRL_WIDTH-1:0]            ar_in_ctrl;
   wire [DATA_WIDTH-1:0]            ar_in_data;
   wire                             ar_in_wr;
   wire                             ar_in_rdy;

   wire                             ar_in_reg_req;
   wire                             ar_in_reg_ack;
   wire                             ar_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   ar_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  ar_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     ar_in_reg_src;

   //------- output queues wires/regs ------
   wire [CTRL_WIDTH-1:0]            oq_in_ctrl;
   wire [DATA_WIDTH-1:0]            oq_in_data;
   wire                             oq_in_wr;
   wire                             oq_in_rdy;

   wire                             oq_in_reg_req;
   wire                             oq_in_reg_ack;
   wire                             oq_in_reg_rd_wr_L;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   oq_in_reg_addr;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  oq_in_reg_data;
   wire [UDP_REG_SRC_WIDTH-1:0]     oq_in_reg_src;

   //-------- UDP register master wires/regs ------- 
   wire                             udp_reg_req_in;
   wire                             udp_reg_ack_in;
   wire                             udp_reg_rd_wr_L_in;
   wire [`UDP_REG_ADDR_WIDTH-1:0]   udp_reg_addr_in;
   wire [`CPCI_NF2_DATA_WIDTH-1:0]  udp_reg_data_in;
   wire [UDP_REG_SRC_WIDTH-1:0]     udp_reg_src_in;


   //--------- Connect the data path -----------
   
   input_arbiter
     #(.DATA_WIDTH(DATA_WIDTH),
       .CTRL_WIDTH(CTRL_WIDTH),
       .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH),
       .STAGE_NUMBER(IN_ARB_STAGE_NUM))
   input_arbiter
     (
    .out_data             (drop_in_data),
    .out_ctrl             (drop_in_ctrl),
    .out_wr               (drop_in_wr),
    .out_rdy              (drop_in_rdy),
                          
      // --- Interface to the input queues
    .in_data_0            (in_data_0),
    .in_ctrl_0            (in_ctrl_0),
    .in_wr_0              (in_wr_0),
    .in_rdy_0             (in_rdy_0),
                          
    .in_data_1            (in_data_1),
    .in_ctrl_1            (in_ctrl_1),
    .in_wr_1              (in_wr_1),
    .in_rdy_1             (in_rdy_1),
                          
    .in_data_2            (in_data_2),
    .in_ctrl_2            (in_ctrl_2),
    .in_wr_2              (in_wr_2),
    .in_rdy_2             (in_rdy_2),
                          
    .in_data_3            (in_data_3),
    .in_ctrl_3            (in_ctrl_3),
    .in_wr_3              (in_wr_3),
    .in_rdy_3             (in_rdy_3),
                          
    .in_data_4            (in_data_4),
    .in_ctrl_4            (in_ctrl_4),
    .in_wr_4              (in_wr_4),
    .in_rdy_4             (in_rdy_4),
                          
    .in_data_5            (in_data_5),
    .in_ctrl_5            (in_ctrl_5),
    .in_wr_5              (in_wr_5),
    .in_rdy_5             (in_rdy_5),
                          
    .in_data_6            (in_data_6),
    .in_ctrl_6            (in_ctrl_6),
    .in_wr_6              (in_wr_6),
    .in_rdy_6             (in_rdy_6),
                          
    .in_data_7            (in_data_7),
    .in_ctrl_7            (in_ctrl_7),
    .in_wr_7              (in_wr_7),
    .in_rdy_7             (in_rdy_7),
                          
      // --- Register interface
    .reg_req_in           (in_arb_in_reg_req),
    .reg_ack_in           (in_arb_in_reg_ack),
    .reg_rd_wr_L_in       (in_arb_in_reg_rd_wr_L),
    .reg_addr_in          (in_arb_in_reg_addr),
    .reg_data_in          (in_arb_in_reg_data),
    .reg_src_in           (in_arb_in_reg_src),

     .reg_req_out          (drop_in_reg_req),
     .reg_ack_out          (drop_in_reg_ack),
     .reg_rd_wr_L_out      (drop_in_reg_rd_wr_L),
     .reg_addr_out         (drop_in_reg_addr),
     .reg_data_out         (drop_in_reg_data),
     .reg_src_out          (drop_in_reg_src),

      // --- Misc
    .reset                (reset),
    .clk                  (clk)
    );

   dropnonip #(
      .DATA_WIDTH(DATA_WIDTH),
      .CTRL_WIDTH(CTRL_WIDTH),
      .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH)
   ) dropnonippackets (
      // --- data path interface
      .in_data                           (drop_in_data),
      .in_ctrl                           (drop_in_ctrl),
      .in_wr                             (drop_in_wr),
      .in_rdy                            (drop_in_rdy),

      .out_data                          (hc_in_data),
      .out_ctrl                          (hc_in_ctrl),
      .out_wr                            (hc_in_wr),
      .out_rdy                           (hc_in_rdy),

     .reg_req_in          (drop_in_reg_req),
    .reg_ack_in           (drop_in_reg_ack),
    .reg_rd_wr_L_in       (drop_in_reg_rd_wr_L),
    .reg_addr_in          (drop_in_reg_addr),
    .reg_data_in          (drop_in_reg_data),
    .reg_src_in           (drop_in_reg_src),

     .reg_req_out          (hc_in_reg_req),
     .reg_ack_out          (hc_in_reg_ack),
     .reg_rd_wr_L_out      (hc_in_reg_rd_wr_L),
     .reg_addr_out         (hc_in_reg_addr),
     .reg_data_out         (hc_in_reg_data),
     .reg_src_out          (hc_in_reg_src),

     
      // --- Misc
      .clk                               (clk),
      .reset                             (reset)
   );
 
   hc #(
      .DATA_WIDTH(DATA_WIDTH),
      .CTRL_WIDTH(CTRL_WIDTH),
      .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH),
      .INPUT_ARBITER_STAGE_NUM(IN_ARB_STAGE_NUM),
      .NUM_OUTPUT_QUEUES(NUM_OUTPUT_QUEUES),
      .NUM_IQ_BITS(NUM_IQ_BITS)
   ) hostcache (
      // --- data path interface
      .in_data                           (hc_in_data),
      .in_ctrl                           (hc_in_ctrl),
      .in_wr                             (hc_in_wr),
      .in_rdy                            (hc_in_rdy),

      .out_data                          (pf_in_data),
      .out_ctrl                          (pf_in_ctrl),
      .out_wr                            (pf_in_wr),
      .out_rdy                           (pf_in_rdy),
      // --- Register interface
      .reg_req_in                        (hc_in_reg_req),
      .reg_ack_in                        (hc_in_reg_ack),
      .reg_rd_wr_L_in                    (hc_in_reg_rd_wr_L),
      .reg_addr_in                       (hc_in_reg_addr),
      .reg_data_in                       (hc_in_reg_data),
      .reg_src_in                        (hc_in_reg_src),

      .reg_req_out                       (pf_in_reg_req),
      .reg_ack_out                       (pf_in_reg_ack),
      .reg_rd_wr_L_out                   (pf_in_reg_rd_wr_L),
      .reg_addr_out                      (pf_in_reg_addr),
      .reg_data_out                      (pf_in_reg_data),
      .reg_src_out                       (pf_in_reg_src),
      

      // --- Misc
      .clk                               (clk),
      .reset                             (reset)
   );
  
   pf #(
      .DATA_WIDTH(DATA_WIDTH),
      .CTRL_WIDTH(CTRL_WIDTH),
      .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH),
      .INPUT_ARBITER_STAGE_NUM(IN_ARB_STAGE_NUM),
      .NUM_OUTPUT_QUEUES(NUM_OUTPUT_QUEUES),
      .NUM_IQ_BITS(NUM_IQ_BITS)
   ) packetfilter (
      // --- data path interface
      .in_data                           (pf_in_data),
      .in_ctrl                           (pf_in_ctrl),
      .in_wr                             (pf_in_wr),
      .in_rdy                            (pf_in_rdy),

      .out_data                          (ar_in_data),
      .out_ctrl                          (ar_in_ctrl),
      .out_wr                            (ar_in_wr),
      .out_rdy                           (ar_in_rdy),
      // --- Register interface
      .reg_req_in                        (pf_in_reg_req),
      .reg_ack_in                        (pf_in_reg_ack),
      .reg_rd_wr_L_in                    (pf_in_reg_rd_wr_L),
      .reg_addr_in                       (pf_in_reg_addr),
      .reg_data_in                       (pf_in_reg_data),
      .reg_src_in                        (pf_in_reg_src),

      .reg_req_out                       (ar_in_reg_req),
      .reg_ack_out                       (ar_in_reg_ack),
      .reg_rd_wr_L_out                   (ar_in_reg_rd_wr_L),
      .reg_addr_out                      (ar_in_reg_addr),
      .reg_data_out                      (ar_in_reg_data),
      .reg_src_out                       (ar_in_reg_src),

      // --- Misc
      .clk                               (clk),
      .reset                             (reset)
   );

  ar #(
      .DATA_WIDTH(DATA_WIDTH),
      .CTRL_WIDTH(CTRL_WIDTH),
      .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH),
      .INPUT_ARBITER_STAGE_NUM(IN_ARB_STAGE_NUM),
      .NUM_OUTPUT_QUEUES(NUM_OUTPUT_QUEUES),
      .NUM_IQ_BITS(NUM_IQ_BITS)
   ) applicationrouting (
      // --- data path interface
      .in_data                           (ar_in_data),
      .in_ctrl                           (ar_in_ctrl),
      .in_wr                             (ar_in_wr),
      .in_rdy                            (ar_in_rdy),

      .out_data                          (oq_in_data),
      .out_ctrl                          (oq_in_ctrl),
      .out_wr                            (oq_in_wr),
      .out_rdy                           (oq_in_rdy),
      // --- Register interface
      .reg_req_in                        (ar_in_reg_req),
      .reg_ack_in                        (ar_in_reg_ack),
      .reg_rd_wr_L_in                    (ar_in_reg_rd_wr_L),
      .reg_addr_in                       (ar_in_reg_addr),
      .reg_data_in                       (ar_in_reg_data),
      .reg_src_in                        (ar_in_reg_src),

      .reg_req_out                       (oq_in_reg_req),
      .reg_ack_out                       (oq_in_reg_ack),
      .reg_rd_wr_L_out                   (oq_in_reg_rd_wr_L),
      .reg_addr_out                      (oq_in_reg_addr),
      .reg_data_out                      (oq_in_reg_data),
      .reg_src_out                       (oq_in_reg_src),


      // --- Misc
      .clk                               (clk),
      .reset                             (reset)
   );


   output_queues
     #(.DATA_WIDTH(DATA_WIDTH),
       .CTRL_WIDTH(CTRL_WIDTH),
       .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH),
       .OP_LUT_STAGE_NUM(OP_LUT_STAGE_NUM),
       .NUM_OUTPUT_QUEUES(NUM_OUTPUT_QUEUES),
       .STAGE_NUM(OQ_STAGE_NUM),
       .SRAM_ADDR_WIDTH(SRAM_ADDR_WIDTH))
   output_queues
     (// --- data path interface
    .out_data_0       (out_data_0),
    .out_ctrl_0       (out_ctrl_0),
    .out_wr_0         (out_wr_0),
    .out_rdy_0        (out_rdy_0),
                      
    .out_data_1       (out_data_1),
    .out_ctrl_1       (out_ctrl_1),
    .out_wr_1         (out_wr_1),
    .out_rdy_1        (out_rdy_1),
                      
    .out_data_2       (out_data_2),
    .out_ctrl_2       (out_ctrl_2),
    .out_wr_2         (out_wr_2),
    .out_rdy_2        (out_rdy_2),
                      
    .out_data_3       (out_data_3),
    .out_ctrl_3       (out_ctrl_3),
    .out_wr_3         (out_wr_3),
    .out_rdy_3        (out_rdy_3),
                      
    .out_data_4       (out_data_4),
    .out_ctrl_4       (out_ctrl_4),
    .out_wr_4         (out_wr_4),
    .out_rdy_4        (out_rdy_4),
                      
    .out_data_5       (out_data_5),
    .out_ctrl_5       (out_ctrl_5),
    .out_wr_5         (out_wr_5),
    .out_rdy_5        (out_rdy_5),
                      
    .out_data_6       (out_data_6),
    .out_ctrl_6       (out_ctrl_6),
    .out_wr_6         (out_wr_6),
    .out_rdy_6        (out_rdy_6),
                      
    .out_data_7       (out_data_7),
    .out_ctrl_7       (out_ctrl_7),
    .out_wr_7         (out_wr_7),
    .out_rdy_7        (out_rdy_7),

      // --- Interface to the previous module
    .in_data          (oq_in_data),
    .in_ctrl          (oq_in_ctrl),
    .in_rdy           (oq_in_rdy),
    .in_wr            (oq_in_wr),
                     
      // --- Register interface
    .reg_req_in       (oq_in_reg_req),
    .reg_ack_in       (oq_in_reg_ack),
    .reg_rd_wr_L_in   (oq_in_reg_rd_wr_L),
    .reg_addr_in      (oq_in_reg_addr),
    .reg_data_in      (oq_in_reg_data),
    .reg_src_in       (oq_in_reg_src),

    .reg_req_out      (udp_reg_req_in),
    .reg_ack_out      (udp_reg_ack_in),
    .reg_rd_wr_L_out  (udp_reg_rd_wr_L_in),
    .reg_addr_out     (udp_reg_addr_in),
    .reg_data_out     (udp_reg_data_in),
    .reg_src_out      (udp_reg_src_in),

      // --- SRAM sm interface
    .wr_0_addr        (wr_0_addr),
    .wr_0_req         (wr_0_req),
    .wr_0_ack         (wr_0_ack),
    .wr_0_data        (wr_0_data),
    .rd_0_ack         (rd_0_ack),
    .rd_0_data        (rd_0_data),
    .rd_0_vld         (rd_0_vld),
    .rd_0_addr        (rd_0_addr),
    .rd_0_req         (rd_0_req),
                      
                      
      // --- Misc
    .clk              (clk),
    .reset            (reset));


   //--------------------------------------------------
   //
   // --- User data path register master
   //
   //     Takes the register accesses from core,
   //     sends them around the User Data Path module
   //     ring and then returns the replies back
   //     to the core
   //
   //--------------------------------------------------

   udp_reg_master #(
      .UDP_REG_SRC_WIDTH (UDP_REG_SRC_WIDTH)
   ) udp_reg_master (
      // Core register interface signals
      .core_reg_req                          (reg_req),
      .core_reg_ack                          (reg_ack),
      .core_reg_rd_wr_L                      (reg_rd_wr_L),

      .core_reg_addr                         (reg_addr),

      .core_reg_rd_data                      (reg_rd_data),
      .core_reg_wr_data                      (reg_wr_data),

      // UDP register interface signals (output)
      .reg_req_out                           (in_arb_in_reg_req),
      .reg_ack_out                           (in_arb_in_reg_ack),
      .reg_rd_wr_L_out                       (in_arb_in_reg_rd_wr_L),

      .reg_addr_out                          (in_arb_in_reg_addr),
      .reg_data_out                          (in_arb_in_reg_data),

      .reg_src_out                           (in_arb_in_reg_src),

      // UDP register interface signals (input)
      .reg_req_in                            (udp_reg_req_in),
      .reg_ack_in                            (udp_reg_ack_in),
      .reg_rd_wr_L_in                        (udp_reg_rd_wr_L_in),

      .reg_addr_in                           (udp_reg_addr_in),
      .reg_data_in                           (udp_reg_data_in),

      .reg_src_in                            (udp_reg_src_in),

      //
      .clk                                   (clk),
      .reset                                 (reset)
   );


endmodule // user_data_path

