///////////////////////////////////////////////////////////////////////////////
// Module: l3l4extract.v
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps


//`define UDP_REG_ADDR_WIDTH 2
//`define CPCI_NF2_DATA_WIDTH 32

module l3l4extract_pf
   #(
      parameter DATA_WIDTH = 64,
      parameter CTRL_WIDTH = DATA_WIDTH/8,
      parameter UDP_REG_SRC_WIDTH = 2
   )
   (
      input  [DATA_WIDTH-1:0]             in_data,
      input  [CTRL_WIDTH-1:0]             in_ctrl,
      input                               in_wr,
      output                              in_rdy,

      output reg [DATA_WIDTH-1:0]         out_data,
      output reg [CTRL_WIDTH-1:0]         out_ctrl,
      output reg                          out_wr,
      input                               out_rdy,

      // misc
      input                                reset,
      input                                clk
   );


   //------------------------- Signals-------------------------------
   localparam SKIPTONEXTPACKET = 0;
   localparam EXTRACTFIELDS    = 1;
   localparam WRITEFIELD0      = 2;
   localparam WRITEFIELD1      = 3;
   localparam WRITEFIELD2      = 4;

   wire [DATA_WIDTH-1:0]       in_fifo_data;
   wire [CTRL_WIDTH-1:0]       in_fifo_ctrl;
   wire                        in_fifo_nearly_full;
   wire                        in_fifo_empty;
   reg                         in_fifo_rd_en;

   reg [31:0] ipsrc, ipdst;
   reg [3:0]  ipversion;
   reg        swap_tuple;

   reg [15:0] tcp_src_port, tcp_dst_port;

   reg [7:0]  protocol_field;
   reg [1:0]  protocol_ident;

   reg [15:0] packet_len_field;  //Polozka delka v ethernet ramci 

   reg [7:0]  tos;
   reg [7:0]  ttl;   
   reg [7:0]  tcpflags;
   
   reg [15:0] doctets;           
   reg [15:0] input_port; 
   reg [7:0]  output_port; 

   wire       l3proto_ok;
   reg        l4proto_ok;

   reg [3:0]  cnt;
   reg        cnt_reset;

   reg [2:0]  fsm_state, nfsm_state;

   wire       in_fifo_data_rdy;
   reg        registers_ready;
   reg        allow_extract;


   //------------------------- Modules-------------------------------
   
   fallthrough_small_fifo #(
      .WIDTH(CTRL_WIDTH+DATA_WIDTH),
      .MAX_DEPTH_BITS(2)
   ) input_fifo (
      .din           ({in_ctrl, in_data}),   // Data in
      .wr_en         (in_wr),                // Write enable
      .rd_en         (in_fifo_rd_en),        // Read the next word 
      .dout          ({in_fifo_ctrl, in_fifo_data}),
      .full          (),
      .nearly_full   (in_fifo_nearly_full),
      .empty         (in_fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );   

   //------------------------- Local assignments -------------------------------

   assign in_rdy           = !in_fifo_nearly_full;
   assign in_fifo_data_rdy = !in_fifo_empty;

   //------------------------- Logic-------------------------------

   always @(posedge clk) begin
      if (allow_extract && in_fifo_data_rdy) begin
         case (cnt)
            0: begin
                  input_port      = in_fifo_data[31:16];    
                  output_port     = in_fifo_data[55:48];    
               end

            2: begin
                  packet_len_field = in_fifo_data[31:16];
                  ipversion        = in_fifo_data[15:12]; 
                  tos              = in_fifo_data[7:0];
               end

            3: begin
                  doctets         = in_fifo_data[63:48];
                  ttl             = in_fifo_data[15:8];
                  protocol_field  = in_fifo_data[7:0];
               end

            4: begin
                  ipsrc           = in_fifo_data[47:16];
                  ipdst[31:16]    = in_fifo_data[15:0]; 
               end   

            5: begin
                  ipdst[15:0]     = in_fifo_data[63:48];
                  tcp_src_port    = in_fifo_data[47:32];
                  tcp_dst_port    = in_fifo_data[31:16];
               end

            6: begin
                  tcpflags        = in_fifo_data[7:0];                      
               end  
            default:   ;     
         endcase    
      end
   end
  
   always @(posedge clk) begin
      swap_tuple <= (ipsrc > ipdst);
   end

   always @(posedge clk) begin
      if (cnt_reset || reset)
         cnt <= 0;
      else begin
         if (allow_extract && in_fifo_data_rdy) begin
            cnt <= cnt + 1;
         end    
      end
   end

   always @(*) begin
      allow_extract  = 0;
      in_fifo_rd_en  = 0;
      cnt_reset      = 0;
      out_wr         = 0;
      out_ctrl       = 'h0;
      out_data       = {ipsrc, ipdst};
      nfsm_state     = fsm_state;    
      case (fsm_state)
         SKIPTONEXTPACKET: begin
            cnt_reset  = 1;
            if (in_fifo_data_rdy) begin
               if (in_fifo_ctrl == 'hff)
                  nfsm_state = EXTRACTFIELDS;
               else
                  in_fifo_rd_en  = 1;
            end
         end

         EXTRACTFIELDS: begin 
            if (in_fifo_data_rdy) begin
               allow_extract = 1;
               in_fifo_rd_en = 1;
               if (cnt == 6) begin
//                  if (l3proto_ok) // && l4proto_ok)      
                     nfsm_state = WRITEFIELD0;
//                  else
//                     nfsm_state = SKIPTONEXTPACKET;
               end
            end
         end

         WRITEFIELD0: begin
            if (out_rdy) begin
               out_wr   = 1;
               out_data = {tos, doctets, ttl, 16'b0,
                           (protocol_ident == 2)? tcpflags : 8'b0, 
                           input_port[7:0]};

               nfsm_state = WRITEFIELD1;
            end
         end

         WRITEFIELD1: begin
            if (out_rdy) begin
               out_wr   = 1;
               //out_data = {ipsrc, ipdst};
               out_data = (swap_tuple)? {ipdst, ipsrc}:{ipsrc, ipdst};
               nfsm_state = WRITEFIELD2;
            end
         end

         WRITEFIELD2: begin
            if (out_rdy) begin
               out_wr   = 1;
               out_ctrl = 'hff;
/*             out_data = {(protocol_ident == 2 || protocol_ident == 3)?
                          {tcp_src_port, tcp_dst_port}:
                           32'b0, input_port[7:0], protocol_field, 16'b0};
*/

               out_data = {(protocol_ident == 2 || protocol_ident == 3)?
                          ((swap_tuple)?{tcp_dst_port, tcp_src_port}:
                                        {tcp_src_port, tcp_dst_port}):
                           32'b0, 
                           input_port[7:0], protocol_field, 
                           output_port[7:0], 7'b0, l3proto_ok};

               nfsm_state = SKIPTONEXTPACKET;
            end
         end
      endcase                    
   end

   always @(posedge clk) begin
      if (reset)
         fsm_state = SKIPTONEXTPACKET;
      else
         fsm_state = nfsm_state;
   end

   // Zpracovavam pouze IPv4         
   assign l3proto_ok = (packet_len_field == 16'h0800)? 1:0;    //XXX zatim neberu stary ethernet

   //Otestovani, zda se jedna o protokol nami zpracovavany
   always @(*) begin
      l4proto_ok     = 1;
      protocol_ident = 0;

      case (protocol_field)
         1: protocol_ident = 1; //ICMP
         6: protocol_ident = 2; //TCP
        17: protocol_ident = 3; //UDP
        default: l4proto_ok = 0;
      endcase
   end
            
endmodule 
