///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps
module hc
   #(
      parameter DATA_WIDTH = 64,
      parameter CTRL_WIDTH = DATA_WIDTH/8,
      parameter UDP_REG_SRC_WIDTH = 1,
      parameter INPUT_ARBITER_STAGE_NUM = 2,
      parameter NUM_OUTPUT_QUEUES = 8,
      parameter STAGE_NUM = 5,
      parameter NUM_IQ_BITS = 3
   )
   (
      // --- data path interface
      input  [DATA_WIDTH-1:0]            in_data,
      input  [CTRL_WIDTH-1:0]            in_ctrl,
      input                              in_wr,
      output                             in_rdy,

      output reg [DATA_WIDTH-1:0]        out_data,
      output [CTRL_WIDTH-1:0]        out_ctrl,
      output reg                         out_wr,
      input                              out_rdy,


      // --- Register interface
      input                              reg_req_in,
      input                              reg_ack_in,
      input                              reg_rd_wr_L_in,
      input  [`UDP_REG_ADDR_WIDTH-1:0]   reg_addr_in,
      input  [`CPCI_NF2_DATA_WIDTH-1:0]  reg_data_in,
      input  [UDP_REG_SRC_WIDTH-1:0]     reg_src_in,

      output                             reg_req_out,
      output                             reg_ack_out,
      output                             reg_rd_wr_L_out,
      output  [`UDP_REG_ADDR_WIDTH-1:0]  reg_addr_out,
      output  [`CPCI_NF2_DATA_WIDTH-1:0] reg_data_out,
      output  [UDP_REG_SRC_WIDTH-1:0]    reg_src_out,
      

      // --- Misc
      input                              clk,
      input                              reset
   );

   `LOG2_FUNC
   
   //--------------------- Internal Parameter-------------------------
   localparam DECIDE = 0;
   localparam ACCEPT0 = 1;
   localparam ACCEPT1 = 2;
   localparam DISCARD0 = 3;
   localparam DISCARD1 = 4;

   //---------------------- Wires and regs----------------------------

   wire [CTRL_WIDTH-1:0]            l3l4extract_in_ctrl;
   wire [DATA_WIDTH-1:0]            l3l4extract_in_data;
   wire                             l3l4extract_in_wr;
   wire                             l3l4extract_in_rdy;

   //------- HashGen wires/regs ------
   wire [CTRL_WIDTH-1:0]            hashgen_in_ctrl;
   wire [DATA_WIDTH-1:0]            hashgen_in_data;
   wire                             hashgen_in_wr;
   wire                             hashgen_in_rdy;

   //------- FlowLookup wires/regs ------
   wire [CTRL_WIDTH-1:0]            flowlookup_in_ctrl0;
   reg  [DATA_WIDTH-1:0]            flowlookup_in_data0;
   wire                             flowlookup_in_wr0;
   wire                             flowlookup_in_rdy0;

   wire [CTRL_WIDTH-1:0]            flowlookup_in_ctrl1;
   wire [DATA_WIDTH-1:0]            flowlookup_in_data1;
   wire                             flowlookup_in_wr1;
   wire                             flowlookup_in_rdy1;

   //------- RecordWrapper wires/regs ------
   wire [CTRL_WIDTH-1:0]            decision_in_ctrl;
   wire [DATA_WIDTH-1:0]            decision_in_data;
   wire                             decision_in_wr;
   wire                             decision_in_rdy;

   wire [CTRL_WIDTH-1:0]            decision_out_ctrl;
   wire [DATA_WIDTH-1:0]            decision_out_data;
   wire                             decision_out_wr;
   wire                             decision_out_rdy;
   
   reg                              in_fifo_rd_en;
   wire [CTRL_WIDTH-1:0]            in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0]            in_fifo_data_dout;
   wire                             in_fifo_nearly_full;
   wire                             in_fifo_empty;

   reg                              decision_fifo_rd_en;
   wire [CTRL_WIDTH-1:0]            decision_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0]            decision_fifo_data_dout;
   wire                             decision_fifo_nearly_full;
   wire                             decision_fifo_empty;
   // -------------------------------------------
   //  Arbitrary signals
   // -------------------------------------------
   reg  [2:0] state, next_state;

   wire [63:0]  key;
   wire [63:0]  hashed_key;
   wire [63:0]  hashed_key_stat;
   wire [31:0]  cmd;
   wire [31:0]  write;
   reg          reg_write;
   reg write_cmd;
  
   wire  [2:0] iface, oface;
   reg  [7:0] output_port;

   reg  classified;
   reg  notclassified;
   reg  [31:0] cnt_classified;
   reg  [31:0] cnt_unclassified;

   wire  [31:0] reg_sw;
   wire   flowlookup_reset;
   wire   flowlookup_bloominit;

   localparam INITSEED = 'hCA;
 localparam NUMBEROFPORTS= 8;
 
   //------------------------- Modules-------------------------------
/*   assign reg_req_out = reg_req_in;
   assign reg_ack_out = reg_ack_in;
   assign reg_rd_wr_L_out = reg_rd_wr_L_in;
   assign reg_addr_out = reg_addr_in;
   assign reg_data_out = reg_data_in;
   assign reg_src_out = reg_src_in;
*/
   generic_regs
   #( 
      .UDP_REG_SRC_WIDTH   (UDP_REG_SRC_WIDTH),
      .TAG                 (`HOST_BLOCK_TAG),
      .REG_ADDR_WIDTH      (`HOST_REG_ADDR_WIDTH),  // Width of block addresses
      .NUM_COUNTERS        (0),                       // How many counters
      .NUM_SOFTWARE_REGS   (5),                       // How many sw regs
      .NUM_HARDWARE_REGS   (7)                        // How many hw regs
   ) hostcache_regs (
      .reg_req_in       (reg_req_in),
      .reg_ack_in       (reg_ack_in),
      .reg_rd_wr_L_in   (reg_rd_wr_L_in),
      .reg_addr_in      (reg_addr_in),
      .reg_data_in      (reg_data_in),
      .reg_src_in       (reg_src_in),

      .reg_req_out      (reg_req_out),
      .reg_ack_out      (reg_ack_out),
      .reg_rd_wr_L_out  (reg_rd_wr_L_out),
      .reg_addr_out     (reg_addr_out),
      .reg_data_out     (reg_data_out),
      .reg_src_out      (reg_src_out),

      // --- counters interface
      .counter_updates  (),
      .counter_decrement(),

      // --- SW regs interface
      .software_regs    ({reg_sw, write, cmd, key}),

      // --- HW regs interface
      .hardware_regs    ({hashed_key_stat, cnt_unclassified, cnt_classified, cmd, hashed_key}),

      .clk              (clk),
      .reset            (reset)
    );


   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(8))
      input_fifo
        (.din ({in_ctrl,in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word 
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );

   //----------------------- Logic -----------------------------

   assign    in_rdy              = l3l4extract_in_rdy && !in_fifo_nearly_full;
   assign    l3l4extract_in_data = in_data;
   assign    l3l4extract_in_ctrl = in_ctrl;
   assign    l3l4extract_in_wr   = in_wr;

   l3l4extract_hc
   #(.DATA_WIDTH(DATA_WIDTH),
     .CTRL_WIDTH(CTRL_WIDTH)
   )
   l3l4extract_hostcache
     (
      // --- Interface to the rx input queues
     .in_data              (l3l4extract_in_data),
     .in_ctrl              (l3l4extract_in_ctrl),
     .in_wr                (l3l4extract_in_wr),
     .in_rdy               (l3l4extract_in_rdy),

     .out_data             (hashgen_in_data),
     .out_ctrl             (hashgen_in_ctrl),
     .out_wr               (hashgen_in_wr),
     .out_rdy              (hashgen_in_rdy),
                           
      // --- Misc
     .clk                  (clk),
     .reset                (reset));

  hashgen_hc
     #(.DATA_WIDTH(DATA_WIDTH),
       .CTRL_WIDTH(CTRL_WIDTH)
    )
  hashgen_hostcache
     (
      // --- Interface to the rx input queues
     .in_data             (hashgen_in_data),
     .in_ctrl             (hashgen_in_ctrl),
     .in_wr               (hashgen_in_wr),
     .in_rdy              (hashgen_in_rdy),

     .out_data            (flowlookup_in_data1),
     .out_ctrl            (flowlookup_in_ctrl1),
     .out_wr              (flowlookup_in_wr1),
     .out_rdy             (flowlookup_in_rdy1),

      // --- Misc
     .clk                  (clk),
     .reset                (reset));

 
  always @(posedge clk) begin
     if (reset)
        reg_write = 0;
     else
        reg_write = write[0];
  end
  
  hash hash_key
   (
      .in({INITSEED[7:0], key[55:0]}),
      .out(hashed_key)
   );
 

  assign flowlookup_in_wr0   = write[0] && !reg_write;

  always @(posedge clk) begin
     flowlookup_in_data0 = {hashed_key[63:8], cmd[19:16], cmd[3:0]};
  end

  assign flowlookup_in_ctrl0  = 'hff;
  assign flowlookup_reset     = reset || reg_sw[0];
  assign flowlookup_bloominit = reset || reg_sw[1];

  flowlookup_hc
     #(.HASHESPERLINE(4),
       .NUMLINES(2**9),
       .HASHWIDTH(32),
       .DATA_WIDTH(DATA_WIDTH),
       .CTRL_WIDTH(CTRL_WIDTH)
    )
  flowlookup_hostcache
     (
      // --- Interface to the rx input queues
     .in_data0           (flowlookup_in_data0),
     .in_ctrl0           (flowlookup_in_ctrl0),
     .in_wr0             (flowlookup_in_wr0),
     .in_rdy0            (flowlookup_in_rdy0),

     .in_data1           (flowlookup_in_data1),
     .in_ctrl1           (flowlookup_in_ctrl1),
     .in_wr1             (flowlookup_in_wr1),
     .in_rdy1            (flowlookup_in_rdy1),

     .bloominit          (flowlookup_bloominit),
     .bloomstathash      (hashed_key),
     .searchedbloomstat  (hashed_key_stat),

     .out_data           (decision_in_data),
     .out_ctrl           (decision_in_ctrl),
     .out_wr             (decision_in_wr),
     .out_rdy            (decision_in_rdy),

      // --- Misc
     .clk                (clk),
     .reset              (flowlookup_reset));

   decision_hc
     #(.DATA_WIDTH(DATA_WIDTH),
       .CTRL_WIDTH(CTRL_WIDTH)
    )
   decision
     (
      // --- Interface to the rx input queues
     .in_data            (decision_in_data),
     .in_ctrl            (decision_in_ctrl),
     .in_wr              (decision_in_wr),
     .in_rdy             (decision_in_rdy),

     .out_data           (decision_out_data),
     .out_ctrl           (decision_out_ctrl),
     .out_wr             (decision_out_wr),
     .out_rdy            (decision_out_rdy),
                           
     .clk                (clk),
     .reset              (reset));

   assign decision_out_rdy = !decision_fifo_nearly_full;

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(8))
      decision_fifo
        (.din ({decision_out_ctrl,decision_out_data}),     // Data in
         .wr_en (decision_out_wr),               // Write enable
         .rd_en (decision_fifo_rd_en),       // Read the next word 
         .dout ({decision_fifo_ctrl_dout, decision_fifo_data_dout}),
         .full (),
         .nearly_full (decision_fifo_nearly_full),
         .empty (decision_fifo_empty),
         .reset (reset),
         .clk (clk)
         );

   assign iface = in_fifo_data_dout[18:16];
   assign oface = iface[2:0] - 1;

   integer j;
   always @(*) begin
      output_port = 'h0;
      for (j = 0; j < NUMBEROFPORTS; j = j + 1) begin
         if (oface == j)
            output_port[j] = 1;
         else
            output_port[j] = 0;
      end
   end

   always @(*) begin
      decision_fifo_rd_en = 0;
      in_fifo_rd_en = 0;
      out_wr        = 0;

      notclassified = 0;
      classified    = 0;

      out_data = in_fifo_data_dout;

      next_state    = state;
      case (state)
         DECIDE: begin
            if (!decision_fifo_empty) begin
               next_state = ACCEPT0;
            end
         end

         ACCEPT0: begin
            out_data[`IOQ_DST_PORT_POS+7:`IOQ_DST_PORT_POS] = 
              (iface[0])? 'h00: decision_fifo_data_dout[7:0];

            if (decision_fifo_data_dout[7:0] == 0)
               notclassified = 1;
            else
               classified = 1;

            if (!in_fifo_empty && out_rdy) begin
               in_fifo_rd_en = 1;
               if (in_fifo_ctrl_dout == 'hff) begin
                  decision_fifo_rd_en = 1;
                  out_wr     = 1;
                  next_state = ACCEPT1;
               end
            end
         end
         ACCEPT1: begin 
            if (!in_fifo_empty && out_rdy) begin
               in_fifo_rd_en = 1;
               out_wr        = 1;
               if (in_fifo_ctrl_dout != 'h0) begin
                  next_state = DECIDE;
               end
            end
         end
      endcase                    
   end
         
   always @(posedge clk) begin
      if (reset)
         state = DECIDE;
      else
         state = next_state;
   end

   assign out_ctrl = in_fifo_ctrl_dout;
   
   always @(posedge clk) begin
      if (reset)
         cnt_classified = 'h0;
      else if (classified)
         cnt_classified = cnt_classified + 1;
   end

   always @(posedge clk) begin
      if (reset)
         cnt_unclassified = 'h0;
      else if (notclassified)
         cnt_unclassified = cnt_unclassified + 1;
   end


endmodule 
