///////////////////////////////////////////////////////////////////////////////
// Module: hashgen.v
// Project: NetFlow
// Autor: Petr Simon (xsimon10@stud.fit.vutbr.cz)
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

`define HASHGEN_BLOCK_TAG 'h0
`define HASHGEN_REG_ADDR_WIDTH 'h0

module hashgen_pf
  #(parameter DATA_WIDTH = 64,
    parameter CTRL_WIDTH=DATA_WIDTH/8
   )

   (
    input  [DATA_WIDTH-1:0]            in_data,
    input  [CTRL_WIDTH-1:0]            in_ctrl,
    input                              in_wr,
    output                             in_rdy,

    output reg [DATA_WIDTH-1:0]        out_data,
    output  [CTRL_WIDTH-1:0]           out_ctrl,
    output                             out_wr,
    input                              out_rdy,

    input                              clk,
    input                              reset
   );

   //------------------------- Signals-------------------------------

   wire  [DATA_WIDTH-1:0]            in_data_masked;
   reg  [DATA_WIDTH-1:0]            hashmask;

   wire [DATA_WIDTH-1:0]         fifo_out_data;
   wire [CTRL_WIDTH-1:0]         fifo_out_ctrl;

   wire                          fifo_nearly_full;
   wire                          fifo_empty;

   reg                           fifo_rd_en;

   reg                           in_rdy_reg;
   reg                           out_wr_reg;
   reg [CTRL_WIDTH-1:0]          out_ctrl_reg;

   reg [2:0]                     in_state;
   reg [2:0]                     in_next_state;
   reg                           out_state;
   reg                           out_next_state;

   reg [DATA_WIDTH - 1:0]                    hash;

   wire [DATA_WIDTH-1:0]         hash_out;
   reg  [DATA_WIDTH-1:0]         reg_hash_out;
   reg                           reg_in_wr;
   reg                           hash_wr;


   localparam INITSEED = 0;

   // state for input automat
   localparam SAVEDATA1 = 0;
   localparam SAVEDATA2 = 1;
   localparam SAVEDATA3 = 2;
   localparam SAVEDATA4 = 3;
   localparam WAITFORHASH  = 4;
   localparam WAITSENDHASH = 5;
   //state for output automat
   localparam SENDHASH = 0;
   localparam SENDPACKET = 1;
 
   
   // crc table
   assign in_rdy     = !fifo_nearly_full & in_rdy_reg;
  
   

   //input FSM
   always @(*) begin 

      in_rdy_reg = 1;
      hash_wr    = 0;
      in_next_state = in_state;
      hashmask   = 'h0;

      case (in_state)
         SAVEDATA1: begin
            hashmask   = 'h0;
            if (in_wr) 
               in_next_state = SAVEDATA3;
         end

         SAVEDATA3: begin
            hashmask = 'hFFFFFFFF_FFFFFFFF;
            if (in_wr) begin 
               hash_wr = 1;
               in_next_state = SAVEDATA4;
            end 
         end
         SAVEDATA4: begin
            hashmask = 'hFFFFFFFF_00FF0000;
            if (in_wr) begin
               hash_wr = 1;
               if (in_ctrl == 'hff)
                  in_next_state = WAITFORHASH;
            end
         end
         
         WAITFORHASH: begin
            in_rdy_reg = 0;
            in_next_state = WAITSENDHASH;
         end

         WAITSENDHASH: begin
            in_rdy_reg = 0;
            if (out_next_state == SENDPACKET) 
               in_next_state = SAVEDATA1;
         end
      endcase      
   end

   always @(posedge clk) begin  
      if (reset) begin
         in_state <= SAVEDATA1;
      end 
      else begin
         in_state <= in_next_state;
      end
   end

   always @(posedge clk) begin 
      if (reset) 
         reg_in_wr = 0;
      else 
         reg_in_wr = hash_wr;
   end

   always @(posedge clk) begin 
      reg_hash_out = hash_out;
   end


   assign in_data_masked = in_data & hashmask;

   hash hash_gen_key (
     .in(in_data_masked),
     .out(hash_out)
   );

/*
   genvar i;
   generate
   for (i=0; i < 4; i = i + 1) begin: GEN_PLUS_HASH1
      always @(posedge clk) begin 
         reg_hash_out[(i+1)*8-1:(i*8)] = 
                           in_data[ ( (i+4) * 8 + 7) : (i+4) * 8] +
                           in_data[ ( (i+0) * 8 + 7) : (i+0) * 8]; 
      end
   end
   endgenerate

   generate
   for (i=0; i < 4; i = i + 1) begin: GEN_PLUS_HASH2
      always @(posedge clk) begin 
         reg_hash_out[(i+1)*8-1+32:(i*8)+32] = 
                           in_data[ ( (2*i+1) * 8 + 7) : (2*i+1) * 8] +
                           in_data[ ( (2*i+0) * 8 + 7) : (2*i+0) * 8];
      end
   end
   endgenerate
*/

   //compute HASH
   always @(posedge clk) begin 
      if (reset || in_state == SAVEDATA1)
         hash = INITSEED;
      else begin
         if (reg_in_wr) begin 
//          hash <= hash ^ hash_out;
            hash = hash + reg_hash_out;
         end
      end
   end

   fallthrough_small_fifo #(
      .WIDTH(CTRL_WIDTH+DATA_WIDTH),
      .MAX_DEPTH_BITS(3)
   ) fifo (
      .din           ({in_ctrl, in_data}),   // Data in
      .wr_en         (in_wr),                // Write enable
      .rd_en         (fifo_rd_en),        // vystupni komponenta muze cist
      .dout          ({fifo_out_ctrl, fifo_out_data}),
      .full          (),
      .nearly_full   (fifo_nearly_full),
      .empty         (fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );

   assign out_wr     = out_wr_reg;
   assign out_ctrl   = out_ctrl_reg;

   //output FSM
   always @(*) begin

      out_wr_reg = 0;
      fifo_rd_en = 0;
      out_data = fifo_out_data;
      out_ctrl_reg = fifo_out_ctrl;
      
      out_next_state = out_state;

      case (out_state)      
         SENDHASH: begin
            out_data = hash;
            out_ctrl_reg = 'h0;
            if (out_rdy && in_state == WAITSENDHASH) begin
               out_wr_reg = 1;
               out_next_state = SENDPACKET;
            end
         end

         SENDPACKET: begin
            if (!fifo_empty && out_rdy) begin
               out_wr_reg = 1;
               fifo_rd_en = 1;
               if (fifo_out_ctrl == 'hff) begin
                  out_next_state = SENDHASH;
               end
            end
         end
      endcase
   end

   always @(posedge clk) begin  
      if (reset) begin
         out_state <= SENDHASH;
      end 
      else begin
         out_state <= out_next_state;
      end
   end

endmodule

