///////////////////////////////////////////////////////////////////////////////
// Module: hashgen.v
// Project: NetFlow
// Autor: Petr Simon (xsimon10@stud.fit.vutbr.cz)
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

`define HASHGEN_BLOCK_TAG 'h0
`define HASHGEN_REG_ADDR_WIDTH 'h0

module hashgen_hc
  #(parameter DATA_WIDTH = 64,
    parameter CTRL_WIDTH=DATA_WIDTH/8
   )

   (
    input  [DATA_WIDTH-1:0]            in_data,
    input  [CTRL_WIDTH-1:0]            in_ctrl,
    input                              in_wr,
    output                             in_rdy,

    output reg [DATA_WIDTH-1:0]        out_data,
    output reg [CTRL_WIDTH-1:0]        out_ctrl,
    output reg                         out_wr,
    input                              out_rdy,

    input                              clk,
    input                              reset
   );

   //------------------------- Signals-------------------------------

   wire [DATA_WIDTH-1:0]         in_fifo_data_dout;
   wire [CTRL_WIDTH-1:0]         in_fifo_ctrl_dout;

   wire                          in_fifo_nearly_full;
   wire                          in_fifo_empty;
   reg                           in_fifo_rd_en;

   reg [2:0]                     in_state;
   reg [2:0]                     in_next_state;
   reg [1:0]                     out_state;
   reg [1:0]                     out_next_state;


   reg  [DATA_WIDTH-1:0]         reg_hash_dst;
   reg  [DATA_WIDTH-1:0]         reg_hash_src;

   wire  [DATA_WIDTH-1:0]        hash_dst_out;
   wire  [DATA_WIDTH-1:0]        hash_src_out;


   reg                           reg_hash_vld;
   reg                           hash_vld;
   reg                           hash_rst;

   reg                           word0_we;
   reg                           word3_we;
   reg                           word4_we;

   reg [7:0]                     tcpflags_reg;
   reg [31:0]                    src_ip_reg;
   reg [31:0]                    dst_ip_reg;
   reg [15:0]                    tcp_dst_port_reg;
   reg [15:0]                    tcp_src_port_reg;
   reg [7:0]                     protocol_reg;
   reg [7:0]                     input_port_reg;

   localparam INITSEED = 'hCA;

   // state for input automat
   localparam SAVEDATA1 = 0;
   localparam SAVEDATA2 = 1;
   localparam SAVEDATA3 = 2;
   localparam SAVEDATA4 = 3;
   localparam WRITEHASH = 4;
   //state for output automat
   localparam WAITFORHASH = 0;
   localparam SENDHASH0   = 1;
   localparam SENDHASH1   = 2;
 
   
   // crc table
   assign in_rdy  = !in_fifo_nearly_full;
  
  fallthrough_small_fifo #(
      .WIDTH(CTRL_WIDTH+DATA_WIDTH),
      .MAX_DEPTH_BITS(3)
   ) in_fifo (
      .din           ({in_ctrl, in_data}),   // Data in
      .wr_en         (in_wr),                // Write enable
      .rd_en         (in_fifo_rd_en),        // vystupni komponenta muze cist
      .dout          ({in_fifo_ctrl_dout, in_fifo_data_dout}),
      .full          (),
      .nearly_full   (in_fifo_nearly_full),
      .empty         (in_fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );
 

   //input FSM
   always @(*) begin 

      in_fifo_rd_en = 0;
      hash_vld      = 0;
      in_next_state = in_state;
      word0_we      = 0;
      word3_we      = 0;
      word4_we      = 0;

      case (in_state)
         SAVEDATA1:
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               word0_we      = 1;
               in_next_state = SAVEDATA3;
            end 

//         SAVEDATA2:
//            if (in_wr) 
//               in_next_state = SAVEDATA3;

         SAVEDATA3: begin
            if (!in_fifo_empty && !reg_hash_vld) begin
               in_fifo_rd_en = 1;
               word3_we      = 1;
               in_next_state = SAVEDATA4;
            end 
         end
         SAVEDATA4: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               word4_we      = 1;
               if (in_fifo_ctrl_dout == 'hff)
                  in_next_state = WRITEHASH;
            end
         end
         
         WRITEHASH: begin
            hash_vld      = 1;
            in_next_state = SAVEDATA1;
         end

      endcase      
   end


   always @(posedge clk) begin
     if (word0_we) begin
        tcpflags_reg <= in_fifo_data_dout[15:8];
     end


     if (word3_we) begin
        src_ip_reg <= in_fifo_data_dout[63:32];
        dst_ip_reg <= in_fifo_data_dout[31:0];
     end

     if (word4_we) begin
        tcp_src_port_reg <= in_fifo_data_dout[63:48];
        tcp_dst_port_reg <= in_fifo_data_dout[47:32];
        input_port_reg   <= in_fifo_data_dout[31:24];
        protocol_reg     <= in_fifo_data_dout[23:16];
     end

   end


   always @(posedge clk) begin  
      if (reset) begin
         in_state <= SAVEDATA1;
      end 
      else begin
         in_state <= in_next_state;
      end
   end

   hash hash_src
   (
      .in({INITSEED[7:0], protocol_reg, tcp_src_port_reg, src_ip_reg}),
      .out(hash_src_out)
   );

   hash hash_dst
   (
      .in({INITSEED[7:0], protocol_reg, tcp_dst_port_reg, dst_ip_reg}),
      .out(hash_dst_out)
   );



   //compute HASH
   always @(posedge clk) begin 
       if (reset) begin
          reg_hash_vld <= 1'b0;
       end
       else begin
          if (hash_vld) begin
             reg_hash_vld <= 1'b1;
          end 
          else begin
             if (hash_rst)
                reg_hash_vld <= 1'b0;
          end
       end
   end


   always @(posedge clk) begin 
       if (hash_vld) begin
          reg_hash_dst <= {hash_dst_out[63:10], (protocol_reg == 8'h06), (tcpflags_reg == 8'h12),hash_dst_out[7:0]};
          reg_hash_src <= {hash_src_out[63:10], (protocol_reg == 8'h06) , (tcpflags_reg == 8'h12),hash_src_out[7:0]};
       end
   end


   //output FSM
   always @(*) begin

      hash_rst   = 0;
      out_data   = reg_hash_dst;
      out_ctrl   = 'hff;
      out_wr     = 0;
      
      out_next_state = out_state;

      case (out_state)      
         WAITFORHASH: begin
            if (reg_hash_vld && out_rdy) begin
               out_next_state = SENDHASH0;
            end
         end
         SENDHASH0: begin
            out_wr   = 1;
            out_data = reg_hash_dst;
            out_next_state = SENDHASH1;
         end

         SENDHASH1: begin
            out_wr   = 1;
            hash_rst = 1;
            out_data = reg_hash_src;
            out_next_state = WAITFORHASH;
         end
      endcase
   end

   always @(posedge clk) begin  
      if (reset)
         out_state <= WAITFORHASH;
      else 
         out_state <= out_next_state;
   end

endmodule

