///////////////////////////////////////////////////////////////////////////////
//
// Module: flowlookup.v
// Description: 
//
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

module flowlookup_pf #(
   parameter HASHESPERLINE = 4,
   parameter NUMLINES  = 2**8,
   parameter HASHWIDTH = 32,
   parameter DATA_WIDTH = 64,
   parameter CTRL_WIDTH = DATA_WIDTH/8
   )
   (

   input  [DATA_WIDTH-1:0]            in_data0,
   input  [CTRL_WIDTH-1:0]            in_ctrl0,
   input                              in_wr0,
   output                             in_rdy0,

   input  [DATA_WIDTH-1:0]            in_data1,
   input  [CTRL_WIDTH-1:0]            in_ctrl1,
   input                              in_wr1,
   output                             in_rdy1,

   output [DATA_WIDTH-1:0]            out_data,
   output reg [CTRL_WIDTH-1:0]        out_ctrl,
   output reg                         out_wr,
   input                              out_rdy,

   output [31:0]            occupation,
   output [31:0]            replaced,
 
   input                              clk,
   input                              reset
);
   
   function integer log2;
      input integer number;
      begin
         log2=0;
         while((2**log2)<number) begin
            log2=log2+1;
         end
      end
   endfunction // log2 

   localparam STATWIDTH = 3;
   localparam ROUTEWIDTH = 3;
   
   localparam WAIT_FOR_FIFO  = 0;
   localparam DELETE_CMDADDR = 1;
   localparam INWORD0 = 2;
   localparam INWORD1 = 3;
   localparam INWORD2 = 4;
   localparam INWORD3 = 5;

   localparam LOOKUP_LATENCY = 4;
   
   localparam CMD_DONTCARE = 4'h0;
   localparam CMD_NEW    = 4'h1;
   localparam CMD_UPDATE = 4'h2;

   localparam CMD_DELETE = 2'h0;
   localparam CMD_INSERT = 2'h1;
   localparam CMD_RESET  = 2'h2;



   localparam LAST_WORD  = 8'hff;

   localparam INIT_MEM         = 0;
   localparam WAIT_FOR_HASH    = 1;
   localparam WAIT_FOR_DELETE  = 2;
   localparam WAIT_FOR_INDEX   = 3;
   localparam WRITE_REST       = 4;

   localparam PROTOICMP  = 8'h01;
   localparam PROTOTCP   = 8'h06;
   localparam PROTOUDP   = 8'h11;

   localparam RSTFLAG    = 2;
   localparam FINFLAG    = 0;
   localparam SYNFLAG    = 1;
   localparam FINMARK    = {STATWIDTH{1'b1}};

   parameter  NUMOUTSTATES = 6;

   localparam LOG2NUMLINES = log2(NUMLINES);
   localparam LOG2HASHESPERLINE = log2(HASHESPERLINE);

   reg [3:0] instate, next_instate;
   reg [3:0] outstate, next_outstate;

   // fifo 0
   wire [CTRL_WIDTH-1:0] in_fifo0_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo0_data_dout;
   wire in_fifo0_nearly_full;
   wire in_fifo0_empty;
   reg  in_fifo0_rd_en;

   // fifo 1 
   wire [CTRL_WIDTH-1:0] in_fifo1_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo1_data_dout;
   wire in_fifo1_nearly_full;
   wire in_fifo1_empty;
   reg  in_fifo1_rd_en;
   
   // fifo shared
   wire [CTRL_WIDTH-1:0] in_ctrl;
   wire [DATA_WIDTH-1:0] in_data;
   reg  in_wr;
   wire [CTRL_WIDTH-1:0] in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo_data_dout;
   wire in_fifo_nearly_full;
   wire in_fifo_empty;
   reg  in_fifo_rd_en;
   wire in_rdy;
   

   // output of FSM0 (instate)
   reg  sel_delete;
   reg  sel_insert;
   reg  sel_reset;
   reg  set_hash_vld;
   reg  sel_data_hash;
   wire dontcare;

   wire [31:0] lookup_data;

   // memory definition   
/*   
   wire [HASHESPERLINE-1:0]  hash_mem_wea;
   wire [HASHESPERLINE-1:0]  stat_mem_wea;
*/
   reg [HASHESPERLINE-1:0]  hash_mem_wea;
   reg [HASHESPERLINE-1:0]  stat_mem_wea;
   reg [HASHESPERLINE-1:0]  route_mem_wea;

   reg [LOG2NUMLINES-1:0] hash_mem_addra [HASHESPERLINE-1:0];
   reg  [HASHWIDTH-1:0]      hash_mem_ina   [HASHESPERLINE-1:0];
   wire [HASHWIDTH-1:0]      hash_mem_outa  [HASHESPERLINE-1:0];
   reg [HASHWIDTH-1:0]      reg_hash_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0] hash_mem_addrb [HASHESPERLINE-1:0];
   wire [HASHWIDTH-1:0]      hash_mem_outb  [HASHESPERLINE-1:0];
       
   reg [LOG2NUMLINES-1:0] stat_mem_addra [HASHESPERLINE-1:0];
   reg  [STATWIDTH-1:0]      stat_mem_ina   [HASHESPERLINE-1:0];
   wire [STATWIDTH-1:0]      stat_mem_outa  [HASHESPERLINE-1:0];
   reg  [STATWIDTH-1:0]      reg_stat_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0] stat_mem_addrb [HASHESPERLINE-1:0];
   wire [STATWIDTH-1:0]      stat_mem_outb  [HASHESPERLINE-1:0];

   reg [LOG2NUMLINES-1:0] route_mem_addra [HASHESPERLINE-1:0];
   reg  [ROUTEWIDTH-1:0]      route_mem_ina   [HASHESPERLINE-1:0];
   wire [ROUTEWIDTH-1:0]      route_mem_outa  [HASHESPERLINE-1:0];
   reg  [ROUTEWIDTH-1:0]      reg_route_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0] route_mem_addrb [HASHESPERLINE-1:0];
   wire [ROUTEWIDTH-1:0]      route_mem_outb  [HASHESPERLINE-1:0];

   reg [ROUTEWIDTH-1:0]      reg_routing;
   wire [ROUTEWIDTH-1:0]      reg_route;

   //reg [STATWIDTH-1:0]      reg_stat;
   wire [STATWIDTH-1:0]      reg_stat;
  

   reg [HASHESPERLINE - 1 : 0]        hash_match;
   reg [LOG2HASHESPERLINE - 1 : 0]  hash_match_addr;
   reg [HASHESPERLINE - 1 : 0]        stat_last;
   reg [LOG2HASHESPERLINE - 1 : 0]  stat_last_addr;
   reg [HASHESPERLINE - 1 : 0]        mem_wr_mask;


   // registers
   reg  reg_cmddelete;
   reg  reg_cmdinsert;
   reg  reg_cmdreset;
   reg [LOG2NUMLINES+LOG2HASHESPERLINE-1:0] reg_addr;
   reg [HASHWIDTH-1:0] reg_hash;

   //logic
   wire delete;
   wire lookedup;
   wire matched;
   reg  reg_matched;
   reg  init;
   reg  [LOG2NUMLINES-1:0] cnt_init;
   reg  [LOG2NUMLINES-1:0] reg_cnt_init;
   wire [LOG2NUMLINES-1:0] line_addr;
   reg  [LOG2NUMLINES-1:0] stat_line_addr;

   // output of FSM1 (outstate)
   reg  rst_hash_vld;
   reg  hash_wr_en;
   reg  stat_wr_en;
   reg  route_wr_en;
   reg  [3:0] cmd;
   reg  [LOOKUP_LATENCY-1:0] hash_vld;
  
   reg  inword0_we;
   reg  inword1_we;
   reg  inword2_we;
   reg  inword3_we;

   reg [31:0] src_ip_addr;   
   reg [31:0] dst_ip_addr;  
   reg [7:0] iface;       
   reg [7:0] oface;       
   reg [15:0] byte_count; 
   reg [15:0] src_port;     
   reg [15:0] dst_port;    
   reg [7:0] tcp_flags;  
   reg       tcp_flags_syn;  
   reg       tcp_flags_fin;  
   reg       tcp_flags_rst;  
   reg [7:0] protocol;  
   reg [7:0] l3proto_ok;  
   reg [7:0] tos;         
   reg [7:0] ttl;        

   reg [31:0] cnt_occupation;        
   wire occupation_we;        
   reg occupation_up;        
   reg occupation_down;        

   reg [31:0] cnt_replaced;        
   reg replaced_we;        
   reg line_full;        


   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input0_fifo
        (.din ({in_ctrl0,in_data0}),     // Data in
         .wr_en (in_wr0),               // Write enable
         .rd_en (in_fifo0_rd_en),       // Read the next word
         .dout ({in_fifo0_ctrl_dout, in_fifo0_data_dout}),
         .full (),
         .nearly_full (in_fifo0_nearly_full),
         .empty (in_fifo0_empty),
         .reset (reset),
         .clk (clk)
         );   
   
   assign in_rdy0 = !in_fifo0_nearly_full;

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input1_fifo
        (.din ({in_ctrl1,in_data1}),     // Data in
         .wr_en (in_wr1),               // Write enable
         .rd_en (in_fifo1_rd_en),       // Read the next word
         .dout ({in_fifo1_ctrl_dout, in_fifo1_data_dout}),
         .full (),
         .nearly_full (in_fifo1_nearly_full),
         .empty (in_fifo1_empty),
         .reset (reset),
         .clk (clk)
         );   
   
   assign in_rdy1 = !in_fifo1_nearly_full;

   always @(*) begin
      next_instate   = instate;

      inword0_we = 0;
      inword1_we = 0;
      inword2_we = 0;
      inword3_we = 0;
      
      sel_delete = 0;
      sel_insert = 0;
      sel_reset  = 0;

      in_fifo0_rd_en = 0;
      in_fifo1_rd_en = 0;
      in_wr          = 0;
      set_hash_vld   = 0;

      case(instate)
         WAIT_FOR_FIFO : begin
            if ( (!hash_vld[0] || rst_hash_vld) && !init) begin
               if (!in_fifo0_empty) begin
                  next_instate = DELETE_CMDADDR;
               end
               else begin
                  if (!in_fifo1_empty) begin
                     next_instate = INWORD0;
                  end
               end
            end
         end
         DELETE_CMDADDR: begin
            sel_delete = in_fifo0_data_dout[1:0] == CMD_DELETE;
            sel_insert = in_fifo0_data_dout[1:0] == CMD_INSERT;
            sel_reset  = in_fifo0_data_dout[1:0] == CMD_RESET;
            if (!in_fifo_nearly_full && !in_fifo0_empty) begin
               set_hash_vld = 1;
               in_fifo0_rd_en = 1;
               if (in_fifo0_ctrl_dout == LAST_WORD)
                  next_instate = WAIT_FOR_FIFO;
            end
         end
         INWORD0: begin
            if (!in_fifo_nearly_full && !in_fifo1_empty) begin
               set_hash_vld = 1;
               in_wr          = 1;
               in_fifo1_rd_en = 1;
               inword0_we = 1;
               next_instate = INWORD1;
            end
         end
         INWORD1: begin
            if (!in_fifo_nearly_full && !in_fifo1_empty) begin
               in_fifo1_rd_en = 1;
               in_wr          = 1;
               inword1_we = 1;
               next_instate = INWORD2;
            end
         end
         INWORD2: begin
            if (!in_fifo_nearly_full && !in_fifo1_empty) begin
               in_fifo1_rd_en = 1;
               in_wr          = 1;
               inword2_we = 1;
               next_instate = INWORD3;
            end
         end
         INWORD3: begin
            if (!in_fifo_nearly_full && !in_fifo1_empty) begin
               in_fifo1_rd_en = 1;
               in_wr          = 1;
               inword3_we = 1;
               if (in_fifo1_ctrl_dout == 'hff)
                  next_instate = WAIT_FOR_FIFO;
            end
         end
      endcase // case(state)
   end // always @ (*) 
   
   always @(posedge clk) begin
      if (reset)
          instate <= WAIT_FOR_FIFO;
      else
          instate <= next_instate;
   end

   always @(posedge clk) begin
      if (reset) begin
         src_ip_addr     <= 'h0;
         dst_ip_addr     <= 'h0;
         iface           <= 'h0;
         oface           <= 'h0;
         byte_count      <= 'h0;
         src_port        <= 'h0;
         dst_port        <= 'h0;
         tcp_flags       <= 'h0;
         protocol        <= 'h0;
         l3proto_ok      <= 'h0;
         tos             <= 'h0;
         ttl             <= 'h0;
      end
      else begin
         //if (inword0_we) begin
         //end
         if (inword1_we) begin
            tos             <= in_fifo1_data_dout[63:56];
            byte_count      <= in_fifo1_data_dout[55:40];
            ttl             <= in_fifo1_data_dout[39:32];
            tcp_flags       <= in_fifo1_data_dout[15:8];
         end
         if (inword2_we) begin
            src_ip_addr     <= in_fifo1_data_dout[63:32];
            dst_ip_addr     <= in_fifo1_data_dout[31:0];
         end
         if (inword3_we) begin
            src_port        <= in_fifo1_data_dout[63:48];
            dst_port        <= in_fifo1_data_dout[47:32];
            iface           <= in_fifo1_data_dout[31:24];
            protocol        <= in_fifo1_data_dout[23:16];
            oface           <= in_fifo1_data_dout[15:8];
            l3proto_ok      <= in_fifo1_data_dout[7:0];
         end
   
      end 
   end 

   always @(posedge clk) begin
      if (reset) begin
         reg_cmddelete <= 'h0;
         reg_cmdinsert <= 'h0;
         reg_cmdreset  <= 'h0;
      end
      else
         if (set_hash_vld) begin
            reg_cmddelete <= sel_delete;
            reg_cmdinsert <= sel_insert;
            reg_cmdreset  <= sel_reset;
            
            reg_addr    <= in_data[DATA_WIDTH/2 - 1: DATA_WIDTH/2 - (LOG2NUMLINES+LOG2HASHESPERLINE)];
            reg_hash    <= in_data[DATA_WIDTH-1:DATA_WIDTH-HASHWIDTH];
            reg_routing <= in_data[4+ROUTEWIDTH-1:4];
         end
   end
   
   assign in_ctrl = in_fifo1_ctrl_dout;
   assign in_data = (sel_delete || sel_insert || sel_reset)? in_fifo0_data_dout : in_fifo1_data_dout;
   

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input_fifo
        (.din ({in_ctrl,in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );

   genvar i;
   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_HASH_DATA
      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(HASHWIDTH)
      )
      memhash
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (hash_mem_wea[i]),
        .addra  (hash_mem_addra[i]),
        .ina    (hash_mem_ina[i]),
        .outa   (hash_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (hash_mem_addrb[i]),
        .outb   (hash_mem_outb[i])
      );

      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(STATWIDTH)
      )
      memstat
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (stat_mem_wea[i]),
        .addra  (stat_mem_addra[i]),
        .ina    (stat_mem_ina[i]),
        .outa   (stat_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (stat_mem_addrb[i]),
        .outb   (stat_mem_outb[i])
      );
      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(ROUTEWIDTH)
      )
      memroute
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (route_mem_wea[i]),
        .addra  (route_mem_addra[i]),
        .ina    (route_mem_ina[i]),
        .outa   (route_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (route_mem_addrb[i]),
        .outb   (route_mem_outb[i])
      );


      always @(posedge clk) begin
         hash_mem_addra[i] = line_addr;
         stat_mem_addra[i] = stat_line_addr;
         route_mem_addra[i] = stat_line_addr;
      end

      always @(posedge clk) begin
         reg_hash_mem_outa[i] <= hash_mem_outa[i];
      end
     
      always @(posedge clk) begin
         reg_stat_mem_outa[i] <= stat_mem_outa[i];
      end

      always @(posedge clk) begin
         reg_route_mem_outa[i] <= route_mem_outa[i];
      end
 
   end
   endgenerate

   always @(posedge clk) begin
      if (reset) begin
         hash_mem_wea = 0;
         stat_mem_wea = 0;
         route_mem_wea = 0;
      end
      else begin
         hash_mem_wea = mem_wr_mask & {HASHESPERLINE{hash_wr_en}};
         stat_mem_wea = (mem_wr_mask & {HASHESPERLINE{stat_wr_en}}) | {HASHESPERLINE{init}};
         route_mem_wea = (mem_wr_mask & {HASHESPERLINE{route_wr_en}}) | {HASHESPERLINE{init}};
      end
      hash_mem_ina[0]               = (delete)? reg_hash_mem_outa[1] : reg_hash;
      hash_mem_ina[HASHESPERLINE-1] = (delete)? 0 : reg_hash_mem_outa[HASHESPERLINE-2];

      stat_mem_ina[0]               = (delete && !init)? reg_stat_mem_outa[1] : reg_stat;
      stat_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_stat_mem_outa[HASHESPERLINE-2];

      route_mem_ina[0]               = (delete && !init)? reg_route_mem_outa[1] : reg_route;
      route_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_route_mem_outa[HASHESPERLINE-2];
 
   end

   generate
   for (i=1; i < HASHESPERLINE - 1 ; i = i + 1) begin: GEN_HASH_LOGIC

      always @(posedge clk) begin
         if (delete)
            hash_mem_ina[i] = reg_hash_mem_outa[i + 1];
         else 
            hash_mem_ina[i] = reg_hash_mem_outa[i - 1];
      end

      always @(posedge clk) begin
         if (init)
            stat_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               stat_mem_ina[i] = reg_stat_mem_outa[i + 1];
            else 
               // shift right 
               stat_mem_ina[i] = reg_stat_mem_outa[i - 1];
         end
      end

      always @(posedge clk) begin
         if (init)
            route_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               route_mem_ina[i] = reg_route_mem_outa[i + 1];
            else 
               // shift right 
               route_mem_ina[i] = reg_route_mem_outa[i - 1];
         end
      end
 
   end
   endgenerate
 
/*
   assign stat_mem_wea = (mem_wr_mask & {HASHESPERLINE{stat_wr_en}}) | {HASHESPERLINE{init}};
   assign hash_mem_wea = mem_wr_mask & {HASHESPERLINE{hash_wr_en}};
   always @(delete, init, reg_stat_mem_outa[1], reg_stat_mem_outa[HASHESPERLINE-1], reg_stat) begin
      stat_mem_ina[0]               = (delete && !init)? reg_stat_mem_outa[1] : reg_stat;
      stat_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_stat_mem_outa[HASHESPERLINE-2];
   end

   always @(delete, init, reg_hash_mem_outa[1], reg_hash_mem_outa[HASHESPERLINE-1], reg_hash) begin
      hash_mem_ina[0]               = (delete)? reg_hash_mem_outa[1] : reg_hash;
      hash_mem_ina[HASHESPERLINE-1] = (delete)? 0 : reg_hash_mem_outa[HASHESPERLINE-2];
   end

   generate
   for (i=1; i < HASHESPERLINE - 1 ; i = i + 1) begin: GEN_HASH_LOGIC

      always @(init, delete, reg_stat_mem_outa[i+1], reg_stat_mem_outa[i-1]) begin
         if (init)
            stat_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               stat_mem_ina[i] = reg_stat_mem_outa[i + 1];
            else 
               // shift right 
               stat_mem_ina[i] = reg_stat_mem_outa[i - 1];
         end
      end

      always @(delete, reg_hash_mem_outa[i+1], reg_hash_mem_outa[i-1]) begin
         if (delete)
            hash_mem_ina[i] = reg_hash_mem_outa[i + 1];
         else 
            hash_mem_ina[i] = reg_hash_mem_outa[i - 1];
      end
   end
   endgenerate
*/
  
   assign line_addr = reg_addr[LOG2NUMLINES+LOG2HASHESPERLINE-1:LOG2HASHESPERLINE];

   // find matching item
   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_HASH_COMPARATORS
      always @(hash_mem_outa[i], reg_hash, stat_mem_outa[i]) begin
//      always @(posedge clk) begin
         if (hash_mem_outa[i] == reg_hash && stat_mem_outa[i] != 'h0)
            hash_match[i] <= 1;
         else
            hash_match[i] <= 0;
      end
   end
   endgenerate
   
   always @(posedge clk) begin
      reg_matched    <= | hash_match;
   end

   assign matched      = | hash_match;

   // find last valid item
   always @(stat_mem_outb[HASHESPERLINE-1]) begin
      stat_last[HASHESPERLINE - 1] = (stat_mem_outb[HASHESPERLINE - 1] != 'h0);
   end

   generate
   for (i=0; i < HASHESPERLINE - 1 ; i = i + 1) begin: GEN_LAST_BIT
      always @(stat_last[i+1], stat_mem_outb[i]) begin
    //  always @(posedge clk) begin
         stat_last[i] = (!stat_last[i+1]) && (stat_mem_outb[i] != 'h0);
      end
   end
   endgenerate

   always @(posedge clk) begin
      line_full   =  ( stat_mem_outa[HASHESPERLINE - 1] != 'h0); 
   end
   
   // generate binary address of matching and last field in line
   integer j;
   always @(posedge clk) begin
      hash_match_addr = 'h0;
      stat_last_addr = 'h0;
      for (j=0; j < HASHESPERLINE ; j = j + 1) begin: GEN_HASH_MATCH_ADDR
         if (hash_match[j]) begin
            hash_match_addr = j;
         end
         if (stat_last[j]) begin
            stat_last_addr = j;
         end
      end
   end

   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_WR_MASK
      always @(*) begin
         mem_wr_mask[i] = 1;
         if (delete) begin 
            if (hash_match_addr > i )
               mem_wr_mask[i] = 0;
            else
               mem_wr_mask[i] = 1;
         end
         else begin
            if (reg_matched)  begin
               if (hash_match_addr < i )
                  mem_wr_mask[i] = 0;
               else
                  mem_wr_mask[i] = 1;
            end
         end
      end
   end
   endgenerate

   always @(*) begin 
      tcp_flags_syn = tcp_flags[SYNFLAG] && (protocol == PROTOTCP);
      tcp_flags_fin = tcp_flags[FINFLAG] && (protocol == PROTOTCP);
      tcp_flags_rst = tcp_flags[RSTFLAG] && (protocol == PROTOTCP);
   end

   assign delete = reg_matched && reg_cmddelete;
/* 
   assign delete = reg_matched &&
                   (
                   reg_cmddelete ||
                   (reg_stat_mem_outa[hash_match_addr] == FINMARK && !tcp_flags_fin) ||
                   tcp_flags_rst
                   );
 */  
//   always @(reset, init, tcp_flags, matched, stat_mem_outa[hash]) begin
/*   always @(posedge clk) begin
      if (reset || init ) begin
         reg_stat <= 'b0;
      end
      else begin
         if (tcp_flags_fin)) begin
            reg_stat <= FINMARK;
         end
         else begin
            if (!matched || tcp_flags_syn)
               reg_stat <= 1;
            else begin
               if (reg_stat_mem_outa[hash_match_addr] < (FINMARK - 1))
                  reg_stat <= stat_mem_outa[hash_match_addr] + 1;
               else
                  reg_stat <= stat_mem_outa[hash_match_addr];
            end
         end
      end
   end
*/
/*
   assign reg_stat = (reset || init)? 0 : 
                     (tcp_flags_fin)? FINMARK :
                     (!reg_matched || tcp_flags_syn)? 1 :
                     (reg_stat_mem_outa[hash_match_addr] < (FINMARK - 1))? reg_stat_mem_outa[hash_match_addr] + 1 :
                      reg_stat_mem_outa[hash_match_addr];
*/
    assign reg_stat = (reset || init)? 0 :
                      ( 
                         (tcp_flags_fin || tcp_flags_syn || 
                         tcp_flags_rst || !reg_matched || reg_cmdreset)? 1 :
                         (
                            (reg_stat_mem_outa[hash_match_addr] < FINMARK)? 
                             reg_stat_mem_outa[hash_match_addr] + 1 :
                             reg_stat_mem_outa[hash_match_addr]
                         )
                      );

                   
    assign reg_route = (reset || init )? 0 : 
                       ((reg_cmdinsert)? reg_routing:
                        ((!reg_matched)? 0 :
                       reg_route_mem_outa[hash_match_addr]));
                      

   always @(*) begin
      next_outstate = outstate;
      rst_hash_vld  = 0;
      hash_wr_en    = 0;
      stat_wr_en    = 0;
      route_wr_en   = 0;
      sel_data_hash = 0;

      init           = 0;
      stat_line_addr = line_addr;

      occupation_up   = 0;
      occupation_down = 0;
      replaced_we     = 0;

      cmd           = 'h0;
      in_fifo_rd_en = 0;
      out_wr        = 0;
      out_ctrl      = in_fifo_ctrl_dout;
      case(outstate)
         INIT_MEM: begin
            stat_line_addr = cnt_init;
            init = 1;
            stat_wr_en = 1;
            route_wr_en = 1;
            if (cnt_init == (NUMLINES - 1) )
               next_outstate = WAIT_FOR_HASH;
         end
         WAIT_FOR_HASH: begin
            if ((reg_cmddelete || reg_cmdinsert || reg_cmdreset)&& hash_vld[0])
               next_outstate = WAIT_FOR_DELETE;
            else 
               if (hash_vld[0])
                  next_outstate = WAIT_FOR_INDEX;
         end
         WAIT_FOR_DELETE: begin
            if (lookedup) begin
               rst_hash_vld = 1;
               next_outstate = WAIT_FOR_HASH;
               if (reg_matched && (reg_cmddelete || reg_cmdreset)) begin
                  hash_wr_en = 1;
                  stat_wr_en = 1;
                  route_wr_en = 1;
               end
               if (reg_matched && reg_cmddelete) begin
                  occupation_down = 1;
               end
               if (reg_cmdinsert) begin
                  hash_wr_en = 1;
                  stat_wr_en = 1;
                  route_wr_en = 1;
               end
               if (!reg_matched && reg_cmdinsert)  begin
                  if (line_full)
                     replaced_we   = 1;
                  else
                     occupation_up = 1;
               end
 
            end
         end
         WAIT_FOR_INDEX: begin
            sel_data_hash = 1;
            out_ctrl      = 'h0;
            if (lookedup && instate == WAIT_FOR_FIFO) begin
               if (out_rdy && !in_fifo_empty) begin
                  rst_hash_vld  = 1;
                  in_fifo_rd_en = 1;
                  out_wr        = 1;   
                  next_outstate = WRITE_REST;

                  if (dontcare) begin
                     cmd = CMD_DONTCARE;
                  end
                  else begin
                     stat_wr_en = 1;
                     hash_wr_en = 1;
                     route_wr_en = 1;
                  
                     if (reg_matched) begin
                        cmd = CMD_UPDATE;
                     end
                     else begin 
                        cmd = CMD_NEW;
                        if (line_full)
                           replaced_we   = 1;
                        else 
                           occupation_up = 1;
                     end
                  end
               end
            end
         end
         WRITE_REST: begin
            if (out_rdy && !in_fifo_empty) begin
               out_wr = 1;   
               in_fifo_rd_en = 1;
               if (in_fifo_ctrl_dout == LAST_WORD)
                  next_outstate = WAIT_FOR_HASH;
            end
         end
      endcase // case(state)
   end // always @ (*) 
   
   always @(posedge clk) begin
      if (reset)
          outstate <= INIT_MEM;
      else
          outstate <= next_outstate;
   end
   
   always @(posedge clk) begin
      if (reset) begin
          cnt_init <= 'b0;
          reg_cnt_init <= 'b0;
       end
       else begin
          reg_cnt_init <= cnt_init;
          if (init)
             cnt_init <= cnt_init + 1;
       end
   end
   
 
   assign occupation_we = occupation_up ^ occupation_down;

   assign occupation = cnt_occupation;

   always @(posedge clk) begin
      if (reset) begin
          cnt_occupation <= 'b0;
       end
       else begin
          if (occupation_we) begin
             if (occupation_up)
                cnt_occupation<= cnt_occupation + 1;
             else
                cnt_occupation <= cnt_occupation - 1;
          end
       end
   end

   always @(posedge clk) begin
      if (reset) begin
          cnt_replaced <= 'b0;
       end
       else begin
          if (replaced_we) begin
             cnt_replaced <= cnt_replaced + 1;
          end
       end
   end
   assign replaced = cnt_replaced;



   generate
   for (i=0; i < LOOKUP_LATENCY ; i = i + 1) begin: GEN_HASH_LATENCY
      always @(posedge clk) begin
         if (rst_hash_vld || reset) begin 
            hash_vld[i] <= 0;
         end
         else begin 
            if (i == 0) begin
               if (set_hash_vld)
                  hash_vld[0] <= 1;
            end
            else 
               hash_vld[i] <= hash_vld[i-1];
         end
      end
   end
   endgenerate
  
   assign lookedup = hash_vld[LOOKUP_LATENCY-1];
   assign dontcare = ((protocol != PROTOUDP) && (protocol != PROTOTCP)) ||
                     (oface != 'h00) || iface[0] || !l3proto_ok[0];
   
   assign lookup_data = (reg_matched)? {27'b0, reg_route_mem_outa[hash_match_addr], 1'b0, reg_stat_mem_outa[hash_match_addr]}:32'b0;
   assign out_data = (sel_data_hash)? {28'b0, cmd, lookup_data} : in_fifo_data_dout;

endmodule 


/* vim:set shiftwidth=3 softtabstop=3 expandtab: */
