///////////////////////////////////////////////////////////////////////////////
//
// Module: flowlookup.v
// Description: 
//
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

module flowlookup_hc #(
   parameter HASHESPERLINE = 4,
   parameter NUMLINES  = 2**8,
   parameter HASHWIDTH = 32,
   parameter DATA_WIDTH = 64,
   parameter CTRL_WIDTH = DATA_WIDTH/8
   )
   (

   input  [DATA_WIDTH-1:0]            in_data0,
   input  [CTRL_WIDTH-1:0]            in_ctrl0,
   input                              in_wr0,
   output                             in_rdy0,

   input  [DATA_WIDTH-1:0]            in_data1,
   input  [CTRL_WIDTH-1:0]            in_ctrl1,
   input                              in_wr1,
   output                             in_rdy1,

   output [DATA_WIDTH-1:0]            out_data,
   output reg [CTRL_WIDTH-1:0]        out_ctrl,
   output reg                         out_wr,
   input                              out_rdy,

   input                              bloominit,
   input  [DATA_WIDTH-1:0]            bloomstathash,
   output reg [DATA_WIDTH-1:0]        searchedbloomstat,
 
   input                              clk,
   input                              reset
);
   
  function integer log2;
      input integer number;
      begin
         log2=0;
         while(2**log2<number) begin
            log2=log2+1;
         end
      end
   endfunction // log2

   localparam STATWIDTH = 3;
   localparam ROUTEWIDTH = 3;
   localparam BLOOMWIDTH = 1;
//   localparam LOG2BLOOMSIZE = 16;
   localparam LOG2BLOOMSIZE = 2;
   localparam BLOOMSIZE = 2**LOG2BLOOMSIZE;
   localparam BLOOMSTATWIDTH = 16;
   
   localparam WAIT_FOR_FIFO  = 0;
   localparam DELETE_CMDADDR = 1;
   localparam INWORD0 = 2;
   localparam INWORD1 = 3;
   localparam INWORD2 = 4;
   localparam INWORD3 = 5;

   localparam LOOKUP_LATENCY = 3;
   
   localparam CMD_DONTCARE = 4'h0;
   localparam CMD_NEW    = 4'h1;
   localparam CMD_UPDATE = 4'h2;

   localparam CMD_DELETE = 2'h0;
   localparam CMD_INSERT = 2'h1;
   localparam CMD_ROUTE  = 2'h2;



   localparam LAST_WORD  = 8'hff;

   localparam INIT_MEM         = 0;
   localparam WAIT_FOR_HASH    = 1;
   localparam WAIT_FOR_DELETE  = 2;
   localparam WAIT_FOR_INDEX   = 3;
   localparam WAIT_FOR_INSERT  = 4;

   localparam INITSBLOOM = 0;
   localparam IDLEBLOOM1 = 1;
   localparam WAITBLOOM1 = 2;
   localparam IDLEBLOOM2 = 3;
   localparam WAITBLOOM2 = 4;
   localparam WAITBLOOMCMD = 5;


   localparam PROTOICMP  = 8'h01;
   localparam PROTOTCP   = 8'h06;
   localparam PROTOUDP   = 8'h11;

   localparam RSTFLAG    = 2;
   localparam FINFLAG    = 0;
   localparam SYNFLAG    = 1;
   localparam FINMARK    = {STATWIDTH{1'b1}};

   parameter  NUMOUTSTATES = 6;

   localparam LOG2NUMLINES = log2(NUMLINES);
   localparam LOG2HASHESPERLINE = log2(HASHESPERLINE);

   reg [3:0] instate, next_instate;
   reg [3:0] outstate, next_outstate;
   reg [2:0] bloomstate, next_bloomstate;

   // fifo 0
   wire [CTRL_WIDTH-1:0] in_fifo0_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo0_data_dout;
   wire in_fifo0_nearly_full;
   wire in_fifo0_empty;
   reg  in_fifo0_rd_en;

   // fifo 1 
   wire [CTRL_WIDTH-1:0] in_fifo1_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo1_data_dout;
   wire in_fifo1_nearly_full;
   wire in_fifo1_empty;
   reg  in_fifo1_rd_en;
   
   // fifo shared
   wire [CTRL_WIDTH-1:0] in_ctrl;
   wire [DATA_WIDTH-1:0] in_data;
   reg  in_wr;
   wire [CTRL_WIDTH-1:0] in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo_data_dout;
   wire in_fifo_nearly_full;
   wire in_fifo_empty;
   reg  in_fifo_rd_en;
   wire in_rdy;
   

   // output of FSM0 (instate)
   reg  sel_delete;
   reg  sel_insert;
   reg  sel_route;
   reg  set_hash_vld;
   reg  sel_data_hash;
   wire dontcare;

   wire [31:0] lookup_data;

   // memory definition   
/*   
   wire [HASHESPERLINE-1:0]  hash_mem_wea;
   wire [HASHESPERLINE-1:0]  stat_mem_wea;
*/
   reg [HASHESPERLINE-1:0]  hash_mem_wea;
   reg [HASHESPERLINE-1:0]  stat_mem_wea;
   reg [HASHESPERLINE-1:0]  route_mem_wea;
   wire  bloom_mem_wea;
   reg [HASHESPERLINE-1:0]  bloomstat_mem_wea;

   wire [LOG2NUMLINES-1:0] hash_mem_addra [HASHESPERLINE-1:0];
   reg  [HASHWIDTH-1:0]      hash_mem_ina   [HASHESPERLINE-1:0];
   wire [HASHWIDTH-1:0]      hash_mem_outa  [HASHESPERLINE-1:0];
   reg [HASHWIDTH-1:0]      reg_hash_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0] hash_mem_addrb [HASHESPERLINE-1:0];
   wire [HASHWIDTH-1:0]      hash_mem_outb  [HASHESPERLINE-1:0];
       
   wire [LOG2NUMLINES-1:0] stat_mem_addra [HASHESPERLINE-1:0];
   reg  [STATWIDTH-1:0]      stat_mem_ina   [HASHESPERLINE-1:0];
   wire [STATWIDTH-1:0]      stat_mem_outa  [HASHESPERLINE-1:0];
   reg  [STATWIDTH-1:0]      reg_stat_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0] stat_mem_addrb [HASHESPERLINE-1:0];
   wire [STATWIDTH-1:0]      stat_mem_outb  [HASHESPERLINE-1:0];

   wire [LOG2NUMLINES-1:0]   route_mem_addra [HASHESPERLINE-1:0];
   reg  [ROUTEWIDTH-1:0]     route_mem_ina   [HASHESPERLINE-1:0];
   wire [ROUTEWIDTH-1:0]     route_mem_outa  [HASHESPERLINE-1:0];
   reg  [ROUTEWIDTH-1:0]     reg_route_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0]   route_mem_addrb [HASHESPERLINE-1:0];
   wire [ROUTEWIDTH-1:0]     route_mem_outb  [HASHESPERLINE-1:0];

   wire [LOG2NUMLINES-1:0]   bloomstat_mem_addra [HASHESPERLINE-1:0];
   reg  [BLOOMSTATWIDTH-1:0] bloomstat_mem_ina   [HASHESPERLINE-1:0];
   wire [BLOOMSTATWIDTH-1:0] bloomstat_mem_outa  [HASHESPERLINE-1:0];
   reg  [BLOOMSTATWIDTH-1:0] reg_bloomstat_mem_outa  [HASHESPERLINE-1:0];
   
   wire [LOG2NUMLINES-1:0]   bloomstat_mem_addrb [HASHESPERLINE-1:0];
   wire [BLOOMSTATWIDTH-1:0] bloomstat_mem_outb  [HASHESPERLINE-1:0];


   wire [LOG2BLOOMSIZE-1:0]  bloom_mem_addra;
   wire  [BLOOMWIDTH-1:0]    bloom_mem_ina  ;
   reg  [BLOOMWIDTH-1:0]     bloom_ina  ;
   wire [BLOOMWIDTH-1:0]     bloom_mem_outa ;
   
   wire [LOG2BLOOMSIZE-1:0]  bloom_mem_addrb;
   wire [BLOOMWIDTH-1:0]     bloom_mem_outb ;



   reg [1:0]                      reg_flags;
   reg [ROUTEWIDTH-1:0]      reg_routing;
   wire [ROUTEWIDTH-1:0]     reg_route;

   //reg [STATWIDTH-1:0]      reg_stat;
   wire [STATWIDTH-1:0]      reg_stat;
   reg  [BLOOMWIDTH-1:0]     reg_bloom;
   wire [BLOOMSTATWIDTH-1:0] reg_bloomstat;
  

   reg [HASHESPERLINE - 1 : 0]      hash_match;
   reg [LOG2HASHESPERLINE - 1 : 0]  hash_match_addr;
   reg [HASHESPERLINE - 1 : 0]      stat_last;
   reg [LOG2HASHESPERLINE - 1 : 0]  stat_last_addr;
   reg [HASHESPERLINE - 1 : 0]      mem_wr_mask;


   // registers
   reg  reg_cmddelete;
   reg  reg_cmdinsert;
   reg  reg_cmdroute;
   reg [LOG2NUMLINES+LOG2HASHESPERLINE-1:0] reg_addr;
   wire [LOG2NUMLINES+LOG2HASHESPERLINE-1:0] reg_addrb;
   reg [HASHWIDTH-1:0] reg_hash;

   //logic
   wire delete;
   wire lookedup;
   wire matched;
   reg  reg_matched;
   reg  reg_matchedb;
   reg  init;
   reg  [LOG2NUMLINES-1:0] cnt_init;
   reg  [LOG2NUMLINES-1:0] reg_cnt_init;
   wire [LOG2NUMLINES-1:0] line_addr;
   reg  [LOG2NUMLINES-1:0] stat_line_addr;

   // output of FSM1 (outstate)
   reg  rst_hash_vld;
   reg  hash_wr_en;
   reg  stat_wr_en;
   reg  route_wr_en;
   reg  [3:0] cmd;
   reg  [LOOKUP_LATENCY-1:0] hash_vld;

// Bloom
   reg  [LOG2BLOOMSIZE-1:0] cnt_initbloom;
   reg  [LOG2BLOOMSIZE-1:0] reg_cnt_initbloom;

   reg  bloomstat_wr_en;
   reg  bloom_wr_en;

   reg  bloomaddr_wr;
   reg  bloomstataddr_wr;
   reg  [LOG2BLOOMSIZE-1:0] reg_bloomaddr;
   reg  [LOG2NUMLINES-1:0]  reg_bloomstataddr;

   reg  [HASHESPERLINE - 1 : 0] reg_oldmask;
   reg  [HASHESPERLINE - 1 : 0] bloommask;
   reg  reg_oldmask_set;

   reg  reg_oldmatch_rst;
   reg  reg_oldmatch_set;
   reg  reg_oldmatch;

   wire  initbloom;
   reg  initbloom_rst;

   
   wire  [LOG2NUMLINES-1:0]  searchedaddr;

   reg [HASHESPERLINE - 1 : 0]      hash_matchb;
   reg [LOG2HASHESPERLINE - 1 : 0]  hash_match_addrb;
   wire [HASHWIDTH-1:0]              searchedhash;

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input0_fifo
        (.din ({in_ctrl0,in_data0}),     // Data in
         .wr_en (in_wr0),               // Write enable
         .rd_en (in_fifo0_rd_en),       // Read the next word
         .dout ({in_fifo0_ctrl_dout, in_fifo0_data_dout}),
         .full (),
         .nearly_full (in_fifo0_nearly_full),
         .empty (in_fifo0_empty),
         .reset (reset),
         .clk (clk)
         );   
   
   assign in_rdy0 = !in_fifo0_nearly_full;

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input1_fifo
        (.din ({in_ctrl1,in_data1}),     // Data in
         .wr_en (in_wr1),               // Write enable
         .rd_en (in_fifo1_rd_en),       // Read the next word
         .dout ({in_fifo1_ctrl_dout, in_fifo1_data_dout}),
         .full (),
         .nearly_full (in_fifo1_nearly_full),
         .empty (in_fifo1_empty),
         .reset (reset),
         .clk (clk)
         );   
   
   assign in_rdy1 = !in_fifo1_nearly_full;

   always @(*) begin
      next_instate   = instate;

      sel_delete = 0;
      sel_insert = 0;
      sel_route  = 0;

      in_fifo0_rd_en = 0;
      in_fifo1_rd_en = 0;
      in_wr          = 0;
      set_hash_vld   = 0;

      case(instate)
         WAIT_FOR_FIFO : begin
            if ( (!hash_vld[0] || rst_hash_vld) && !init && !initbloom) begin
               if (!in_fifo0_empty) begin
                  next_instate = DELETE_CMDADDR;
               end
               else begin
                  if (!in_fifo1_empty) begin
                     next_instate = INWORD0;
                  end
               end
            end
         end
         DELETE_CMDADDR: begin
            sel_delete = in_fifo0_data_dout[1:0] == CMD_DELETE;
            sel_insert = in_fifo0_data_dout[1:0] == CMD_INSERT;
            sel_route  = in_fifo0_data_dout[1:0] == CMD_ROUTE;
            if (!in_fifo_nearly_full && !in_fifo0_empty) begin
               set_hash_vld = 1;
               in_fifo0_rd_en = 1;
               if (in_fifo0_ctrl_dout == LAST_WORD)
                  next_instate = WAIT_FOR_FIFO;
            end
         end
         INWORD0: begin
            if (!in_fifo_nearly_full && !in_fifo1_empty) begin
               set_hash_vld = 1;
               in_wr          = 1;
               in_fifo1_rd_en = 1;
               next_instate = WAIT_FOR_FIFO;
            end
         end
      endcase // case(state)
   end // always @ (*) 
   
   always @(posedge clk) begin
      if (reset)
          instate <= WAIT_FOR_FIFO;
      else
          instate <= next_instate;
   end


   always @(posedge clk) begin
      if (reset) begin
         reg_cmddelete <= 'h0;
         reg_cmdinsert <= 'h0;
         reg_cmdroute  <= 'h0;
      end
      else
         if (set_hash_vld) begin
            reg_cmddelete <= sel_delete;
            reg_cmdinsert <= sel_insert;
            reg_cmdroute  <= sel_route;
            
            reg_addr    <= in_data[DATA_WIDTH/2 - 1: DATA_WIDTH/2 - (LOG2NUMLINES+LOG2HASHESPERLINE)];
            reg_hash    <= in_data[DATA_WIDTH-1:DATA_WIDTH-HASHWIDTH];
            reg_routing <= in_data[4+ROUTEWIDTH-1:4];
            reg_flags   <= in_data[9:8];
         end
   end
   
   assign in_ctrl = in_fifo1_ctrl_dout;
   assign in_data = (sel_delete || sel_insert || sel_route )? in_fifo0_data_dout : in_fifo1_data_dout;
   
   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input_fifo
        (.din ({in_ctrl,in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );

   genvar i;
   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_HASH_DATA
      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(HASHWIDTH)
      )
      memhash
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (hash_mem_wea[i]),
        .addra  (hash_mem_addra[i]),
        .ina    (hash_mem_ina[i]),
        .outa   (hash_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (hash_mem_addrb[i]),
        .outb   (hash_mem_outb[i])
      );

      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(STATWIDTH)
      )
      memstat
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (stat_mem_wea[i]),
        .addra  (stat_mem_addra[i]),
        .ina    (stat_mem_ina[i]),
        .outa   (stat_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (stat_mem_addrb[i]),
        .outb   (stat_mem_outb[i])
      );
      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(ROUTEWIDTH)
      )
      memroute
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (route_mem_wea[i]),
        .addra  (route_mem_addra[i]),
        .ina    (route_mem_ina[i]),
        .outa   (route_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (route_mem_addrb[i]),
        .outb   (route_mem_outb[i])
      );

      blockram #(
        .RAM_ADDR_BITS(LOG2NUMLINES),
        .RAM_WIDTH(BLOOMSTATWIDTH)
      )
      membloomstat
      (
        .clka   (clk),
        .ena    (1'b1),
        .wea    (bloomstat_mem_wea[i]),
        .addra  (bloomstat_mem_addra[i]),
        .ina    (bloomstat_mem_ina[i]),
        .outa   (bloomstat_mem_outa[i]),

        .clkb   (clk),
        .enb    (1'b1),
        .addrb  (bloomstat_mem_addrb[i]),
        .outb   (bloomstat_mem_outb[i])
      );


      assign hash_mem_addra[i]  = line_addr;
      assign stat_mem_addra[i]  = stat_line_addr;
      assign route_mem_addra[i] = stat_line_addr;
      assign bloomstat_mem_addra[i] = 
             (init)? cnt_init[LOG2NUMLINES-1:0] : reg_bloomstataddr;

      assign hash_mem_addrb[i]      = searchedaddr;
      assign stat_mem_addrb[i]      = searchedaddr;
      assign route_mem_addrb[i]     = searchedaddr;
      assign bloomstat_mem_addrb[i] = searchedaddr;

      always @(posedge clk) begin
         reg_hash_mem_outa[i] <= hash_mem_outa[i];
      end
     
      always @(posedge clk) begin
         reg_stat_mem_outa[i] <= stat_mem_outa[i];
      end

      always @(posedge clk) begin
         reg_route_mem_outa[i] <= route_mem_outa[i];
      end

      always @(posedge clk) begin
         reg_bloomstat_mem_outa[i] <= bloomstat_mem_outa[i];
      end
 
   end
   endgenerate

   blockram #(
     .RAM_ADDR_BITS(LOG2BLOOMSIZE),
     .RAM_WIDTH(BLOOMWIDTH)
   )
   membloom
   (
     .clka   (clk),
     .ena    (1'b1),
     .wea    (bloom_mem_wea),
     .addra  (bloom_mem_addra),
     .ina    (bloom_mem_ina),
     .outa   (bloom_mem_outa),

     .clkb   (clk),
     .enb    (1'b1),
     .addrb  (bloom_mem_addrb),
     .outb   (bloom_mem_outb)
   );
   assign bloom_mem_addra = (initbloom)? cnt_initbloom : reg_bloomaddr;
   assign bloom_mem_wea   = bloom_wr_en || initbloom;
   assign bloom_mem_ina   = ~initbloom;

//   always @(posedge clk) begin
   always @(*) begin
      if (reset || initbloom)
         reg_bloom <= 'h0;
      else
         reg_bloom <= bloom_mem_outa;
   end
 
   always @(posedge clk) begin
      if (reset) begin
          cnt_initbloom     <= 'h0;
          reg_cnt_initbloom <= 'h0;
       end
       else begin
          if (initbloom) begin
             cnt_initbloom     <= cnt_initbloom + 1;
             reg_cnt_initbloom <= cnt_initbloom;
          end
       end
   end

   assign initbloom = bloominit || (reg_cnt_initbloom < (BLOOMSIZE-1));


   always @(*) begin
      next_bloomstate = bloomstate;
      bloomaddr_wr     = 0;
      bloomstataddr_wr = 0;
      bloomstat_wr_en  = 0;
      bloom_wr_en      = 0;
      reg_oldmask_set  = 0;
      reg_oldmatch_rst = 0;
      reg_oldmatch_set = 0;
      bloommask        = mem_wr_mask;
      case(bloomstate)
//         INITSBLOOM: begin
      //      bloommask     = 'hF;
      //      initbloom_rst = 0;
      //      initbloom   = 1;
      //      bloom_wr_en = 1;
      //      bloom_mem_ina = 0;

 //           if (cnt_initbloom == (BLOOMSIZE-1)) 
 //              next_bloomstate = IDLEBLOOM1;
  //       end
         IDLEBLOOM1: begin

            reg_oldmatch_rst = 1;

            if (set_hash_vld) begin
               if (sel_delete ||
               sel_insert || 
               sel_route) begin 
                  next_bloomstate = WAITBLOOMCMD;
               end 
               else begin
                  bloomaddr_wr     = 1;
                  bloomstataddr_wr = 1;
                  next_bloomstate = WAITBLOOM1;
               end
            end
         end
         WAITBLOOM1 : begin
            if (lookedup) begin
               if (reg_matched) begin
                 reg_oldmatch_set = 1;
                  reg_oldmask_set  = 1;
               end
               next_bloomstate = IDLEBLOOM2;
            end
         end
         IDLEBLOOM2: begin
            if (set_hash_vld) begin
               if (reg_oldmatch)
                  bloomaddr_wr     = 1;
               else 
                  bloomstataddr_wr = 1;

               next_bloomstate = WAITBLOOM2;
            end
         end
         WAITBLOOM2: begin
            if (lookedup) begin
               if ((reg_matched || reg_oldmatch) && 
                   ((!reg_bloom && !reg_flags[1]) || 
                    (reg_flags[1] && reg_flags[0]))) begin
                     bloomstat_wr_en  = 1;
                     if (!reg_flags[1])
                        bloom_wr_en   = 1;
               end
               if (reg_oldmatch) begin
                  bloommask = reg_oldmask;
               end
               next_bloomstate = IDLEBLOOM1;
            end
         end
         WAITBLOOMCMD: begin
            if (lookedup) begin
               if (reg_matched) begin
                  bloomstat_wr_en  = 1;
               end
               next_bloomstate = IDLEBLOOM1;
            end
         end
      endcase
   end

   always @(posedge clk) begin
      if (reset)
         bloomstate = INITSBLOOM;
      else
         bloomstate = next_bloomstate;
   end

   always @(posedge clk) begin
      if (reset)
         reg_oldmask = 'h0;
      else if (reg_oldmask_set) begin
         reg_oldmask = mem_wr_mask;
      end
   end

   always @(posedge clk) begin
      if (reset || reg_oldmatch_rst)
         reg_oldmatch = 0;
      else if (reg_oldmatch_set) begin
         reg_oldmatch = 1;
      end
   end


   always @(posedge clk) begin
      if (reset) begin
         reg_bloomaddr     <= 'h0;
         reg_bloomstataddr <= 'h0;
      end
      else
         if (bloomaddr_wr) begin
            reg_bloomaddr     <= in_data[DATA_WIDTH-1:DATA_WIDTH-LOG2BLOOMSIZE];
         end
         if (bloomstataddr_wr) begin
            reg_bloomstataddr <= in_data[DATA_WIDTH/2 - 1: DATA_WIDTH/2 - (LOG2NUMLINES)];
         end
   end
 

   always @(posedge clk) begin
      if (reset) begin
         hash_mem_wea = 0;
         stat_mem_wea = 0;
         route_mem_wea = 0;
         bloomstat_mem_wea = 0;
      end
      else begin
         hash_mem_wea = mem_wr_mask & {HASHESPERLINE{hash_wr_en}};
         stat_mem_wea = (mem_wr_mask & {HASHESPERLINE{stat_wr_en}}) 
                        | {HASHESPERLINE{init}};
         route_mem_wea = (mem_wr_mask & {HASHESPERLINE{route_wr_en}}) 
                         | {HASHESPERLINE{init}};
         bloomstat_mem_wea 
                       = (bloommask & {HASHESPERLINE{bloomstat_wr_en}}) 
                         | {HASHESPERLINE{init}};
      end
   end

   always @(posedge clk) begin
      hash_mem_ina[0]               = (delete)? reg_hash_mem_outa[1] : reg_hash;
      hash_mem_ina[HASHESPERLINE-1] = (delete)? 0 : reg_hash_mem_outa[HASHESPERLINE-2];

      stat_mem_ina[0]               = (delete && !init)? reg_stat_mem_outa[1] : reg_stat;
      stat_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_stat_mem_outa[HASHESPERLINE-2];

      route_mem_ina[0]               = (delete && !init)? reg_route_mem_outa[1] : reg_route;
      route_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_route_mem_outa[HASHESPERLINE-2];
       bloomstat_mem_ina[0]              = (delete && !init)? reg_bloomstat_mem_outa[1] : reg_bloomstat;
      bloomstat_mem_ina[HASHESPERLINE-1] = (delete || init)? 0 : reg_bloomstat_mem_outa[HASHESPERLINE-2];
 
   end

   generate
   for (i=1; i < HASHESPERLINE - 1 ; i = i + 1) begin: GEN_HASH_LOGIC

      always @(posedge clk) begin
         if (delete)
            hash_mem_ina[i] = reg_hash_mem_outa[i + 1];
         else 
            hash_mem_ina[i] = reg_hash_mem_outa[i - 1];
      end

      always @(posedge clk) begin
         if (init)
            stat_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               stat_mem_ina[i] = reg_stat_mem_outa[i + 1];
            else 
               // shift right 
               stat_mem_ina[i] = reg_stat_mem_outa[i - 1];
         end
      end

      always @(posedge clk) begin
         if (init)
            route_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               route_mem_ina[i] = reg_route_mem_outa[i + 1];
            else 
               // shift right 
               route_mem_ina[i] = reg_route_mem_outa[i - 1];
         end
      end

      always @(posedge clk) begin
         if (init)
            bloomstat_mem_ina[i] = 'h0;
         else begin
            if (delete)
               // shift left when delete
               bloomstat_mem_ina[i] = reg_bloomstat_mem_outa[i + 1];
            else 
               // shift right 
               bloomstat_mem_ina[i] = reg_bloomstat_mem_outa[i - 1];
         end
      end
 
   end
   endgenerate
 
/*
*/
  
   assign line_addr = reg_addr[LOG2NUMLINES+LOG2HASHESPERLINE-1:LOG2HASHESPERLINE];

//-------------------------------------
// Bloom
// find matching item on port b
// --------------------------------------

   assign searchedhash = bloomstathash[DATA_WIDTH-1:DATA_WIDTH-HASHWIDTH];
   assign searchedaddr = reg_addrb[LOG2NUMLINES+LOG2HASHESPERLINE-1:LOG2HASHESPERLINE];

   assign reg_addrb    = bloomstathash[DATA_WIDTH/2 - 1: DATA_WIDTH/2 - (LOG2NUMLINES+LOG2HASHESPERLINE)];

   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_HASH_COMPARATORSB
      always @(hash_mem_outb[i], searchedhash, stat_mem_outb[i]) begin
//      always @(posedge clk) begin
         if (hash_mem_outb[i] == searchedhash && stat_mem_outb[i] != 'h0)
            hash_matchb[i] <= 1;
         else
            hash_matchb[i] <= 0;
      end
   end
   endgenerate

   always @(posedge clk) begin
      reg_matchedb    <= | hash_matchb;
   end


   // generate binary address of matching and last field in line
   integer j;
   always @(posedge clk) begin
      hash_match_addrb = 'h0;
      for (j=0; j < HASHESPERLINE ; j = j + 1) begin: GEN_HASH_MATCH_ADDRB
         if (hash_matchb[j]) begin
            hash_match_addrb = j;
         end
      end
   end

   //always @(*) begin
   always @(posedge clk) begin
      searchedbloomstat <= {searchedhash[15:0], 
                            hash_mem_outb[hash_match_addrb][15:0], 
                            1'b0, stat_mem_outb[hash_match_addrb],   
                            1'b0, bloomstate,
                            8'b0,
                            bloomstat_mem_outb[hash_match_addrb]
                             };
   end

//---------------------------------------------
 
   // find matching item
   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_HASH_COMPARATORS
      always @(hash_mem_outa[i], reg_hash, stat_mem_outa[i]) begin
//      always @(posedge clk) begin
         if (hash_mem_outa[i] == reg_hash && stat_mem_outa[i] != 'h0)
            hash_match[i] <= 1;
         else
            hash_match[i] <= 0;
      end
   end
   endgenerate
   
   always @(posedge clk) begin
      reg_matched    <= | hash_match;
   end

   assign matched      = | hash_match;

   // find last valid item
   always @(stat_mem_outb[HASHESPERLINE-1]) begin
      stat_last[HASHESPERLINE - 1] = (stat_mem_outb[HASHESPERLINE - 1] != 'h0);
   end

   generate
   for (i=0; i < HASHESPERLINE - 1 ; i = i + 1) begin: GEN_LAST_BIT
      always @(stat_last[i+1], stat_mem_outb[i]) begin
    //  always @(posedge clk) begin
         stat_last[i] = (!stat_last[i+1]) && (stat_mem_outb[i] != 'h0);
      end
   end
   endgenerate
   
   // generate binary address of matching and last field in line
   always @(posedge clk) begin
      hash_match_addr = 'h0;
      stat_last_addr = 'h0;
      for (j=0; j < HASHESPERLINE ; j = j + 1) begin: GEN_HASH_MATCH_ADDR
         if (hash_match[j]) begin
            hash_match_addr = j;
         end
         if (stat_last[j]) begin
            stat_last_addr = j;
         end
      end
   end

   generate
   for (i=0; i < HASHESPERLINE ; i = i + 1) begin: GEN_WR_MASK
      always @(*) begin
         mem_wr_mask[i] = 1;
         if (delete) begin 
            if (hash_match_addr > i )
               mem_wr_mask[i] = 0;
            else
               mem_wr_mask[i] = 1;
         end
         else begin
            if (reg_matched)  begin
               if (hash_match_addr < i )
                  mem_wr_mask[i] = 0;
               else
                  mem_wr_mask[i] = 1;
            end
         end
      end
   end
   endgenerate


    assign delete = reg_matched && reg_cmddelete;
   
    assign reg_stat = (reset || init)? 0 : 
                     ( (reg_cmddelete)? 0 : 1);
                   
    assign reg_route = (reset || init )? 0 :     
                       ((reg_cmdinsert)? reg_routing:
                        ((!reg_matched)? 0 :
                       reg_route_mem_outa[hash_match_addr]));


    assign reg_bloomstat = (reset || init || reg_cmddelete)? 'h0 : (reg_bloomstat_mem_outa[hash_match_addr] + ((reg_bloomstat_mem_outa[hash_match_addr] == 'b1)? 0 :  1));


   always @(*) begin
      next_outstate = outstate;
      rst_hash_vld  = 0;
      hash_wr_en    = 0;
      stat_wr_en    = 0;
      route_wr_en   = 0;
      sel_data_hash = 0;

      init           = 0;
      stat_line_addr = line_addr;

      cmd           = 'h0;
      in_fifo_rd_en = !in_fifo_empty;  // always read from fifo
      out_wr        = 0;
      out_ctrl      = 'hff;
      case(outstate)
         INIT_MEM: begin
            stat_line_addr = cnt_init;
            init = 1;
            stat_wr_en = 1;
            route_wr_en = 1;
            if (cnt_init == (NUMLINES - 1) )
               next_outstate = WAIT_FOR_HASH;
         end
         WAIT_FOR_HASH: begin
            if (hash_vld[0]) begin
               if (reg_cmddelete) begin
                  next_outstate = WAIT_FOR_DELETE;
               end
               else begin
                  if (reg_cmdinsert) begin
                     next_outstate = WAIT_FOR_INSERT;
                  end
                  else begin
                     next_outstate = WAIT_FOR_INDEX;
                  end
               end
            end
         end
         WAIT_FOR_DELETE: begin
            if (lookedup) begin
               rst_hash_vld = 1;
               next_outstate = WAIT_FOR_HASH;
               if (reg_matched) begin
                  hash_wr_en  = 1;
                  stat_wr_en  = 1;
                  route_wr_en = 1;
               end
            end
         end
         WAIT_FOR_INSERT: begin
            if (lookedup) begin
               rst_hash_vld = 1;
               next_outstate = WAIT_FOR_HASH;
               hash_wr_en  = 1;
               stat_wr_en  = 1;
               route_wr_en = 1;
            end
         end
  
         WAIT_FOR_INDEX: begin
            sel_data_hash = 1;
            out_ctrl      = 'h0;
            if (lookedup && instate == WAIT_FOR_FIFO) begin
               if (out_rdy) begin
                  rst_hash_vld  = 1;
                  out_wr        = 1;   
                  next_outstate = WAIT_FOR_HASH;
                 
                  if (reg_matched) begin
                     stat_wr_en  = 1;
                     hash_wr_en  = 1;
                     route_wr_en = 1;
                     cmd = CMD_UPDATE;
                  end
                  else begin 
                     cmd = CMD_NEW;
                  end
               end
            end
         end
      endcase // case(state)
   end // always @ (*) 
   
   always @(posedge clk) begin
      if (reset)
          outstate <= INIT_MEM;
      else
          outstate <= next_outstate;
   end
   
   always @(posedge clk) begin
      if (reset) begin
          cnt_init <= 'b0;
          reg_cnt_init <= 'b0;
       end
       else begin
          reg_cnt_init <= cnt_init;
          if (init)
             cnt_init <= cnt_init + 1;
       end
   end

   generate
   for (i=0; i < LOOKUP_LATENCY ; i = i + 1) begin: GEN_HASH_LATENCY
      always @(posedge clk) begin
         if (rst_hash_vld || reset) begin 
            hash_vld[i] <= 0;
         end
         else begin 
            if (i == 0) begin
               if (set_hash_vld)
                  hash_vld[0] <= 1;
            end
            else 
               hash_vld[i] <= hash_vld[i-1];
         end
      end
   end
   endgenerate
  
   assign lookedup = hash_vld[LOOKUP_LATENCY-1];
   
   assign lookup_data = {27'b0, reg_route_mem_outa[hash_match_addr], 1'b0, reg_stat_mem_outa[hash_match_addr]};
   assign out_data = {28'b0, cmd, lookup_data};

endmodule 


/* vim:set shiftwidth=3 softtabstop=3 expandtab: */
