///////////////////////////////////////////////////////////////////////////////
// Module: l3l4extract.v
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps


//`define UDP_REG_ADDR_WIDTH 2
//`define CPCI_NF2_DATA_WIDTH 32

module dropnonip 
   #(
      parameter DATA_WIDTH = 64,
      parameter CTRL_WIDTH = DATA_WIDTH/8,
      parameter UDP_REG_SRC_WIDTH = 1

   )
   (
      input  [DATA_WIDTH-1:0]             in_data,
      input  [CTRL_WIDTH-1:0]             in_ctrl,
      input                               in_wr,
      output                              in_rdy,

      output [DATA_WIDTH-1:0]         out_data,
      output [CTRL_WIDTH-1:0]         out_ctrl,
      output reg                          out_wr,
      input                               out_rdy,

      // --- Register interface
      input                              reg_req_in,
      input                              reg_ack_in,
      input                              reg_rd_wr_L_in,
      input  [`UDP_REG_ADDR_WIDTH-1:0]   reg_addr_in,
      input  [`CPCI_NF2_DATA_WIDTH-1:0]  reg_data_in,
      input  [UDP_REG_SRC_WIDTH-1:0]     reg_src_in,

      output                             reg_req_out,
      output                             reg_ack_out,
      output                             reg_rd_wr_L_out,
      output  [`UDP_REG_ADDR_WIDTH-1:0]  reg_addr_out,
      output  [`CPCI_NF2_DATA_WIDTH-1:0] reg_data_out,
      output  [UDP_REG_SRC_WIDTH-1:0]    reg_src_out,


      // misc
      input                                reset,
      input                                clk
   );


   //------------------------- Signals-------------------------------
   localparam SKIPTONEXTPACKET = 0;
   localparam EXTRACTFIELDS    = 1;
   localparam WRITEFIELD0      = 2;

   localparam WAITFORDECISION  = 0;
   localparam DUPLICATEPACKET0 = 1;
   localparam DUPLICATEPACKET1 = 2;
   localparam DISCARDPACKET0   = 3;
   localparam DISCARDPACKET1   = 4;


   wire [DATA_WIDTH-1:0]       in_fifo_data;
   wire [CTRL_WIDTH-1:0]       in_fifo_ctrl;
   wire                        in_fifo_nearly_full;
   wire                        in_fifo_empty;
   reg                         in_fifo_rd_en;

   wire                        out_fifo_nearly_full;
   wire                        out_fifo_empty;
   reg                         out_fifo_rd_en;

   wire       cmd_fifo_accepted;
   wire       cmd_fifo_nearly_full;
   wire       cmd_fifo_empty;
   reg        cmd_fifo_rd_en;
   reg        cmd_fifo_wr;


   reg [31:0] ipsrc, ipdst;
   reg [3:0]  ipversion;
   reg        swap_tuple;

   reg [15:0] tcp_src_port, tcp_dst_port;

   reg [7:0]  protocol_field;
   reg [1:0]  protocol_ident;

   reg [15:0] packet_len_field;  //Polozka delka v ethernet ramci 

   reg [7:0]  tos;
   reg [7:0]  ttl;   
   reg [7:0]  tcpflags;
   
   reg [15:0] doctets;           
   reg [15:0] input_port; 

   wire       l3proto_ok;
   reg        l4proto_ok;

   reg [3:0]  cnt;
   reg        cnt_reset;

   reg [2:0]  fsm_state, nfsm_state;
   reg [2:0]  out_fsm_state, out_nfsm_state;

   wire       in_fifo_data_rdy;
   reg        registers_ready;
   reg        allow_extract;
   reg        accepted;
   reg [31:0] cnt_accepted;
   reg [31:0] cnt_discarded;
   reg        total_en;
   reg [31:0] cnt_total;

   wire [95:0] reg_sw;

   wire pipeline_stopped;

   //------------------------- Modules-------------------------------

   generic_regs
   #(
      .UDP_REG_SRC_WIDTH   (UDP_REG_SRC_WIDTH),
      .TAG                 (`DROP_BLOCK_TAG),
      .REG_ADDR_WIDTH      (`DROP_REG_ADDR_WIDTH),  // Width of block addresses
      .NUM_COUNTERS        (0),                       // How many counters
      .NUM_SOFTWARE_REGS   (3),                       // How many sw regs
      .NUM_HARDWARE_REGS   (3)                        // How many hw regs
   ) fxpf_regs (
      .reg_req_in       (reg_req_in),
      .reg_ack_in       (reg_ack_in),
      .reg_rd_wr_L_in   (reg_rd_wr_L_in),
      .reg_addr_in      (reg_addr_in),
      .reg_data_in      (reg_data_in),
      .reg_src_in       (reg_src_in),

      .reg_req_out      (reg_req_out),
      .reg_ack_out      (reg_ack_out),
      .reg_rd_wr_L_out  (reg_rd_wr_L_out),
      .reg_addr_out     (reg_addr_out),
      .reg_data_out     (reg_data_out),
      .reg_src_out      (reg_src_out),

      // --- counters interface
      .counter_updates  (),
      .counter_decrement(),

      // --- SW regs interface
      .software_regs    ({reg_sw}),

      // --- HW regs interface
      .hardware_regs    ({cnt_total, cnt_discarded, cnt_accepted}),

      .clk              (clk),
      .reset            (reset)
    );

   
   fallthrough_small_fifo #(
      .WIDTH(CTRL_WIDTH+DATA_WIDTH),
      .MAX_DEPTH_BITS(2)
   ) input_fifo (
      .din           ({in_ctrl, in_data}),   // Data in
      .wr_en         (in_wr),                // Write enable
      .rd_en         (in_fifo_rd_en),        // Read the next word 
      .dout          ({in_fifo_ctrl, in_fifo_data}),
      .full          (),
      .nearly_full   (in_fifo_nearly_full),
      .empty         (in_fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );   


   fallthrough_small_fifo #(
      .WIDTH(CTRL_WIDTH+DATA_WIDTH),
      .MAX_DEPTH_BITS(4)
   ) output_fifo (
      .din           ({in_fifo_ctrl, in_fifo_data}),   // Data in
      .wr_en         (in_fifo_rd_en),                // Write enable
      .rd_en         (out_fifo_rd_en),        // Read the next word 
      .dout          ({out_ctrl, out_data}),
      .full          (),
      .nearly_full   (out_fifo_nearly_full),
      .empty         (out_fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );   

   fallthrough_small_fifo #(
      .WIDTH(1),
      .MAX_DEPTH_BITS(3)
   ) cmd_fifo (
      .din           (accepted),   // Data in
      .wr_en         (cmd_fifo_wr),                // Write enable
      .rd_en         (cmd_fifo_rd_en),        // Read the next word 
      .dout          (cmd_fifo_accepted),
      .full          (),
      .nearly_full   (cmd_fifo_nearly_full),
      .empty         (cmd_fifo_empty),
      .reset         (reset),
      .clk           (clk)
   );   


   //------------------------- Local assignments -------------------------------

   assign in_rdy           = !in_fifo_nearly_full;
   assign in_fifo_data_rdy = !in_fifo_empty;

   //------------------------- Logic-------------------------------

   always @(posedge clk) begin
      if (allow_extract && in_fifo_data_rdy) begin
         case (cnt)
            0: begin
                  input_port      = in_fifo_data[31:16];    
               end

            2: begin
                  packet_len_field = in_fifo_data[31:16];
                  ipversion        = in_fifo_data[15:12]; 
                  tos              = in_fifo_data[7:0];
               end

            3: begin
                  doctets         = in_fifo_data[63:48];
                  ttl             = in_fifo_data[15:8];
                  protocol_field  = in_fifo_data[7:0];
               end

            4: begin
                  ipsrc           = in_fifo_data[47:16];
                  ipdst[31:16]    = in_fifo_data[15:0]; 
               end   

            5: begin
                  ipdst[15:0]     = in_fifo_data[63:48];
                  tcp_src_port    = in_fifo_data[47:32];
                  tcp_dst_port    = in_fifo_data[31:16];
               end

            6: begin
                  tcpflags        = in_fifo_data[7:0];                      
               end  
            default:   ;     
         endcase    
      end
   end


   always @(posedge clk) begin
      if (cnt_reset || reset)
         cnt <= 0;
      else begin
         if (allow_extract && in_fifo_data_rdy) begin
            cnt <= cnt + 1;
         end    
      end
   end

   assign pipeline_stopped = reg_sw[0];

   always @(*) begin
      allow_extract  = 0;
      in_fifo_rd_en  = 0;
      cnt_reset      = 0;

      cmd_fifo_wr    = 0;
      accepted       = 0;
      total_en       = 0;

      nfsm_state     = fsm_state;    

      case (fsm_state)
         SKIPTONEXTPACKET: begin
            cnt_reset  = 1;
            if (in_fifo_data_rdy) begin
               if (in_fifo_ctrl == 'hff) begin
                  if (!pipeline_stopped) begin
                     nfsm_state = EXTRACTFIELDS;
                  end
               end
               else begin
                  in_fifo_rd_en  = 1;
               end
            end
         end

         EXTRACTFIELDS: begin 
            if (in_fifo_data_rdy && !out_fifo_nearly_full) begin
               allow_extract = 1;
               in_fifo_rd_en = 1;
               if (cnt == 6) begin
                  nfsm_state = WRITEFIELD0;
               end
            end
         end

         WRITEFIELD0: begin
                           
            if (in_fifo_data_rdy && !out_fifo_nearly_full && !cmd_fifo_nearly_full) begin
               in_fifo_rd_en = 1;
               total_en      = 1;
               cmd_fifo_wr   = 1;

               if (l3proto_ok) begin
                  accepted   = 1;
               end
               nfsm_state = SKIPTONEXTPACKET;
            end
         end

      endcase                    
   end

   always @(posedge clk) begin
      if (reset)
         fsm_state = SKIPTONEXTPACKET;
      else
         fsm_state = nfsm_state;
   end
   

   always @(*) begin
      out_fifo_rd_en = 0;
      out_wr         = 0;

      cmd_fifo_rd_en = 0;

      out_nfsm_state = out_fsm_state;    

      case (out_fsm_state)
         WAITFORDECISION: begin
            if (!cmd_fifo_empty) begin
               cmd_fifo_rd_en = 1;
               if (cmd_fifo_accepted)
                  out_nfsm_state = DUPLICATEPACKET0;
               else
                  out_nfsm_state = DISCARDPACKET0;
            end
         end

         DUPLICATEPACKET0: begin
            if (!out_fifo_empty && out_rdy) begin
               out_fifo_rd_en = 1;
               if (out_ctrl == 'hff) begin
                  out_wr         = 1;
                  out_nfsm_state = DUPLICATEPACKET1;
               end
            end
         end

         DUPLICATEPACKET1: begin
            if (!out_fifo_empty && out_rdy) begin
               out_fifo_rd_en = 1;
               out_wr         = 1;
               if (out_ctrl != 'h0) begin
                  out_nfsm_state = WAITFORDECISION;
               end
            end
         end

         DISCARDPACKET0: begin
            if (!out_fifo_empty && out_rdy) begin
               out_fifo_rd_en = 1;
               if (out_ctrl == 'hff) begin
                  out_nfsm_state = DISCARDPACKET1;
               end
            end
         end

         DISCARDPACKET1: begin
            if (!out_fifo_empty && out_rdy) begin
               out_fifo_rd_en = 1;
               if (out_ctrl != 'h0) begin
                  out_nfsm_state = WAITFORDECISION;
               end
            end
         end

      endcase                    
   end

   always @(posedge clk) begin
      if (reset)
         out_fsm_state = WAITFORDECISION;
      else
         out_fsm_state = out_nfsm_state;
   end

 
   // Zpracovavam pouze IPv4         
   assign l3proto_ok = (packet_len_field == 16'h0800)? 1:0;    //XXX zatim neberu stary ethernet

   //Otestovani, zda se jedna o protokol nami zpracovavany
   always @(*) begin
      l4proto_ok     = 1;
      protocol_ident = 0;

      case (protocol_field)
         1: protocol_ident = 1; //ICMP
         6: protocol_ident = 2; //TCP
        17: protocol_ident = 3; //UDP
        default: l4proto_ok = 0;
      endcase
   end

   always @(posedge clk) begin
      if (reset)
         cnt_accepted = 'h0;
      else
         if (accepted)
         cnt_accepted = cnt_accepted + 1;
   end
   always @(posedge clk) begin
     cnt_discarded = cnt_total - cnt_accepted;
   end

   always @(posedge clk) begin
      if (reset)
         cnt_total = 'h0;
      else
         if (total_en)
         cnt_total = cnt_total + 1;
   end
           
endmodule 
