///////////////////////////////////////////////////////////////////////////////
//
// Module: decision.v
// Description: 
//
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

module decision_pf #(
   parameter DATA_WIDTH = 64,
   parameter CTRL_WIDTH = DATA_WIDTH/8
   )
   (
   input  [31:0]            cutoff,

   input  [DATA_WIDTH-1:0]            in_data,
   input  [CTRL_WIDTH-1:0]            in_ctrl,
   input                              in_wr,
   output                             in_rdy,

   output reg [DATA_WIDTH-1:0]        out_data,
   output reg [CTRL_WIDTH-1:0]        out_ctrl,
   output reg                         out_wr,
   input                              out_rdy,

    // --- Misc
   input                              clk,
   input                              reset
);
   
   function integer log2;
      input integer number;
      begin
         log2=0;
         while(2**log2<number) begin
            log2=log2+1;
         end
      end
   endfunction // log2 
   localparam MAXPACKETS = 3;
   localparam FINMARK    = 7;

   localparam CMD_DONTCARE = 'h0;
   localparam CMD_NEW    = 'h1;
   localparam CMD_UPDATE = 'h2;

   localparam FINFLAG = 0;
   localparam SYNFLAG = 1;
   localparam RSTFLAG = 2;

   localparam PROTOTCP = 6;

   localparam INWORD0  = 0;
   localparam INWORD1  = 1;
   localparam INWORD2  = 2;
   localparam INWORD3  = 3;
   localparam OUTWORD0 = 4;

   localparam NEXTHOP = 32'b0;
   localparam NUMBEROFPORTS= 8;

   reg [7:0]  iface;
   reg [7:0]  output_port;
   reg [7:0]  oface;
   wire [2:0] next_iface;
   reg [31:0] src_ip_addr;
   reg [31:0] dst_ip_addr;
   reg [31:0] lookup_data;

   reg [31:0] cmd;


   reg [31:0] byte_count;
   reg [31:0] start_timestamp;
   reg [31:0] end_timestamp;
   reg [15:0] src_port;
   reg [15:0] dst_port;

   reg [7:0] tcp_flags;
   reg [7:0] protocol;
   reg [7:0] tos;
   reg [7:0] ttl;

   reg [3:0] state, next_state;
   wire      accept;
   wire      dontcare;

   wire [CTRL_WIDTH-1:0] in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo_data_dout;

   wire in_fifo_nearly_full;
   wire in_fifo_empty;
   reg  in_fifo_rd_en;

   reg  inword0_we;
   reg  inword1_we;
   reg  inword2_we;
   reg  inword3_we;

   assign in_rdy = !in_fifo_nearly_full;

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input_fifo
        (.din ({in_ctrl, in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );
   
   always @(posedge clk) begin
      if (reset) begin
         src_ip_addr     <= 0;
         dst_ip_addr     <= 0;
         iface           <= 0;
         oface           <= 0;
         byte_count      <= 0;
         cmd             <= 0;
         lookup_data     <= 0;
         src_port        <= 0;
         dst_port        <= 0;
         tcp_flags       <= 0;
         protocol        <= 0;
         tos             <= 0;
         ttl             <= 0;
      end
      else begin
         if (inword0_we) begin
            cmd             <= in_fifo_data_dout[63:32];
            lookup_data     <= in_fifo_data_dout[31:0];
         end
         if (inword1_we) begin
            tos             <= in_fifo_data_dout[63:56];
            byte_count      <= in_fifo_data_dout[55:40];
            ttl             <= in_fifo_data_dout[39:32];
            tcp_flags       <= in_fifo_data_dout[15:8];
            iface           <= in_fifo_data_dout[7:0];
         end
         if (inword2_we) begin
            src_ip_addr     <= in_fifo_data_dout[63:32];
            dst_ip_addr     <= in_fifo_data_dout[31:0];
         end
         if (inword3_we) begin
            src_port        <= in_fifo_data_dout[63:48];
            dst_port        <= in_fifo_data_dout[47:32];
            protocol        <= in_fifo_data_dout[23:16];
            oface           <= in_fifo_data_dout[15:8];
         end
      end 
   end 
   
   assign next_iface = iface[2:0] + lookup_data[6:4];

   integer j;
   always @(*) begin
      output_port = 'h0;
      for (j = 0; j < NUMBEROFPORTS; j = j + 1) begin
         if (next_iface == j)
            output_port[j] = 1;
         else
            output_port[j] = 0;
      end
   end

   always @(*) begin
      next_state = state;

      inword0_we = 0;
      inword1_we = 0;
      inword2_we = 0;
      inword3_we = 0;
      out_wr    = 0;
      out_ctrl   = 'h00;
      in_fifo_rd_en = 0;

      out_data[63:0] = 'h0;
      case(state)
         INWORD0: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword0_we = 1;
               next_state = INWORD1;
            end
         end
         INWORD1: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword1_we = 1;
               next_state = INWORD2;
            end
         end
         INWORD2: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword2_we = 1;
               next_state = INWORD3;
            end
         end
         INWORD3: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword3_we = 1;
               if (in_fifo_ctrl_dout == 'hff)
                  next_state = OUTWORD0;
            end
         end
         OUTWORD0: begin
            if (accept) begin
               out_data[7:0] = {5'b0, 1'b1, lookup_data[5:4]};
            end
            else begin
               out_data[7:0] = {5'b0, lookup_data[6:4]};
            end
            if (out_rdy) begin
               out_wr  = 1;
               next_state = INWORD0;
            end
         end
      endcase
   end

   assign accept = 
      (
      (protocol == PROTOTCP) && 
      (
      (tcp_flags[SYNFLAG] == 1'b1) ||
      (tcp_flags[RSTFLAG] == 1'b1) ||
      (tcp_flags[FINFLAG] == 1'b1)
      )
      ) 
      ||
      (cmd[1:0] == CMD_NEW) ||
      (cmd[1:0] == CMD_DONTCARE) ||
      (lookup_data[2:0] < cutoff[2:0]  && cmd[1:0] == CMD_UPDATE);

   assign dontcare = (cmd[1:0] == CMD_DONTCARE);

   always @(posedge clk) begin
      if (reset)
          state <= INWORD0;
      else
          state <= next_state;
   end 

endmodule 


/* vim:set shiftwidth=3 softtabstop=3 expandtab: */
