///////////////////////////////////////////////////////////////////////////////
//
// Module: decision.v
// Description: 
//
//
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps

module decision_hc #(
   parameter DATA_WIDTH = 64,
   parameter CTRL_WIDTH = DATA_WIDTH/8
   )
   (

   input  [DATA_WIDTH-1:0]            in_data,
   input  [CTRL_WIDTH-1:0]            in_ctrl,
   input                              in_wr,
   output                             in_rdy,

   output reg [DATA_WIDTH-1:0]        out_data,
   output reg [CTRL_WIDTH-1:0]        out_ctrl,
   output reg                         out_wr,
   input                              out_rdy,

    // --- Misc
   input                              clk,
   input                              reset
);
   
   function integer log2;
      input integer number;
      begin
         log2=0;
         while(2**log2<number) begin
            log2=log2+1;
         end
      end
   endfunction // log2 
   localparam MAXPACKETS = 3;
   localparam FINMARK    = 7;

   localparam CMD_DONTCARE = 'h0;
   localparam CMD_NEW    = 'h1;
   localparam CMD_UPDATE = 'h2;

   localparam FINFLAG = 0;
   localparam SYNFLAG = 1;
   localparam RSTFLAG = 2;

   localparam PROTOTCP = 6;

   localparam INWORD0  = 0;
   localparam INWORD1  = 1;
   localparam INWORD2  = 2;
   localparam INWORD3  = 3;
   localparam OUTWORD0 = 4;

   localparam NEXTHOP = 32'b0;
   localparam NUMBEROFPORTS= 8;

   reg [7:0]  iface;
   reg [7:0]  output_port;
   reg [2:0]  oface;
   wire [7:0] next_iface;
   reg [31:0] src_ip_addr;
   reg [31:0] dst_ip_addr;

   reg [31:0] lookup_data0;
   reg [31:0] cmd0;

   reg [31:0] lookup_data1;
   reg [31:0] cmd1;


   reg [31:0] byte_count;
   reg [31:0] start_timestamp;
   reg [31:0] end_timestamp;
   reg [15:0] src_port;
   reg [15:0] dst_port;

   reg [7:0] tcp_flags;
   reg [7:0] protocol;
   reg [7:0] tos;
   reg [7:0] ttl;

   reg [3:0] state, next_state;
   wire      accept;

   wire [CTRL_WIDTH-1:0] in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0] in_fifo_data_dout;

   wire in_fifo_nearly_full;
   wire in_fifo_empty;
   reg  in_fifo_rd_en;

   reg  inword0_we;
   reg  inword1_we;
   reg  inword2_we;
   reg  inword3_we;

   assign in_rdy = !in_fifo_nearly_full;
   assign next_iface = (iface - 1);

   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(4))
      input_fifo
        (.din ({in_ctrl, in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );
   
   always @(posedge clk) begin
      if (reset) begin
         cmd0             <= 0;
         lookup_data0     <= 0;
         cmd1             <= 0;
         lookup_data1     <= 0;
      end
      else begin
         if (inword0_we) begin
            cmd0             <= in_fifo_data_dout[63:32];
            lookup_data0     <= in_fifo_data_dout[31:0];
         end
         if (inword1_we) begin
            cmd1             <= in_fifo_data_dout[63:32];
            lookup_data1     <= in_fifo_data_dout[31:0];
         end
      end 
   end 
   
   always @(*) begin
      oface = 'h0;
      if (cmd0 == CMD_UPDATE) begin
         oface = lookup_data0[6:4];
      end
      else begin
         if (cmd1 == CMD_UPDATE) begin
            oface = lookup_data1[6:4];
         end
      end
  end

   integer j;
   always @(*) begin
      output_port = 'h0;
      for (j = 0; j < NUMBEROFPORTS; j = j + 1) begin
         if (oface == j)
            output_port[j] = 1;
         else
            output_port[j] = 0;
      end
   end


   always @(*) begin
      next_state = state;

      inword0_we = 0;
      inword1_we = 0;
      inword2_we = 0;
      inword3_we = 0;
      out_wr    = 0;
      out_ctrl   = 'h00;
      in_fifo_rd_en = 0;

      out_data     = 'h0;
      case(state)
         INWORD0: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword0_we = 1;
               next_state = INWORD1;
            end
         end
         INWORD1: begin
            if (!in_fifo_empty) begin
               in_fifo_rd_en = 1;
               inword1_we = 1;
               next_state = OUTWORD0;
            end
         end
         OUTWORD0: begin
            out_data[7:0] = {5'h0, oface[2:0]};
            if (out_rdy) begin
               out_wr  = 1;
               next_state = INWORD0;
            end
         end
      endcase
   end

   assign accept = 
      (
      (cmd0[1:0] == CMD_NEW) && 
      (cmd1[1:0] == CMD_NEW)
      );
      

   always @(posedge clk) begin
      if (reset)
          state <= INWORD0;
      else
          state <= next_state;
   end 

endmodule 


/* vim:set shiftwidth=3 softtabstop=3 expandtab: */
