///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
`timescale 1ns/1ps
module ar
   #(
      parameter DATA_WIDTH = 64,
      parameter CTRL_WIDTH = DATA_WIDTH/8,
      parameter UDP_REG_SRC_WIDTH = 1,
      parameter INPUT_ARBITER_STAGE_NUM = 2,
      parameter NUM_OUTPUT_QUEUES = 8,
      parameter STAGE_NUM = 5,
      parameter NUM_IQ_BITS = 3
   )
   (
      // --- data path interface
      input  [DATA_WIDTH-1:0]            in_data,
      input  [CTRL_WIDTH-1:0]            in_ctrl,
      input                              in_wr,
      output                             in_rdy,

      output reg [DATA_WIDTH-1:0]        out_data,
      output [CTRL_WIDTH-1:0]        out_ctrl,
      output reg                         out_wr,
      input                              out_rdy,


      // --- Register interface
      input                              reg_req_in,
      input                              reg_ack_in,
      input                              reg_rd_wr_L_in,
      input  [`UDP_REG_ADDR_WIDTH-1:0]   reg_addr_in,
      input  [`CPCI_NF2_DATA_WIDTH-1:0]  reg_data_in,
      input  [UDP_REG_SRC_WIDTH-1:0]     reg_src_in,

      output                             reg_req_out,
      output                             reg_ack_out,
      output                             reg_rd_wr_L_out,
      output  [`UDP_REG_ADDR_WIDTH-1:0]  reg_addr_out,
      output  [`CPCI_NF2_DATA_WIDTH-1:0] reg_data_out,
      output  [UDP_REG_SRC_WIDTH-1:0]    reg_src_out,
      

      // --- Misc
      input                              clk,
      input                              reset
   );

   `LOG2_FUNC
   
   //--------------------- Internal Parameter-------------------------
   localparam DECIDE = 0;
   localparam ACCEPT0 = 1;
   localparam ACCEPT1 = 2;
   localparam ACCEPT2 = 3;
   localparam DISCARD0 = 3;
   localparam DISCARD1 = 4;

   //---------------------- Wires and regs----------------------------

   reg                              in_fifo_rd_en;
   wire [CTRL_WIDTH-1:0]            in_fifo_ctrl_dout;
   wire [DATA_WIDTH-1:0]            in_fifo_data_dout;
   wire                             in_fifo_nearly_full;
   wire                             in_fifo_empty;

   // -------------------------------------------
   //  Arbitrary signals
   // -------------------------------------------
   reg  [2:0] state, next_state;

   wire [2:0] iface, classnum;
   reg  [2:0] reg_classnum;
   reg        classwrite;
   wire [2:0] swoface;
   reg  [2:0] hwoface;
   wire [2:0] hwoface0;
   wire [2:0] hwoface1;
   wire [2:0] hwoface2;
   wire [2:0] hwoface3;

   reg  [7:0] sw_output_port;
   reg  [7:0] hw_output_port;

   wire [31:0] sw_routing;

   reg  [31:0] cnt_swpackets;
   reg   swpackets_en;

   localparam NUMBEROFPORTS= 8;
 
   //------------------------- Modules-------------------------------
   generic_regs
   #( 
      .UDP_REG_SRC_WIDTH   (UDP_REG_SRC_WIDTH),
      .TAG                 (`AR_BLOCK_TAG),
      .REG_ADDR_WIDTH      (`AR_REG_ADDR_WIDTH),  // Width of block addresses
      .NUM_COUNTERS        (0),                       // How many counters
      .NUM_SOFTWARE_REGS   (1),                       // How many sw regs
      .NUM_HARDWARE_REGS   (1)                        // How many hw regs
   ) hostcache_regs (
      .reg_req_in       (reg_req_in),
      .reg_ack_in       (reg_ack_in),
      .reg_rd_wr_L_in   (reg_rd_wr_L_in),
      .reg_addr_in      (reg_addr_in),
      .reg_data_in      (reg_data_in),
      .reg_src_in       (reg_src_in),

      .reg_req_out      (reg_req_out),
      .reg_ack_out      (reg_ack_out),
      .reg_rd_wr_L_out  (reg_rd_wr_L_out),
      .reg_addr_out     (reg_addr_out),
      .reg_data_out     (reg_data_out),
      .reg_src_out      (reg_src_out),

      // --- counters interface
      .counter_updates  (),
      .counter_decrement(),

      // --- SW regs interface
      .software_regs    (sw_routing),

      // --- HW regs interface
      .hardware_regs    (cnt_swpackets),

      .clk              (clk),
      .reset            (reset)
    );


   fallthrough_small_fifo #(.WIDTH(DATA_WIDTH+CTRL_WIDTH), .MAX_DEPTH_BITS(8))
      input_fifo
        (.din ({in_ctrl,in_data}),     // Data in
         .wr_en (in_wr),               // Write enable
         .rd_en (in_fifo_rd_en),       // Read the next word 
         .dout ({in_fifo_ctrl_dout, in_fifo_data_dout}),
         .full (),
         .nearly_full (in_fifo_nearly_full),
         .empty (in_fifo_empty),
         .reset (reset),
         .clk (clk)
         );

   //----------------------- Logic -----------------------------

   assign    in_rdy              = !in_fifo_nearly_full;


   assign iface    = in_fifo_data_dout[18:16];
   assign classnum = in_fifo_data_dout[50:48];

   assign swoface = iface[2:0] - 1;
   assign hwoface0 = iface[2:0] + sw_routing[2:0];
   assign hwoface1 = iface[2:0] + sw_routing[10:8];
   assign hwoface2 = iface[2:0] + sw_routing[18:16];
   assign hwoface3 = iface[2:0] + sw_routing[26:24];

   always @(*) begin
      hwoface = hwoface0;
      case (classnum[1:0])
        2'b00: hwoface = hwoface0;
        2'b01: hwoface = hwoface1;
        2'b10: hwoface = hwoface2;
        2'b11: hwoface = hwoface3;
      endcase
   end

   integer j;

   always @(posedge clk) begin
      sw_output_port = 'h0;
      for (j = 0; j < NUMBEROFPORTS; j = j + 1) begin
         if (swoface == j)
            sw_output_port[j] = 1'b1;
         else
            sw_output_port[j] = 1'b0;
      end
   end

   always @(posedge clk) begin
     hw_output_port = (classnum[2])? 'h02 : 'h00;
     for (j = 0; j < NUMBEROFPORTS; j = j + 1) begin
        if (hwoface == j)
           hw_output_port[j] = 1'b1;
     end
   end

   always @(posedge clk) begin
      if (classwrite)
         reg_classnum = classnum;
   end

   always @(*) begin
      in_fifo_rd_en = 0;
      out_wr        = 0;
      classwrite    = 0;
      swpackets_en  = 0;
      next_state    = state;

      out_data = in_fifo_data_dout;

      case (state)
         DECIDE: begin
            if (!in_fifo_empty) begin

               if (in_fifo_ctrl_dout == 'hff)
                  next_state = ACCEPT0;
               else
                  in_fifo_rd_en = 1;
            end
         end

         ACCEPT0: begin
            out_data[`IOQ_DST_PORT_POS+7:`IOQ_DST_PORT_POS] = 
              (iface[0])? sw_output_port : hw_output_port;

            if (!in_fifo_empty && out_rdy) begin
               in_fifo_rd_en = 1;
               out_wr     = 1;
               classwrite = 1;
               swpackets_en  = 1;
               next_state = ACCEPT1;
            end
         end
         ACCEPT1: begin
            out_data[23:16] = in_fifo_data_dout[23:16] + {5'b0, reg_classnum[2:0]} + 8'h01;
            if (!in_fifo_empty && out_rdy) begin
               in_fifo_rd_en = 1;
               out_wr     = 1;
               next_state = ACCEPT2;
            end
         end
 
         ACCEPT2: begin 
            if (!in_fifo_empty && out_rdy) begin
               in_fifo_rd_en = 1;
               out_wr        = 1;
               if (in_fifo_ctrl_dout != 'h0) begin
                  next_state = DECIDE;
               end
            end
         end
      endcase                    
   end
         
   always @(posedge clk) begin
      if (reset)
         state = DECIDE;
      else
         state = next_state;
   end

   assign out_ctrl = in_fifo_ctrl_dout;

   always @(posedge clk) begin
      if (reset)
         cnt_swpackets = 'h0;
      else
         if (swpackets_en && !iface[0] && hw_output_port[1])
         cnt_swpackets = cnt_swpackets + 1;
   end

endmodule 
