#include <stdint.h>
#include <asm/types.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <linux/ip.h>
#include <linux/if_ether.h>
#include <linux/udp.h>
#include <linux/tcp.h>
#include <assert.h>
#include <err.h>
#include <pcap.h>
#include <printf.h>
#include <unistd.h>
#include <linux/in.h>

struct vlan_ethhdr {
   unsigned char        h_dest[ETH_ALEN];
   unsigned char        h_source[ETH_ALEN];
   unsigned short       h_vlan_proto;
   unsigned short       h_vlan_TCI;
   unsigned short       h_vlan_encapsulated_proto;
};

unsigned long csum_partial(const void *buf, unsigned len,
			   unsigned int partial);

#define MAX_IP_DGRAM_SIZE 1500
#define MAX_ETH_FRAME_SIZE 1600
#define MAX_TCP_MSS 65536
#define MAX_IP_TTL 256

/* ECN stuff */
static unsigned
ecn_counters[4];
static unsigned
tcp_cwr_packets;
static unsigned
tcp_ece_packets;
static int
tcp_ecn_ce;

/* Distributions of things */
static unsigned
ip_types_of_service[64];
static unsigned
ip_datagram_sizes[MAX_IP_DGRAM_SIZE+1];
static unsigned
tcp_mss_dist[MAX_TCP_MSS+1];
static unsigned
ip_ttl_dist[MAX_IP_TTL+1];
static unsigned
tcp_option_dist[256];
static unsigned
ip_options_dist[256];

/* IP things */
static unsigned
ip_csum_error;
static unsigned
ip_option_packets;
static unsigned
ip_options;
static unsigned
ip_bogo_options;
static int
nr_fragments;
static int
nr_errors;
static int
total_packets;

/* UDP stuff */
static int
udp_packets;
static int
udp_bad_csum;
static int
udp_missing_csum;

/* DNS stuff */
static int
dns_packets;
static int
dns_bad_csum;
static int
dns_missing_csum;

/* TCP stuff */
static int
tcp_bad_csum;
static int
tcp_packets;
static int
tcp_options;
static int
tcp_packets_with_options;
static int
tcp_bad_options;

static char
pcap_errbuf[PCAP_ERRBUF_SIZE];


/* Distribution of packet sizes for packets with errors */
static unsigned
err_packet_size_dist[MAX_IP_DGRAM_SIZE+1];

int Zflag = 0;

static inline unsigned short ip_fast_csum(unsigned char *iph, unsigned int ihl) 
{
        unsigned int sum;

        asm(    "  movl (%1), %0\n"
                "  subl $4, %2\n"
                "  jbe 2f\n"
                "  addl 4(%1), %0\n"
                "  adcl 8(%1), %0\n"
                "  adcl 12(%1), %0\n"
                "1: adcl 16(%1), %0\n"
                "  lea 4(%1), %1\n"
                "  decl %2\n"
                "  jne  1b\n"
                "  adcl $0, %0\n"
                "  movl %0, %2\n"
                "  shrl $16, %0\n"
                "  addw %w2, %w0\n"
                "  adcl $0, %0\n"
                "  notl %0\n"
                "2:"
        /* Since the input registers which are loaded with iph and ipl
           are modified, we must also specify them as outputs, or gcc
           will assume they contain their original values. */
        : "=r" (sum), "=r" (iph), "=r" (ihl)
        : "1" (iph), "2" (ihl) 
        : "memory");
        return(sum);
}

static inline unsigned int csum_fold(unsigned int sum)
{
        __asm__(
                "addl %1, %0            ;\n"
                "adcl $0xffff, %0       ;\n"
                : "=r" (sum)
                : "r" (sum << 16), "0" (sum & 0xffff0000)
        );
        return (~sum) >> 16;
}

static inline unsigned long csum_tcpudp_nofold(unsigned long saddr,
                                                   unsigned long daddr,
                                                   unsigned short len,
                                                   unsigned short proto,
                                                   unsigned int sum)
{
    __asm__(
        "addl %1, %0    ;\n"
        "adcl %2, %0    ;\n"
        "adcl %3, %0    ;\n"
        "adcl $0, %0    ;\n"
        : "=r" (sum)
        : "g" (daddr), "g"(saddr), "g"((ntohs(len)<<16)+proto*256), "0"(sum));
    return sum;
}

static inline unsigned short int csum_tcpudp_magic(unsigned long saddr,
                                                   unsigned long daddr,
                                                   unsigned short len,
                                                   unsigned short proto,
                                                   unsigned int sum)
{
        return csum_fold(csum_tcpudp_nofold(saddr,daddr,len,proto,sum));
}

static unsigned short
udp_sum_calc(const struct iphdr *ipp,
	     const struct udphdr *udpp,
	     unsigned len,
	     const unsigned char *buf)
{
  unsigned short claimed = udpp->check;
  unsigned short actual;
  unsigned int partial;

  partial = ~claimed;
  partial = csum_partial(buf, len, partial);
  actual = csum_tcpudp_magic(ipp->saddr, ipp->daddr, len,
			     IPPROTO_UDP, partial);
  if (actual == 0)
	  return 0xffff;
  else
	  return actual;
}

static unsigned short
tcp_sum_calc(const struct iphdr *ipp,
	     const struct tcphdr *udpp,
	     unsigned len,
	     const unsigned char *buf)
{
  unsigned short claimed = udpp->check;
  unsigned short actual;
  unsigned int partial;

  partial = ~claimed;
  partial = csum_partial(buf, len, partial);
  actual = csum_tcpudp_magic(ipp->saddr, ipp->daddr, len,
			     IPPROTO_TCP, partial);
  return actual;
}

static int mode;

static void
proc_packet_callback(u_char *ignore,
		     const struct pcap_pkthdr *phdr,
		     const u_char *buf)
{
	struct iphdr *ip_hdr;
	const struct ethhdr *eth_hdr;
	int claimed_ip_csum;
	int actual_ip_csum;
	const u_char *end_of_packet = buf + phdr->caplen;
	const u_char *ip_payload;
	unsigned ip_payload_len;
	unsigned ip_flags;
	int is_dns;
	unsigned ecn_codepoint;

	if (total_packets++ % 100000 == 0)
		printf("Packet %d.\n", total_packets);
	if (mode == 1) {
		eth_hdr = (const struct ethhdr *)buf;
		if (eth_hdr->h_proto != 0x8) {
			printf("Non-ip packet (%x)!\n", eth_hdr->h_proto);
			printf("Data: %.*B\n", phdr->caplen,
			       buf);
			return;
		}

		if (phdr->len - sizeof(*eth_hdr) > MAX_IP_DGRAM_SIZE)
			printf("Stupidly big IP dgram.\n");
		else
			ip_datagram_sizes[phdr->len - sizeof(*eth_hdr)]++;
		ip_hdr = (struct iphdr *)(eth_hdr + 1);
	} else if (mode == 0) {
		ip_hdr = (struct iphdr *)buf;
		if (phdr->len > MAX_IP_DGRAM_SIZE)
			printf("Stupidly big IP dgram.\n");
		else
			ip_datagram_sizes[phdr->len]++;
	}
	if ((u_char *)ip_hdr >= end_of_packet) {
		printf("Lost IP header due to truncation.\n");
		nr_errors++;
		return;
	}

	claimed_ip_csum = ip_hdr->check;
	ip_hdr->check = 0;
	actual_ip_csum = ip_fast_csum((unsigned char *)ip_hdr, ip_hdr->ihl);

	if (claimed_ip_csum != actual_ip_csum) {
		printf("IP csum mismatch: %x != %x\n", claimed_ip_csum,
		       actual_ip_csum);
		ip_csum_error ++;
		return;
	}

	ip_hdr->tot_len = ntohs(ip_hdr->tot_len);
	if (ip_hdr->tot_len > (unsigned)end_of_packet-(unsigned)ip_hdr) {
		printf("Odd: %d > %d.\n", ip_hdr->tot_len,
		       (unsigned)end_of_packet - (unsigned)ip_hdr);
		nr_errors++;
		return;
	}

	ip_payload = (const u_char *)ip_hdr + ip_hdr->ihl * 4;
	ip_payload_len = ip_hdr->tot_len - ip_hdr->ihl * 4;

	if (ip_hdr->ihl < sizeof(struct iphdr) / 4) {
		printf("IP options had negative size?\n");
		ip_bogo_options++;
	} else if (ip_hdr->ihl != sizeof(struct iphdr) / 4) {
		unsigned char *options;
		unsigned opt_avail;
		unsigned opt_consumed;
		unsigned opt_nmr;
		printf("Option.\n");
		ip_option_packets++;
		opt_avail = sizeof(struct iphdr) - ip_hdr->ihl * 4;
		options = (unsigned char *)(ip_hdr + 1);
		opt_consumed = 0;
		while (opt_consumed < opt_avail) {
			opt_nmr = options[opt_consumed];
			ip_options_dist[opt_nmr]++;
			ip_options++;
			switch (opt_nmr) {
			case 0:
				/* End of options */
				opt_consumed = opt_avail + 1;
				break;
			case 1:
				/* NOOP */
				opt_consumed++;
				break;
			default:
				if (opt_consumed + 1 < opt_avail &&
				    options[opt_consumed + 1] != 0) {
					opt_consumed += options[opt_consumed+1];
				} else {
					ip_bogo_options++;
					opt_consumed = opt_avail + 1;
				}
				break;
			}
		}
	}

	ecn_codepoint = ip_hdr->tos >> 6;
	assert(ecn_codepoint < 4);
	ecn_counters[ecn_codepoint]++;
	assert((ip_hdr->tos & 0x3f) < 64);
	ip_types_of_service[ip_hdr->tos & 0x3f]++;

	ip_hdr->frag_off = ntohs(ip_hdr->frag_off);
	ip_flags = ip_hdr->frag_off >> 13;
	ip_hdr->frag_off &= 0x1fff;
	if ((ip_flags & 1) || (ip_hdr->frag_off != 0)) {
		printf("Fragment (frag_off %x, flags %x)!\n",
		       ip_hdr->frag_off, ip_flags);
		nr_fragments++;
		return;
	}

	ip_ttl_dist[ip_hdr->ttl]++;

	if (ip_hdr->protocol == IPPROTO_UDP) {
		struct udphdr *udp_hdr =
			(struct udphdr *)ip_payload;
		int claimed_udp_csum;
		int actual_udp_csum;
		int dumpable_len;

		is_dns = (udp_hdr->source == ntohs(53) ||
			  udp_hdr->dest == ntohs(53));
		claimed_udp_csum = udp_hdr->check;
		if (claimed_udp_csum == 0) {
			udp_missing_csum++;
			if (is_dns)
				dns_missing_csum++;
		} else if (ip_payload_len >= sizeof(*udp_hdr)) {
			actual_udp_csum = udp_sum_calc(ip_hdr,
						       udp_hdr,
						       ip_payload_len,
						       ip_payload);
			dumpable_len = ntohs(udp_hdr->len);
			if (dumpable_len != ip_payload_len)
				printf("UDP length mismatch: %d != %d.\n",
				       udp_hdr->len,
				       ip_payload_len);
			if (dumpable_len > ip_payload_len)
				dumpable_len = ip_payload_len;
			if (claimed_udp_csum != actual_udp_csum) {
				udp_bad_csum++;
				if (is_dns)
					dns_bad_csum++;
				printf("Packet at time %ld.%lu.\n",
				       phdr->ts.tv_sec,
				       phdr->ts.tv_usec);
				printf("UDP csum mismatch: %x != %x (%x).\n",
				       claimed_udp_csum, actual_udp_csum,
				       claimed_udp_csum ^ actual_udp_csum);
				printf("Packet: %d -> %d, len %d(%d).\n",
				       ntohs(udp_hdr->source),
				       ntohs(udp_hdr->dest),
				       ntohs(udp_hdr->len),
				       ip_payload_len - sizeof(*udp_hdr));
				printf("IP packet %.*B\n",
				      (unsigned)end_of_packet-(unsigned)ip_hdr,
				       ip_hdr);
				err_packet_size_dist[
					(unsigned)end_of_packet-
					(unsigned)ip_hdr]++;
			}
		} else {
			printf("UDP too short for header!\n");
		}
		udp_packets++;
		if (is_dns)
			dns_packets++;
	} else if (ip_hdr->protocol == IPPROTO_TCP) {
		struct tcphdr *tcp_hdr =
			(struct tcphdr *)ip_payload;
		int claimed_tcp_csum;
		int actual_tcp_csum;

		if (ecn_codepoint == 3)
			tcp_ecn_ce++;
		claimed_tcp_csum = tcp_hdr->check;
		actual_tcp_csum = tcp_sum_calc(ip_hdr,
					       tcp_hdr,
					       ip_payload_len,
					       ip_payload);
		if (claimed_tcp_csum != actual_tcp_csum &&
		    !(claimed_tcp_csum == 0 && actual_tcp_csum == 0xffff) &&
		    !(claimed_tcp_csum == 0xffff && actual_tcp_csum == 0)) {
			tcp_bad_csum++;
			printf("TCP csum mismatch: %x != %x (%x).\n",
			       claimed_tcp_csum, actual_tcp_csum,
			       claimed_tcp_csum ^ actual_tcp_csum);
			printf("IP packet %.*B\n",
			       (unsigned)end_of_packet-(unsigned)ip_hdr,
			       ip_hdr);
			err_packet_size_dist[(unsigned)end_of_packet-
				(unsigned)ip_hdr]++;
		}
		if (tcp_hdr->doff != sizeof(struct tcphdr) / 4) {
			unsigned char *opt_ptr;
			unsigned opt_avail;
			unsigned opt_consumed;

			opt_avail = tcp_hdr->doff * 4 - sizeof(struct tcphdr);
			tcp_packets_with_options++;
			opt_ptr = (unsigned char *)(tcp_hdr + 1);
			opt_consumed = 0;
			while (opt_consumed < opt_avail) {
				unsigned mss;
				tcp_option_dist[opt_ptr[opt_consumed]]++;
				tcp_options++;
				switch (opt_ptr[opt_consumed]) {
				case 0:
					/* End of options. */
					opt_consumed = opt_avail + 1;
					break;
				case 1:
					/* Do nothing. */
					opt_consumed++;
					break;
				case 2:
					/* MSS */
					mss = (opt_ptr[opt_consumed+2] << 8)
						+ opt_ptr[opt_consumed+3];
					tcp_mss_dist[mss]++;
				default:
					if (opt_ptr[opt_consumed+1] == 0) {
						printf("Bad option.\n");
						tcp_bad_options++;
						opt_consumed = opt_avail + 1;
					} else
						opt_consumed += opt_ptr[opt_consumed+1];
					break;
				}
			}
		}
		tcp_packets++;
		tcp_cwr_packets += tcp_hdr->cwr;
		tcp_ece_packets += tcp_hdr->ece;
	}
	return;
}

static int
printf_special_B_handler_func(FILE *f, const struct printf_info *info,
                              const void *const *args)
{
        int x;
        int count = 0;
        int r;
        const unsigned char *buf = *(char **)args[0];

        for (x = 0; x < info->prec; x++) {
                r = fprintf(f, "%.2x%c", buf[x], (char)info->pad);
                if (r < 0)
                        return r; /* uh oh... */
                count += r;
        }
        return count;
}

static int
printf_special_B_arginfo_func(const struct printf_info *info,
                              size_t n,
                              int *argtypes)
{
        if (n > 0)
                argtypes[0] = PA_POINTER;
        return 1;
}


int
main(int argc, char *argv[])
{
	pcap_t *pcap;
	unsigned long long acc;
	int x;
	static const char *const tcp_option_names[256] = {
		[0]  = "end",
		[1]  = "noop",
		[2]  = "mss",
		[3]  = "wsopt",
		[4]  = "SACK permitted",
		[5]  = "SACK",
		[6]  = "Echo (OBSOLETE)",
		[7]  = "Echo reply (OBSOLETE)",
		[8]  = "Timestamp",
		[9]  = "Partial order connection permitted",
		[10] = "Partial order service profile",
		[11] = "CC",
		[12] = "CC.NEW",
		[13] = "CC.ECHO",
		[14] = "Alternate checksum request",
		[15] = "Alternate checksum data",
		[16] = "Skeeter",
		[17] = "Bubba",
		[18] = "Trailer checksum",
		[19] = "MD5 signature",
		[20] = "SCPS capabilities",
		[21] = "NACK",
		[22] = "Record boundaries",
		[23] = "Corruption experienced",
		[24] = "SNAP",
		[26] = "TCP compression"
	};
	static const char *const ip_option_names[256] = {
		[0] = "end",
		[1] = "noop",
		[130] = "Security",
		[131] = "Loose source route",
		[68] = "Time Stamp",
		[133] = "Extended Security",
		[134] = "Commercial security",
		[7] = "Record Route",
		[136] = "Stream ID",
		[137] = "Strict source route",
		[10] = "Experimental measure (ZSU)",
		[11] = "MTU Probe (Obsolete)",
		[12] = "MTU Reply (Obsolete)",
		[205] = "Experimental Flow Control (FINN)",
		[142] = "Experimental Access Control (VISA)",
		[15] = "ENCODE",
		[144] = "IMI Traffic Descriptor",
		[145] = "Extended Internet Protocol (RFC1385)",
		[82] = "Traceroute",
		[147] = "Address Extension",
		[148] = "Router Alert",
		[149] = "Selective Directed Broadcast",
		[150] = "NSAP Addresses",
		[151] = "Dynamic Packet State",
		[152] = "Upstream Multicast Pkt."
	};

	printf("find_stats version 2.\n");

        register_printf_function('B', printf_special_B_handler_func,
                                 printf_special_B_arginfo_func);

	if (argc != 2 && argc != 3)
		errx(1, "need a filename to open");
	if (argv[1][0] == '-' && argv[1][1] == 'n') {
		argc--;
		argv++;
		mode = 1;
	} else
		mode = 0;
	if (argc != 2)
		errx(1, "need a filename to open");
	pcap = pcap_open_offline(argv[1], pcap_errbuf);
	if (pcap == NULL)
		errx(1, "openning %s: %s", argv[1], pcap_errbuf);

	pcap_loop(pcap, -1, proc_packet_callback, NULL);

	printf("\nIP packets: %d.\n", total_packets);
	printf("ip csum errors: %d.\n", ip_csum_error);
	printf("IP options: %d on %d packets, %d bogus.\n",
	       ip_options,
	       ip_option_packets,
	       ip_bogo_options);
	printf("Fragments: %d.\n", nr_fragments);
	printf("Errors: %d.\n", nr_errors);
	printf("\nUDP:\ntotal: %d\nbad: %d\nmissing: %d\n",
	       udp_packets, udp_bad_csum, udp_missing_csum);
	printf("\nDNS:\ntotal: %d\nbad: %d\nmissing: %d\n",
	       dns_packets, dns_bad_csum, dns_missing_csum);
	printf("\nTCP:\ntotal: %d\nbad: %d\nOptions: %d (%d packets, %d bad)\n",
	       tcp_packets, tcp_bad_csum, tcp_options,
	       tcp_packets_with_options, tcp_bad_options);

	printf("\n\nIP dgram sizes (size, n(%%), cumulative(%%)):\n");
	acc = 0;
	for (x = 0; x <= MAX_IP_DGRAM_SIZE; x++) {
		acc += ip_datagram_sizes[x];
		printf("ipdgs\t%d\t%d(%f)\t%lld(%f)\n", x,
		       ip_datagram_sizes[x],
		       ip_datagram_sizes[x] * 100.0 / total_packets,
		       acc,
		       acc * 100.0 / total_packets);
	}
	printf("\n\nIP options:\n");
	for (x = 0; x < 256; x++) {
		if (ip_options_dist[x] != 0)
			printf("%3d(%35s) %8d(%f)\n", x,
			       ip_option_names[x] ?: "unknown",
			       ip_options_dist[x],
			       ip_options_dist[x] * 100.0 / ip_options);
	}
	printf("\n\nIP TTL dist (ttl, n(%%)):\n");
	for (x = 0; x <= 256; x++) {
		printf("ipttl\t%d\t%d(%f)\n", x,
		       ip_ttl_dist[x],
		       ip_ttl_dist[x] * 100.0 / total_packets);
	}
	printf("\n\nIP TOS dist (tos, n(%%)):\n");
	for (x = 0; x <= 64; x++) {
		if (ip_types_of_service[x] != 0)
			printf("iptos\t%x\t%d(%f)\n", x,
			       ip_types_of_service[x],
			       ip_types_of_service[x] * 100.0 / total_packets);
	}
	printf("\n\nNon-ECN:\t%d\nECT(1):\t%d\nECT(0):\t%d\nCE:\t%d.\n",
	       ecn_counters[0], ecn_counters[1], ecn_counters[2],
	       ecn_counters[3]);
	printf("TCP CWR:\t%d\nTCP ECE:\t%d\nTCP ECN CE:\t%d.\n",
	       tcp_cwr_packets,
	       tcp_ece_packets,
	       tcp_ecn_ce);
	printf("\n\nTCP explicit MSSs (size, n):\n");
	for (x = 0; x <= MAX_TCP_MSS; x++) {
		printf("tcpmss\t%d\t%d\n", x,
		       tcp_mss_dist[x]);
	}
	printf("\n\nTCP options:\n");
	for (x = 0; x < 256; x++) {
		if (tcp_option_dist[x] != 0)
			printf("%3d(%35s) %8d(%f)\n", x,
			       tcp_option_names[x] ?: "unknown",
			       tcp_option_dist[x],
			       tcp_option_dist[x] * 100.0 / tcp_options);
	}

	printf("\n\nError prob. against packet size (size, %%)\n");
	for (x = 0; x <= MAX_IP_DGRAM_SIZE; x++) {
		if (err_packet_size_dist[x] + ip_datagram_sizes[x] != 0)
			printf("%d\t%f\n", x,
			       (double)err_packet_size_dist[x] /
			       ip_datagram_sizes[x]);
	}

	return 0;
}
