/* A few random utility functions */
#define _GNU_SOURCE
#include <err.h>
#include <printf.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <netinet/in.h>
#include <sys/time.h>

#include "demux.h"
#include "proto.h"

unsigned long
__net_to_host(const void *x, unsigned s)
{
	n16 *_n16;
	n32 *_n32;
	switch (s) {
	case 2:
		_n16 = (n16 *)x;
		return ntohs(_n16->a);
	case 4:
		_n32 = (n32 *)x;
		return ntohl(_n32->a);
	default:
		abort();
	}
}

int
cmp_streamid(const struct streamid *sid1, const struct streamid *sid2)
{
	return !(sid1->saddr.a == sid2->saddr.a &&
		 sid1->sport.a == sid2->sport.a &&
		 sid1->daddr.a == sid2->daddr.a &&
		 sid1->dport.a == sid2->dport.a &&
		 sid1->protocol == sid2->protocol);
}

unsigned long
hash_streamid(const struct streamid *sid)
{
	unsigned long h;
	h = (sid->saddr.a ^ sid->daddr.a ^ sid->sport.a ^ sid->dport.a);
	while (h >= HASH_SIZE)
		h = (h / HASH_SIZE) ^ (h % HASH_SIZE);
	return h;
}

int
seq_eq(seqnr_t s1, seqnr_t s2)
{
	return s1.s == s2.s;
}

int
seq_gt(seqnr_t s1, seqnr_t s2)
{
	unsigned long S1, S2;
	unsigned long d;
	S1 = ntohl(s1.s);
	S2 = ntohl(s2.s);
	d = S1 - S2;
	if (d > 0 && d < 0x40000000)
		return 1;
	else
		return 0;
}

int
seq_le(seqnr_t s1, seqnr_t s2)
{
	unsigned long S1, S2;
	unsigned long d;
	S1 = ntohl(s1.s);
	S2 = ntohl(s2.s);
	d = S1 - S2;
	if (d > 0xc0000000)
		return 1;
	else
		return 0;
}

seqnr_t
seq_plus(seqnr_t s, unsigned long x)
{
	return (seqnr_t){s:htonl(ntohl(s.s) + x)};
}

seqnr_t
seq_sub(seqnr_t s, unsigned long x)
{
	return (seqnr_t){s:htonl(ntohl(s.s) - x)};
}

void *
xcalloc(unsigned s, unsigned n)
{
	void *r;
	r = calloc(s, n);
	if (!r)
		err(1, "out of memory");
	return r;
}

static int
printf_special_D_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	int r;
	struct datagram *dg = *(struct datagram **)args[0];
	r = fprintf(f, "%Q->%Q (id %x)",
		    dg->head_packet->ip->saddr.a,
		    dg->head_packet->ip->daddr.a,
		    dg->head_packet->ip->id.a);
	return r;
}

static int
printf_special_D_arginfo_func(const struct printf_info *info,
                              size_t n,
                              int *argtypes)
{
        if (n > 0)
                argtypes[0] = PA_POINTER;
        return 1;
}

static int
printf_special_Q_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	int r;
	unsigned long addr = *(unsigned long *)args[0];
	unsigned char *a = (unsigned char *)&addr;
	r = fprintf(f, "%ld.%ld.%ld.%ld",
		    (unsigned long)a[0],
		    (unsigned long)a[1],
		    (unsigned long)a[2],
		    (unsigned long)a[3]);
	return r;
}

static int
printf_special_Q_arginfo_func(const struct printf_info *info,
                              size_t n,
                              int *argtypes)
{
        if (n > 0)
                argtypes[0] = PA_FLAG_LONG | PA_INT;
        return 1;
}

static int
printf_special_P_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	int r;
	struct tcpconn *p = *(struct tcpconn **)args[0];
	switch (p->sid.protocol) {
	case IPPROTO_UDP:
	case IPPROTO_TCP:
		r = fprintf(f, "%d/%Q/%.05d/%Q/%.05d/%W",
			    (int)p->sid.protocol,
			    p->sid.saddr.a,
			    net_to_host(p->sid.sport),
			    p->sid.daddr.a,
			    net_to_host(p->sid.dport),
			    &p->start);
		break;
	default:
		r = fprintf(f, "%d/%Q/%Q/%W",
			    (int)p->sid.protocol,
			    p->sid.saddr.a,
			    p->sid.daddr.a,
			    &p->start);
		break;
	}
	return r;
}

static int
printf_special_P_arginfo_func(const struct printf_info *info,
                              size_t n,
                              int *argtypes)
{
        if (n > 0)
                argtypes[0] = PA_POINTER;
        return 1;
}

static void
register_printf_specials(void)
{
	register_printf_function('P', printf_special_P_handler_func,
                                 printf_special_P_arginfo_func);
	register_printf_function('Q', printf_special_Q_handler_func,
                                 printf_special_Q_arginfo_func);
	register_printf_function('D', printf_special_D_handler_func,
                                 printf_special_D_arginfo_func);
}

void
setup_util(void)
{
	register_printf_specials();
}

struct tee_file_cookie {
	FILE *f1, *f2;
};

static ssize_t
tee_write_func(void *cookie, const char *buffer, size_t size)
{
	struct tee_file_cookie *tfc = cookie;
	return fwrite(buffer, 1, fwrite(buffer, 1, size, tfc->f1), tfc->f2);
}

static int
tee_close_func(void *cookie)
{
	free(cookie);
	return 0;
}

FILE *
open_tee(FILE *f1, FILE *f2)
{
	struct tee_file_cookie *tfc;
	FILE *f;
	tfc = xcalloc(sizeof(*tfc), 1);
	tfc->f1 = f1;
	tfc->f2 = f2;
	f = fopencookie(tfc, "w",
			(cookie_io_functions_t){write:tee_write_func,
						close:tee_close_func});
	if (!f)
		err(1, "openning tee file");
	setvbuf(f, NULL, _IONBF, 0);
	return f;
}

#define NR_HISTO_BUCKETS 65536
#define HISTO_SCALE 1.0005
static inline double
HISTO_BUCKET_TO_VAL(int x)
{
	return pow(HISTO_SCALE, x) + 299;
}

static inline int
VAL_TO_HISTO_BUCKET(double v)
{
	int x;
	x = log10(v - 299) / log10(HISTO_SCALE);
	if (x < 0)
		return 0;
	if (x >= NR_HISTO_BUCKETS)
		return NR_HISTO_BUCKETS-1;
	return x;
}

static unsigned long long
histo_buckets[NR_HISTO_BUCKETS];

void
histo_sample(double val)
{
	histo_buckets[VAL_TO_HISTO_BUCKET(val)]++;
}

void
dump_stats(timestamp_t ts)
{
	struct timeval now, rt_passed, pass_time;
	static struct timeval last_pass_finish;
	timediff_t time_passed;
	static FILE *histo_file;
	int x;

	gettimeofday(&now, NULL);

	rt_passed.tv_sec = now.tv_sec - stats.rt_start_time.tv_sec;
	rt_passed.tv_usec = now.tv_usec - stats.rt_start_time.tv_usec;
	if (rt_passed.tv_usec < 0) {
		rt_passed.tv_usec += 1000000;
		rt_passed.tv_sec--;
	}

	time_passed = timestamp_sub(ts, stats.start_time);

	pass_time.tv_sec = now.tv_sec - last_pass_finish.tv_sec;
	pass_time.tv_usec = now.tv_usec - last_pass_finish.tv_usec;
	if (pass_time.tv_usec < 0) {
		pass_time.tv_usec += 1000000;
		pass_time.tv_sec--;
	}
	last_pass_finish = now;

	fprintf(logfile,
		"Processed %w seconds.  %lld packets (%lld datagrams, %d incomplete), of which %f tcp.\n",
		&time_passed,
		stats.total_packets, stats.total_datagrams,
		stats.incomplete_datagrams,
		(double)stats.tcp_datagrams / stats.total_datagrams);
	fprintf(logfile,
		"Real time %ld.%.06ld / trace time = %f; pass time %ld.%.06ld\n",
		rt_passed.tv_sec, rt_passed.tv_usec,
		(rt_passed.tv_sec + rt_passed.tv_usec * 1e-6) /
		timediff_to_double(time_passed),
		pass_time.tv_sec, pass_time.tv_usec);
	fprintf(logfile,
		"Buffers: %lld collected, %lld fullness flushes\n",
		stats.gced_buffers, stats.buffer_full_flushes);
	fprintf(logfile,
		"File descriptors: %d, %lld collected\n",
		stats.fds_used, stats.gced_fds);
	fprintf(logfile,
		"Connections: %d/%d (%d TCP, %d comatose, %d aborted), %d collected (%d non-comatose)\n",
		stats.live_connections, NR_LIVE_CONNS,
		stats.nr_tcp_connections, stats.comatose_connections,
		stats.aborted_connections, stats.gced_conns,
		stats.gced_non_comatose_conns);
	fprintf(logfile,
		"%d packets on comatose connections.\n",
		stats.datagrams_on_comatose_conns);
	fprintf(logfile,
		"Hash badness %f, fd hit rate %f, buffer hit rate %f.\n",
		(double)stats.hash_probes / stats.conn_lookups,
		(double)stats.fd_hits / stats.fd_requests,
		(double)stats.buffer_hits / stats.buffer_requests);

	if (!histo_file) {
		collect_fds();
		histo_file = fopen("histo_file", "w");
		if (!histo_file)
			err(1, "openning histo_file");
	}
	fseek(histo_file, 0, SEEK_SET);
	for (x = 0; x < NR_HISTO_BUCKETS; x++)
		fprintf(histo_file, "%f %f\n", HISTO_BUCKET_TO_VAL(x),
			(double)histo_buckets[x] /
			(HISTO_BUCKET_TO_VAL(x+1)-HISTO_BUCKET_TO_VAL(x)));
	fflush(histo_file);
}
